#ifndef BmlCaen767VmeDevice_HH
#define BmlCaen767VmeDevice_HH

#include <unistd.h>

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

// dual/inc/bml
#include "BmlCaen767RunData.hh"
#include "BmlCaen767ConfigurationData.hh"
#include "BmlCaen767OpcodeData.hh"
#include "BmlCaen767EventData.hh"
#include "BmlCaen767TriggerData.hh"
#include "BmlCaen767TestData.hh"
#include "BmlCaen767TdcErrorData.hh"


class BmlCaen767VmeDevice : public HAL::VMEDevice {
public:
  BmlCaen767VmeDevice(HAL::VMEAddressTable &t,
		      HAL::VMEBusAdapterInterface &b,
		      unsigned short a=0x00c5)
    : HAL::VMEDevice(t,b,a<<16), _baseAddress(a) {
  }

  unsigned short baseAddress() const {
    return _baseAddress;
  }

  bool alive() {
    /*unsigned long*/ uint32_t value;
    try {
      UtlPack bid(0);
      for(unsigned i(0);i<7;i+=2) {
	read("BoardId",&value,2*i);
	//std::cout << "BoardId+" << std::setw(2) << i
	//	  << " = " << printHex((unsigned)value) << std::endl;
	bid.byte(3-(i/2),value&0xff);
      }
      return UtlPack(0x000002ff)==bid;

    } catch ( HAL::HardwareAccessException& e ) {
      //std::cout << "*** HA Exception occurred : " << e.what() << endl;
      return false;

    } catch ( std::exception e ) {
      //std::cout << "*** Unknown exception occurred" << endl;
      return false;
    }
  }

  bool reset() {
    write("SingleShotReset",0);    
    sleep(2);

    assert(writeOpcode(0x56)); // Zero all adjusts
    write("GeoAddress",_baseAddress&0x1f);  // Sets geo address in data

    write("AddressDecoder24",_baseAddress&0xff);  // Sets base address (unused)
    write("AddressDecoder32",_baseAddress>>8);    // Sets base address (unused)

    write("Clear",0);

    return true;
  }

  bool softTrigger() {
    ///*unsigned long*/ uint32_t value;
    //read("EventCounter",&value);
    //std::cout << "Event counter = " << value << std::endl;

    //write("SoftwareTrigger",1);
    write("SoftwareTrigger",0);

    //read("EventCounter",&value);
    //std::cout << "Event counter = " << value << std::endl;

    return true;
  }

  bool clear() {
    write("Clear",0);

    write("ClearEventCounter",0);
    /*unsigned long*/ uint32_t value;
    read("Buffer",&value);
    read("Buffer",&value);

      /*
    bool empty(false);
    for(unsigned i(0);i<100000 && !empty;i++) {
      read("Buffer",&value);
      read("Status2",&value);
      empty=((value&1)==1);
    }

    return empty;
    */
    return true;
  }

  bool readRunData(BmlCaen767RunData &d) {
    /*unsigned long*/ uint32_t value;

    for(unsigned i(0);i<64;i++) {
      read("CRRomStart",&value,4*i);
      d.crRom(i,value&0xff);
      //std::cout << "CR ROM byte " << std::setw(2) << i << " = "
      //	<< printHex((unsigned)value) << std::endl;
    }

    /*
    read("F0AccessWidth",&value);
    std::cout << "F0AccessWidth " << printHex((unsigned)value) << std::endl;

    read("F1AccessWidth",&value);
    std::cout << "F1AccessWidth " << printHex((unsigned)value) << std::endl;

    for(unsigned i(0);i<8;i++) {
      read("F0AmCodeMask",&value,4*i);
      std::cout << "F0AmCodeMask " << i << printHex((unsigned)value) << std::endl;
    }

    for(unsigned i(0);i<8;i++) {
      read("F1AmCodeMask",&value,4*i);
      std::cout << "F1AmCodeMask " << i << printHex((unsigned)value) << std::endl;
    }

    for(unsigned i(0);i<4;i++) {
      read("F0AddressMask",&value,4*i);
      std::cout << "F0AddressMask " << i << printHex((unsigned)value) << std::endl;
    }

    for(unsigned i(0);i<4;i++) {
      read("F1AddressMask",&value,4*i);
      std::cout << "F1AddressMask " << i << printHex((unsigned)value) << std::endl;
    }
    */

    for(unsigned i(0);i<4;i++) {
      read("SerialNumber",&value,4*i);
      //std::cout << "SerialNumber " << i 
      //	<< printHex((unsigned)value) << std::endl;
      d.serialNumber(i,value&0xff);
    }

    read("AddressDecoder24",&value);
    //std::cout << "AddressDecoder24 " << printHex((unsigned)value) << std::endl;
    d.addressDecoder(0,value&0xff);

    read("AddressDecoder32",&value);
    //std::cout << "AddressDecoder32 " << printHex((unsigned)value) << std::endl;
    d.addressDecoder(1,value&0xff);

    read("McstAddress",&value);
    //std::cout << "MCST address " << printHex((unsigned)value) << std::endl;
    d.mcstAddress(value&0xffff);

    return true;
  }

  bool writeOpcodeData(const BmlCaen767OpcodeData &d) {
    UtlPack y[4];
    unsigned short *x((unsigned short*)y);
    short *z((short*)y);

    if(d.memoryTest()) assert(writeOpcode(0x01));
    else               assert(writeOpcode(0x02));

    if(d.stopTriggerMatching())  assert(writeOpcode(0x10));
    if(d.startTriggerMatching()) assert(writeOpcode(0x11));
    if(d.startGating())          assert(writeOpcode(0x12));
    if(d.continuousStorage())    assert(writeOpcode(0x13));

    if(d.autoLoad()) assert(writeOpcode(0x18));
    else             assert(writeOpcode(0x19));

    for(unsigned i(0);i<8;i++) x[i]=d.enablePattern(i);
    assert(writeOpcodeData(0x25,x,8));

    x[0]=d.windowWidth();
    assert(writeOpcodeData(0x30,x,1));
    z[0]=d.windowOffset();
    assert(writeOpcodeData(0x32,x,1));
    x[0]=d.triggerLatency();
    assert(writeOpcodeData(0x34,x,1));
    if(d.subtractTriggerTime()) assert(writeOpcode(0x36));
    else                        assert(writeOpcode(0x37));
    if(d.overlappingTriggers()) assert(writeOpcode(0x38));
    else                        assert(writeOpcode(0x39));

    if(d.readoutStartTime())         assert(writeOpcode(0x40));
    else if (d.readout4StartTimes()) assert(writeOpcode(0x41));
    else                             assert(writeOpcode(0x42));
    if(d.subtractStartTime())        assert(writeOpcode(0x43));
    else                             assert(writeOpcode(0x44));
    if(d.emptyStart())               assert(writeOpcode(0x45));
    else                             assert(writeOpcode(0x46));
    
    x[0]=d.globalOffset();
    assert(writeOpcodeData(0x52,x,1));
    if(d.allAdjusts()) assert(writeOpcode(0x54));
    else               assert(writeOpcode(0x55));
    
    if(d.risingEdgeAllChannels())  assert(writeOpcode(0x60));
    if(d.fallingEdgeAllChannels()) assert(writeOpcode(0x61));
    if(d.risingOddFallingEven())   assert(writeOpcode(0x62));
    if(d.risingEvenFallingOdd())   assert(writeOpcode(0x63));
    if(d.risingEdgeStart())        assert(writeOpcode(0x64));
    if(d.fallingEdgeStart())       assert(writeOpcode(0x65));
    if(d.bothEdgesAllChannels())   assert(writeOpcode(0x66));

    if(d.dataReadyEventReady())       assert(writeOpcode(0x70));
    if(d.dataReadyBufferAlmostFull()) assert(writeOpcode(0x71));
    if(d.dataReadyBufferNotEmpty())   assert(writeOpcode(0x72));
    x[0]=d.almostFullLevel();
    assert(writeOpcodeData(0x74,x,1));
    
    return true;
  }

  bool readOpcodeData(BmlCaen767OpcodeData &d) {
    UtlPack y[4];
    unsigned short *x((unsigned short*)y);
    short *z((short*)y);

    assert(readOpcodeData(0x03,x));
    //std::cout << "Opcode 0x0300 = " << printHex(x[0]) << std::endl;
    d.memoryTest(y[0].bit(0));

    assert(readOpcodeData(0x14,x));
    //std::cout << "Opcode 0x1400 = " << printHex(x[0]) << std::endl;
    d.acquisitionMode(y[0].bits(0,1));

    assert(readOpcodeData(0x1a,x));
    //std::cout << "Opcode 0x1a00 = " << printHex(x[0]) << std::endl;
    d.autoLoad(y[0].bit(0));

    /*
    for(unsigned i(0);i<128;i++) {
      assert(readOpcodeData(0x22,i,x));
      //std::cout << "Opcode 0x22 Ch " << std::setw(3) << i << " = " << printHex(x[0]) << std::endl;
    }
    */
    assert(readOpcodeData(0x26,x,8));
    for(unsigned i(0);i<8;i++) {
      //std::cout << "Opcode 0x2600 " << i << " = " << printHex(x[i]) << std::endl;
      d.enablePattern(i,x[i]);
    }

    assert(readOpcodeData(0x31,x));
    //std::cout << "Opcode 0x3100 = " << printHex(x[0]) << std::endl;
    d.windowWidth(x[0]);
    assert(readOpcodeData(0x33,x));
    //std::cout << "Opcode 0x3300 = " << printHex(x[0]) << std::endl;
    d.windowOffset(z[0]);
    assert(readOpcodeData(0x35,x));
    //std::cout << "Opcode 0x3500 = " << printHex(x[0]) << std::endl;
    d.triggerLatency(x[0]);
    assert(readOpcodeData(0x3a,x));
    //std::cout << "Opcode 0x3a00 = " << printHex(x[0]) << std::endl;
    d.triggerConfiguration(y[0].bits(0,1));

    assert(readOpcodeData(0x47,x));
    //std::cout << "Opcode 0x4700 = " << printHex(x[0]) << std::endl;
    d.startConfiguration(y[0].bits(0,3));

    /*
    for(unsigned i(0);i<128;i++) {
      assert(readOpcodeData(0x51,i,x));
      //std::cout << "Opcode 0x51 Ch " << std::setw(3) << i << " = " << printHex(x[0]) << std::endl;
    }
    */
    assert(readOpcodeData(0x53,x));
    //std::cout << "Opcode 0x5300 = " << printHex(x[0]) << std::endl;
    d.globalOffset(x[0]);
    assert(readOpcodeData(0x57,x));
    //std::cout << "Opcode 0x5700 = " << printHex(x[0]) << std::endl;
    d.allAdjusts(y[0].bit(0));

    assert(readOpcodeData(0x67,x,3));
    for(unsigned i(0);i<3;i++) {
      //std::cout << "Opcode 0x6700 " << i << " = " << printHex(x[i]) << std::endl;
      if(i==0) d.detection(0,y[0].bits( 0, 1));
      if(i==1) d.detection(1,y[0].bits(16,17));
      if(i==2) d.detection(2,y[1].bits( 0, 1));
    }

    assert(readOpcodeData(0x73,x));
    //std::cout << "Opcode 0x7300 = " << printHex(x[0]) << std::endl;
    d.dataReadyMode(y[0].bits(0,1));
    assert(readOpcodeData(0x75,x));
    //std::cout << "Opcode 0x7500 = " << printHex(x[0]) << std::endl;
    d.almostFullLevel(x[0]);

    for(unsigned i(0);i<4;i++) {
      assert(readOpcodeData(0x80,i,x));
      //std::cout << "Opcode 0x80 TDC " << i << " = " << printHex(x[0]) << std::endl;
    }
    for(unsigned i(0);i<4;i++) {
      assert(readOpcodeData(0x81,i,x,2));
      for(unsigned j(0);j<2;j++) {
	//std::cout << "Opcode 0x81 TDC " << i << " " << j << " = " << printHex(x[j]) << std::endl;
      }
    }

    assert(readOpcodeData(0x91,x));
    //std::cout << "Opcode 0x9100 = " << printHex(x[0]) << std::endl;
    assert(readOpcodeData(0x94,x));
    //std::cout << "Opcode 0x9400 = " << printHex(x[0]) << std::endl;

    assert(readOpcodeData(0xa1,x));
    //std::cout << "Opcode 0xa100 = " << printHex(x[0]) << std::endl;
    assert(readOpcodeData(0xa3,x));
    //std::cout << "Opcode 0xa300 = " << printHex(x[0]) << std::endl;

    assert(readOpcodeData(0xb1,x));
    //std::cout << "Opcode 0xb100 = " << printHex(x[0]) << std::endl;
    assert(readOpcodeData(0xb7,x));
    //std::cout << "Opcode 0xb700 = " << printHex(x[0]) << std::endl;
    assert(readOpcodeData(0xb9,x));
    //std::cout << "Opcode 0xb900 = " << printHex(x[0]) << std::endl;

    return true;
  }

  bool readTdcErrorData(BmlCaen767TdcErrorData &d) {
    unsigned short x;

    for(unsigned i(0);i<4;i++) {
      assert(readOpcodeData(0x80,i,&x));
      //std::cout << "Opcode 0x80 TDC " << i << " = " << printHex(x) << std::endl;
      d.tdcErrors(i,x);
    }

    return true;
  }

  bool writeConfigurationData(const BmlCaen767ConfigurationData &d) {

    write("Control1",d.controlRegister().halfWord(0));
    //write("Control2",d.controlRegister().halfWord(1)); // Read-only!
    write("InterruptLevel",d.interruptRegister().halfWord(0));
    write("InterruptVector",d.interruptRegister().halfWord(1));

    // Special case
    if(d.memoryTest()) assert(writeOpcode(0x01));
    else               assert(writeOpcode(0x02));
    usleep(10000);

    return true;
  }

  bool readConfigurationData(BmlCaen767ConfigurationData &d) {
    /*unsigned long*/ uint32_t value;

    read("Control1",&value);
    //std::cout << "Control1 = " << printHex((unsigned)value) << std::endl;
    d.controlRegister(0,value&0xffff);

    read("Control2",&value);
    //std::cout << "Control2 = " << printHex((unsigned)value) << std::endl;
    d.controlRegister(1,value&0xffff);

    read("InterruptLevel",&value);
    //std::cout << "InterruptLevel = " << printHex((unsigned)value) << std::endl;
    d.interruptLevel(value&0xffff);

    read("InterruptVector",&value);
    //std::cout << "InterruptVector = " << printHex((unsigned)value) << std::endl;
    d.interruptVector(value&0xffff);

    return true;
  }

  bool writeTestData(const BmlCaen767TestData &d) {
    UtlPack *p((UtlPack*)d.data());

    for(unsigned r(0);r<d.numberOfRepeats();r++) {
      for(unsigned i(0);i<d.numberOfWords();i++) {
	write("TestwordLo",p[i].halfWord(0));
	write("TestwordHi",p[i].halfWord(1));
      }    
    }

    return true;
  }

  bool readTriggerData(BmlCaen767TriggerData &d) {
    /*unsigned long*/ uint32_t value;

    read("Status1",&value);
    d.statusRegister(0,value&0xffff);
    read("Status2",&value);
    d.statusRegister(1,value&0xffff);
    read("EventCounter",&value);
    d.eventCounter(value);

    return true;
  
  }

  bool readEventData(BmlCaen767EventData &d, bool blt=true, bool array=false) {
    /*unsigned long*/ uint32_t value;

    d.numberOfWords(0);

    read("Status1",&value);
    d.statusRegister(0,value&0xffff);
    read("Status2",&value);
    d.statusRegister(1,value&0xffff);

    BmlCaen767EventDatum *p(d.data());
    /*unsigned long*/ uint32_t *q((/*unsigned long*/ uint32_t*)d.data());

    unsigned nWords(64*1024);

    // Block transfer
    if(blt) {
      if(array) {
	readBlock("BufferBltArray",256,(char*)_blt,HAL::HAL_DO_INCREMENT);
	for(unsigned i(0);i<128;i++) {
	  std::cout << i << " " << _blt[i].print(std::cout," RAW ");
	}

	for(unsigned i(0);i<64 && i<nWords;i++) {
	  p[i]=_blt[2*i];
	  if((p[i].label()==BmlCaen767EventDatum::eob) ||
	     (p[i].label()==BmlCaen767EventDatum::invalid))
	    nWords=i+1;
	}
	nWords=64;

      } else {
	memset(_blt,0,128*4);
	p[0].setInvalid();
	//readBlock("BufferBlt",8,(char*)_blt,HAL::HAL_NO_INCREMENT);
	readBlock("BufferBlt",256,(char*)_blt,HAL::HAL_NO_INCREMENT);
	std::cout << std::dec; // Set to hex by readBlock call above
	//read("Buffer",&value);
	for(unsigned i(0);i<128;i++) {
	  std::cout << i << " ";
	  _blt[i].print(std::cout," RAW ");
	}

	read("BitSet",&value);
	std::cout << "BitSet = " << printHex((unsigned)value) << std::endl;

	for(unsigned i(0);i<64 && i<nWords;i++) {
	  p[i]=_blt[2*i];
	  if((p[i].label()==BmlCaen767EventDatum::eob) ||
	     (p[i].label()==BmlCaen767EventDatum::invalid))
	    nWords=i+1;
	}
	//nWords=64;
      }

    // Standard read
    } else {
      for(unsigned i(0);i<32*1024 && i<nWords;i++) {
	if(array) read("BufferArray",q+i,4*i);
	else      read("Buffer",q+i);
	//std::cout << "Buffer = " << printHex((unsigned)q[i]) << std::endl;
	
	if((p[i].label()==BmlCaen767EventDatum::eob) ||
	   (p[i].label()==BmlCaen767EventDatum::invalid))
	  nWords=i+1;
      }
    }

    d.numberOfWords(nWords);

    return true;
  }

  void print(std::ostream &o) {
    o << "BmlCaen767VmeDevice::print()" << std::endl;
    /*
    unsigned long value;
    read("FirmwareId",&value);
    o << hex << "  FirmwareId = 0x" << value << dec << std::endl;
    */
    o << std::endl;

  }

private:
  bool writeOpcode(unsigned char c) {
    return writeOpcode(c,0);
  }

  bool writeOpcode(unsigned char c, unsigned char o) {
    //std::cout << "Writing 0x" << std::hex << std::setfill('0') << std::setw(2) << (unsigned)c 
    //      << std::setw(2) << (unsigned)o << std::setfill(' ') << std::dec << std::endl;

    /*unsigned long*/ uint32_t value;
    for(unsigned i(0);i<1000000;i++) {
      read("OpcodeHandshake",&value);
      //std::cout << "writeOpcode " << printHex((unsigned)value) << std::endl;
      
      // Check for write bit
      if((value&2)!=0) {
	usleep(10000);
	write("OpcodeRegister",(c<<8)+o);
	return true;
      }
    }
    return false;
  }

  bool writeOpcodeData(unsigned char c, unsigned short *d, unsigned n=1) {
    return writeOpcodeData(c,0,d,n);
  }

  bool writeOpcodeData(unsigned char c, unsigned char o, unsigned short *d, unsigned n=1) {

    // Check that there is data if requested
    assert(n>0 && d!=0);

    // First write the opcode itself
    writeOpcode(c,o);
    
    //std::cout << " Writing opcode data" << std::endl;
    //for(unsigned j(0);j<n;j++) {
      //std::cout << "  Data word " << j << " = " << printHex(d[j]) << std::endl;
    //}    
    
    // Now write the data
    /*unsigned long*/ uint32_t value;
    for(unsigned j(0);j<n;j++) {
      
      // Loop until write bit seen
      bool done(false);
      for(unsigned i(0);i<1000000 && !done;i++) {
	read("OpcodeHandshake",&value);
	//std::cout << "writeOpcode " << printHex((unsigned)value) << std::endl;
	
	// Check for write bit
	if((value&2)!=0) {
	  done=true;
	  usleep(10000);
	  write("OpcodeRegister",d[j]);
	}
      }
	
      // If read bit never appeared, give up
      if(!done) return false;
    }
    
    return true;
  }

  bool readOpcodeData(unsigned char c, unsigned short *d, unsigned n=1) {
    return readOpcodeData(c,0,d,n);
  }

  bool readOpcodeData(unsigned char c, unsigned char o, unsigned short *d, unsigned n=1) {

    // Check some data to read back are requested
    assert(n>0);

    //std::cout << "Reading 0x" << std::hex << std::setfill('0') << std::setw(2) << (unsigned)c 
    //      << std::setw(2) << (unsigned)o << std::setfill(' ') << std::dec << std::endl;
    //std::cout << " Reading opcode data" << std::endl;

    // Write the opcode to start the transfer
    writeOpcode(c,o);

    // Now read each requested word in turn
    /*unsigned long*/ uint32_t value;
    for(unsigned j(0);j<n;j++) {

      // Loop until read bit seen
      bool done(false);
      for(unsigned i(0);i<1000000 && !done;i++) {
	read("OpcodeHandshake",&value);
	//std::cout << "readOpcodeHandshake " << printHex((unsigned)value) << std::endl;
	
	// Check for read bit
	if((value&1)!=0) {
	  done=true;
	  usleep(10000);

	  // Now read data from register
	  read("OpcodeRegister",&value);
	  //std::cout << "readOpcodeRegister " << printHex((unsigned)value) << std::endl;
	  d[j]=value&0xffff;
	  //std::cout << "  Data word " << j << " = " << printHex(d[j]) << std::endl;
	}
      }

      // If read bit never appeared, give up
      if(!done) return false;
    }

    return true;
  }

  const unsigned short _baseAddress;
  BmlCaen767EventDatum _blt[128];
};

#endif
