#ifndef CrcDummyDevice_HH
#define CrcDummyDevice_HH

#include "CrcVmeRunData.hh"
#include "CrcBeTrgRunData.hh"
#include "CrcBeTrgConfigurationData.hh"
#include "CrcBeTrgEventData.hh"
#include "CrcSerialHeader.hh"

class CrcDummyDevice {

public:
  CrcDummyDevice(unsigned slot) : _baseAddress(slot<<16) {
    _vmeRunData.firmwareId(0x21000316);
    _vmeRunData.clockSelect(0);
    _vmeRunData.serialNumber(20+slot);
    _serialIO=false;
    _epromControl=0;
    _beTrgCommands=0xdeadbeef;
  }

  bool read(unsigned a, unsigned &r) {
    if((a&0x00ff0000)!=_baseAddress) return false;
    unsigned localAddress(a&0xffff);

    // Serial buffer
    if(localAddress<0x0800) {
      if((localAddress/4)<_numberOfSerialWords)
	r=_serialBuffer[localAddress/4];
      else
	r=0;
      return true;
    }

    // Serial status
    if(localAddress==0x080c) {
      if(_serialIO) r=0xffffffff;
      else          r=0x00000000;
      _serialIO=false;
      return true;
    }

    // Firmware id
    if(localAddress==0x0830) {
      r=_vmeRunData.firmwareId();
      return true;
    }

    // Clock select
    if(localAddress==0x0834) {
      r=_vmeRunData.clockSelect();
      return true;
    }

    // VME status
    if(localAddress==0x083c) {
      r=0;
      return true;
    }

    // TTC clock counter
    if(localAddress==0x0840) {
      r=rand()&0xfff;
      return true;
    }

    // Backplane clock counter
    if(localAddress==0x0844) {
      r=rand()&0xfff;
      return true;
    }

    // EPROM status
    if(localAddress==0x0864) {
      if((_epromControl&0xffcff)==0x80000) {
	r=_vmeRunData.epromHeader()>>((_epromControl&0x300)>>8);
      } else {
	r=0;
      }
      return true;
    }

    // Unknown local address
    std::cerr << "CrcDummyDevice::read() Unknown local address "
	      << std::hex << localAddress << std::endl;

    return false;
  }


  bool write(unsigned a, unsigned w) {
    std::cout << "CrcDummyDevice::write() Address "
	      << std::hex << a << std::dec << std::endl;

    if((a&0x00ff0000)!=_baseAddress) return false;
    unsigned short localAddress(a&0xffff);

    // Serial buffer
    if(localAddress<0x0800) {
      _serialBuffer[localAddress/4]=w;
      return true;
    }

    // Serial write
    if(localAddress==0x0800) {
      assert(!_serialIO);
      _serialIO=true;
      _numberOfSerialWords=w;
      std::cout << "Number of serial words = " << w << std::endl;

      _lastHeader=*((CrcSerialHeader*)_serialBuffer);
      _lastHeader.print(std::cout);

      if(_lastHeader.target()==CrcSerialHeader::beTrg) {
	if(_lastHeader.designator()==CrcSerialHeader::beTrgCommands) {
	  assert(_numberOfSerialWords==2);
	  _beTrgCommands=_serialBuffer[1];
	}
      }

      return true;
    }

    // Serial read
    if(localAddress==0x0804) {
      assert(!_serialIO);
      _serialIO=true;
      w--; // reduce length by one
      _lastHeader=*((CrcSerialHeader*)(&w));
      _lastHeader.print(std::cout);

      _numberOfSerialWords=(_lastHeader.length()+31)/32;

      if(_lastHeader.target()==CrcSerialHeader::beTrg) {
	if(_lastHeader.designator()==CrcSerialHeader::beTrgCommands) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgCommands;
	}

	if(_lastHeader.designator()==CrcSerialHeader::beTrgFirmwareId) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgRunData.firmwareId();
	}
	if(_lastHeader.designator()==CrcSerialHeader::beTrgFirmwareDate) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgRunData.firmwareDate();
	}

	if(_lastHeader.designator()==CrcSerialHeader::beTrgInputEnable) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgConfigurationData.inputEnable();
	}
	if(_lastHeader.designator()==CrcSerialHeader::beTrgGeneralEnable) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgConfigurationData.generalEnable();
	}
	if(_lastHeader.designator()==CrcSerialHeader::beTrgOscillationPeriod) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgConfigurationData.oscillationPeriod();
	}
	if(_lastHeader.designator()==CrcSerialHeader::beTrgBurstCounter) {
	  assert(_numberOfSerialWords==1);
	  _serialBuffer[0]=_beTrgConfigurationData.burstCounter();
	}

      }

      return true;
    }

    // Serial status
    if(localAddress==0x080c) {
      assert(false); //????
    }

    // Clock select
    if(localAddress==0x0834) {
      _vmeRunData.clockSelect(w);
      return true;
    }

    // Software reset
    if(localAddress==0x0838) {
      _vmeRunData.clockSelect(0);
      return true;
    }

    // EPROM control
    if(localAddress==0x0860) {
      _epromControl=w;
      return true;
    }

    // Unknown local address
    std::cerr << "CrcDummyDevice::write() Unknown local address "
	      << std::hex << localAddress << std::endl;

    return false;
  }



private:
  unsigned _baseAddress;
  unsigned _epromControl;

  // Serial I/O
  bool _serialIO;
  CrcSerialHeader _lastHeader;
  unsigned _numberOfSerialWords;
  unsigned _serialBuffer[0x0800/4];

  // VME data
  CrcVmeRunData _vmeRunData;

  // BETrg data
  unsigned _beTrgCommands;
  CrcBeTrgRunData _beTrgRunData;
  CrcBeTrgConfigurationData _beTrgConfigurationData;
  CrcBeTrgEventData _beTrgEventData;
};

#endif
