#ifndef CrcReadout_HH
#define CrcReadout_HH

#include <iostream>
#include <fstream>

#include "CrcVmeRunData.hh"
#include "CrcBeRunData.hh"
#include "CrcBeTrgRunData.hh"
#include "CrcFeRunData.hh"

#include "OnlReadout.hh"
#include "RcdHeader.hh"
#include "SubInserter.hh"

#include "CrcVmeDevice.hh"


class CrcReadout : public OnlReadout {

public:
  CrcReadout() : _crateNumber(0) {
    _trgSlot=99;
    for(unsigned i(0);i<=21;i++) _device[i]=0;
  }

  CrcReadout(unsigned char c) : _crateNumber(c) {
    _trgSlot=99;
    for(unsigned i(0);i<=21;i++) _device[i]=0;
  }

  virtual ~CrcReadout() {
  }

  void device(CrcVmeDevice *c, unsigned char mask=0xff, bool trg=false) {
    if(c==0) return;

    unsigned slot(c->slot());
    if(slot>21) return;

    if(_printLevel>0) {
      if(_device[slot]==0) {
	std::cout << std::endl << std::endl 
		  << "CrcReadout::device()  CrcVmeDevice registered in slot ";
      } else {
	std::cout << std::endl << std::endl 
		  << "CrcReadout::device()  CrcVmeDevice replaced in slot ";
      }

      std::cout << slot << " with mask " << printHex(mask);
      if(trg) std::cout << " as BE-Trg";
      std::cout << std::endl;
    }

    _device[slot]=c;
    for(unsigned f(0);f<8;f++) _feEnable[slot][f]=(mask&(1<<f))!=0;
    if(trg) _trgSlot=slot;
  }

  void record(RcdRecord &r) {

    // Print initial record
    unsigned pl(4);

    if(_printLevel>pl) {
      std::cout << std::endl << std::endl << "Initial record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>pl+1) {
	SubAccessor ex(r);
	ex.print(std::cout);
	if(_printLevel>pl+2) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }


    // Check record type
    switch (r.recordType()) {


    // Run start and end are treated the same
    case RcdHeader::runStart:
    case RcdHeader::runEnd:
    case RcdHeader::runStop: {

      readSlwStartData(r);

      SubInserter inserter(r);

      // Loop over all the CRCs
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {
	  CrcLocation cl(_crateNumber,i,CrcLocation::vme);

	  // Handle VME
	  assert(_device[i]->reset());

	  CrcLocationData<CrcVmeRunData> *d(inserter.insert< CrcLocationData<CrcVmeRunData> >());
	  cl.crcComponent(CrcLocation::vme);
	  d->location(cl);
	  assert(_device[i]->readVmeRunData(*d->data())); // Unreliable EPROM read!
	  //_device[i]->readVmeRunData(*d->data());
	  if(_printLevel>1) d->print(std::cout);

	  // Handle BE
	  assert(_device[i]->beSoftReset());

	  CrcLocationData<CrcBeRunData> *b(inserter.insert< CrcLocationData<CrcBeRunData> >());
	  cl.crcComponent(CrcLocation::be);
	  b->location(cl);
	  assert(_device[i]->readBeRunData(*b->data()));
	  if(_printLevel>1) b->print(std::cout);

	  // Handle BE-Trg
	  if(i==_trgSlot) {
	    assert(_device[i]->beTrgSoftReset());

	    CrcLocationData<CrcBeTrgRunData> *t(inserter.insert< CrcLocationData<CrcBeTrgRunData> >());
	    cl.crcComponent(CrcLocation::beTrg);
	    t->location(cl);
	    assert(_device[i]->readBeTrgRunData(*t->data()));
	    if(_printLevel>1) t->print(std::cout);
	  }

	  // Handle FEs
	  assert(_device[i]->feSoftReset()); // This is an feBroadcast

	  for(unsigned f(0);f<8;f++) {
	    if(_feEnable[i][f]) {
	      CrcLocationData<CrcFeRunData> *e(inserter.insert< CrcLocationData<CrcFeRunData> >());
	      cl.crcComponent((CrcLocation::CrcComponent)f);
	      e->location(cl);
	      assert(_device[i]->readFeRunData(cl.crcComponent(),*e->data()));
	      if(_printLevel>1) e->print(std::cout);
	    }
	  }
	}
      }

      break;
    }

    // Configuration start is used to set up system
    case RcdHeader::configurationStart: {

      SubAccessor extracter(r);

      std::vector<const CrcReadoutConfigurationData*>
	bc(extracter.extract< CrcReadoutConfigurationData>());
      assert(bc.size()==1);
      _config=*(bc[0]);
      if(_printLevel>2) _config.print(std::cout) << std::endl;
      
      //CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
      
      // Handle VME
      // No CrcVmeConfigurationData

      // Handle BE
      std::vector<const CrcLocationData<CrcBeConfigurationData>*>
	b(extracter.extract< CrcLocationData<CrcBeConfigurationData> >());
      
      bool beDone[22];
      for(unsigned i(0);i<=21;i++) beDone[i]=false;
      
      for(unsigned j(0);j<b.size();j++) {
	if(b[j]->crateNumber()==_crateNumber &&
	   _device[b[j]->slotNumber()]!=0) {
	  
	  if(_printLevel>2) b[j]->print(std::cout);
	  assert(_device[b[j]->slotNumber()]->writeBeConfigurationData(*b[j]->data()));
	  beDone[b[j]->slotNumber()]=true;
	}
      }
      
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0 && !beDone[i]) {
	  
	}
      }
    
      /*  
      CrcBeConfigurationData r;
      assert(_device[i]->readBeConfigurationData(r));
      //if(r!=*b[j]->data()) {
      r.print(std::cout," ");
      //}
      */ 
   
      // Handle BE-Trg
      std::vector<const CrcLocationData<CrcBeTrgConfigurationData>*>
	bt(extracter.extract< CrcLocationData<CrcBeTrgConfigurationData> >());
      assert(bt.size()<=1);

      for(unsigned j(0);j<bt.size();j++) {
	if(bt[j]->crateNumber()==_crateNumber &&
	   _device[bt[j]->slotNumber()]!=0 &&
	   _trgSlot==bt[j]->slotNumber()) {
	  
	  if(_printLevel>2) bt[j]->print(std::cout);
	  assert(_device[bt[j]->slotNumber()]->writeBeTrgConfigurationData(*bt[j]->data()));
	}
      }
      
      // Handle FEs

      std::vector<const  CrcLocationData<CrcFeConfigurationData>*>
	f(extracter.extract< CrcLocationData<CrcFeConfigurationData> >());
      
      for(unsigned j(0);j<f.size();j++) {
	if(f[j]->crateNumber()==_crateNumber &&
	   _device[f[j]->slotNumber()]!=0 &&
	   _feEnable[f[j]->slotNumber()][f[j]->componentNumber()]) {
	
	  if(_printLevel>2) f[j]->print(std::cout);
	  assert(_device[f[j]->slotNumber()]->writeFeConfigurationData(f[j]->crcComponent(),*f[j]->data()));
	}
      }

      std::vector<const  CrcLocationData<CrcFeFakeEventData>*>
	fk(extracter.extract< CrcLocationData<CrcFeFakeEventData> >());
      
      for(unsigned j(0);j<fk.size();j++) {
	if(fk[j]->crateNumber()==_crateNumber &&
	   _device[fk[j]->slotNumber()]!=0 &&
	   _feEnable[fk[j]->slotNumber()][fk[j]->componentNumber()]) {
	
	  if(_printLevel>2) fk[j]->print(std::cout);
	  assert(_device[fk[j]->slotNumber()]->writeFeFakeEventData(fk[j]->crcComponent(),*fk[j]->data()));
	}
      }

      /*
	CrcFeConfigurationData r;
	assert(_device[i]->readFeConfigurationData(cl.crcComponent(),r));
	//if(r!=*b[j]->data()) {
	r.print(std::cout," ");
	//}
	*/
      
      // Read back configuration!

      /*

      // Flush any data out
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {
	  
	  // Handle BE
	  assert(_device[i]->clearBeFifos());

	  // Handle FE (broadcast)
	  assert(_device[i]->clearFeFifos());
	}
      }

      assert(_device[_trgSlot]->clearBeTrgFifos());

      // Enables triggers
      assert(_device[_trgSlot]->beTrgForceBusy(false));
      */

      _nEvents=0;
      break;
    }


    // Configuration end reads back setup
    case RcdHeader::configurationEnd:
    case RcdHeader::configurationStop: {

      SubInserter inserter(r);

      // Loop over all the CRCs
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {
	  CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);

	  // Handle VME
	  // No CrcVmeConfigurationData

	  // Handle BE
	  CrcLocationData<CrcBeConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeConfigurationData> >());
	  cl.crcComponent(CrcLocation::be);
	  b->location(cl);
	  assert(_device[i]->readBeConfigurationData(*b->data()));
	  if(_printLevel>2) b->print(std::cout);
	  
	  // Handle BE-Trg
	  if(i==_trgSlot) {
	    CrcLocationData<CrcBeTrgConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
	    cl.crcComponent(CrcLocation::beTrg);
	    b->location(cl);
	    assert(_device[i]->readBeTrgConfigurationData(*b->data()));
	    if(_printLevel>2) b->print(std::cout);
	  }

	  // Handle FEs
	  for(unsigned f(0);f<8;f++) {
	    if(_feEnable[i][f]) {
	      cl.crcComponent((CrcLocation::CrcComponent)f);
	      
	      CrcLocationData<CrcFeConfigurationData> *e(inserter.insert< CrcLocationData<CrcFeConfigurationData> >());
	      e->location(cl);
	      assert(_device[i]->readFeConfigurationData(cl.crcComponent(),*e->data()));
	      if(_printLevel>2) e->print(std::cout);

	      // Fake event ???
	    }
	  }
	}
      }
      break;
    }

    case RcdHeader::spillStart: {

      // Flush any data out
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {

	  // Handle BE
	  assert(_device[i]->clearBeFifos());

	  // Handle FE (broadcast)
	  assert(_device[i]->clearFeFifos());
	}
      }

      assert(_device[_trgSlot]->clearBeTrgFifos());

      // Enables triggers
      assert(_device[_trgSlot]->beTrgForceBusy(false));

      _nEvents=0;
      break;
    }

    case RcdHeader::event: {

      SubInserter inserter(r);

      // Loop over all the CRCs
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {

	  CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vme);
	  
	  // Handle VME
	  if(_config.vmePeriod()>0 && (_nEvents%_config.vmePeriod())==0) {
	    cl.crcComponent(CrcLocation::vme);
	    CrcLocationData<CrcVmeEventData> *bv(inserter.insert< CrcLocationData<CrcVmeEventData> >());
	    bv->location(cl);
	    assert(_device[i]->readVmeEventData(*bv->data()));
	    if(_printLevel>4) bv->print(std::cout);
	  }

	  // Handle BE
	  if(_config.bePeriod()>0 && (_nEvents%_config.bePeriod())==0) {
	    cl.crcComponent(CrcLocation::be);
	    CrcLocationData<CrcBeEventData> *b(inserter.insert< CrcLocationData<CrcBeEventData> >());
	    b->location(cl);
	    assert(_device[i]->readBeEventData(*b->data()));
	    if(_printLevel>4) b->print(std::cout);
	  }

	  // Handle BE-Trg
	  if(i==_trgSlot) {
	    cl.crcComponent(CrcLocation::beTrg);
	    CrcLocationData<CrcBeTrgEventData> *t(inserter.insert< CrcLocationData<CrcBeTrgEventData> >());
	    t->location(cl);
	    //assert(_device[i]->readBeTrgEventData(*t->data()));
	    _device[i]->readBeTrgEventData(*t->data(),_config.beTrgSquirt()); // Not assert because of FIFO
	    inserter.extend(t->data()->numberOfFifoWords()*4); // For Be-Trg FIFO
	    if(_printLevel>4) t->print(std::cout);
	  }

	  // Handle FEs
	  if(_config.fePeriod()>0 && (_nEvents%_config.fePeriod())==0) {
	    for(unsigned f(0);f<8;f++) {
	      if(_feEnable[i][f]) {
		cl.crcComponent((CrcLocation::CrcComponent)f);
		
		CrcLocationData<CrcFeEventData> *e(inserter.insert< CrcLocationData<CrcFeEventData> >());
		e->location(cl);
		assert(_device[i]->readFeEventData(cl.crcComponent(),*e->data()));
		if(_printLevel>4) e->print(std::cout);
	      }
	    }
	  }

	  // Handle Vlink/
	  /*
	  cl.crcComponent(CrcLocation::be);
	  CrcLocationData<CrcVlinkEventData> *l(inserter.insert< CrcLocationData<CrcVlinkEventData> >());
	  l->location(cl);
	  assert(_device[i]->readVlinkEventData(*l->data(),true));
	  inserter.extend(l->data()->numberOfWords()*4);
	  if(_printLevel>4) l->print(std::cout);
	  */
	}
      }

      if(_config.vlinkSpill()) _events[_nEvents]=&r;
      else                     readVlinkEventData(r);
      _nEvents++;

      break;
    }

    case RcdHeader::spillEnd:
    case RcdHeader::spillStop: {

      cout << "About to read " << _nEvents << " events from Vlink" << endl;
      if(_config.vlinkSpill() && _nEvents>0) {
	UtlTime start;
	start.update();

	for(unsigned i(0);i<_nEvents;i++) {
	  readVlinkEventData(*(_events[i]));
	}

	UtlTime finish;
	finish.update();
	cout << "Done " << _nEvents << " in " << (finish-start).deltaTime() << " sec, "
	     << _nEvents/(finish-start).deltaTime() << " events/sec, "
	     << (finish-start).deltaTime()/_nEvents << " secs/event" << std::endl;
      }

      // Check for any remaining Vlink data
      RcdArena arena;
      arena.deleteData();
      arena.recordType(RcdHeader::event);

      readVlinkEventData(arena);

      unsigned nw(0);
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) nw+=3; // SubHeader, CrcLocation, No of Words = 0
      }

      if(arena.numberOfWords()>nw) {
	cout << "ERROR ERROR ERROR" << endl;
	arena.print(std::cout);
      }

      break;
    }

    case RcdHeader::startUp: {
      readSlwStartData(r);
      break;
    }

    case RcdHeader::slowControl: {
      SubAccessor extracter(r);

      std::vector<const CrcLocationData<CrcAdm1025SlowControlsData>*>
	a(extracter.extract< CrcLocationData<CrcAdm1025SlowControlsData> >());
      
      for(unsigned j(0);j<a.size();j++) {
	if(a[j]->crateNumber()==_crateNumber &&
	   _device[a[j]->slotNumber()]!=0) {
	  
	  if(_printLevel>2) a[j]->print(std::cout);
	  assert(_device[a[j]->slotNumber()]->writeAdm1025SlowControls(*a[j]->data()));
	}
      }
      
      std::vector<const CrcLocationData<CrcLm82SlowControlsData>*>
	b(extracter.extract< CrcLocationData<CrcLm82SlowControlsData> >());
      
      for(unsigned j(0);j<b.size();j++) {
	if(b[j]->crateNumber()==_crateNumber &&
	   _device[b[j]->slotNumber()]!=0) {
	  
	  if(_printLevel>2) b[j]->print(std::cout);
	  //assert(_device[b[j]->slotNumber()]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()));
	  _device[b[j]->slotNumber()]->writeLm82SlowControls(b[j]->crcComponent(),*b[j]->data()); // No assert as prototypes don't respond to FE
	}
      }
      
      break;
    }

    case RcdHeader::slowReadout: {
      SubInserter inserter(r);

      // Loop over all the CRCs
      for(unsigned i(0);i<=21;i++) {
	if(_device[i]!=0) {

	  // Handle VME
	  CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025);

	  CrcLocationData<CrcAdm1025SlowReadoutData> *a(inserter.insert< CrcLocationData<CrcAdm1025SlowReadoutData> >());
	  a->location(cl);
	  assert(_device[i]->readAdm1025SlowReadout(*a->data()));
	  if(_printLevel>3) a->print(std::cout);

	  cl.crcComponent(CrcLocation::vmeLm82);
	  CrcLocationData<CrcLm82SlowReadoutData> *d(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	  d->location(cl);
	  assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*d->data()));
	  if(_printLevel>3) d->print(std::cout);

	  // Handle BE
	  cl.crcComponent(CrcLocation::be);
	  CrcLocationData<CrcLm82SlowReadoutData> *b(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	  b->location(cl);
	  assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*b->data()));
	  if(_printLevel>3) b->print(std::cout);

	  // Handle FEs
	  for(unsigned f(0);f<8;f++) {
	    if(_feEnable[i][f]) {
	      cl.crcComponent((CrcLocation::CrcComponent)f);

	      CrcLocationData<CrcLm82SlowReadoutData> *e(inserter.insert< CrcLocationData<CrcLm82SlowReadoutData> >());
	      e->location(cl);
	      //assert(_device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()));
	      _device[i]->readLm82SlowReadout(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	      if(_printLevel>3) e->print(std::cout);
	    }
	  }
	}
      }
      break;
    }

    default: {
      break;
    }
    };


    // Print final record
    if(_printLevel>pl) {
      std::cout << std::endl << std::endl << "Final record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>pl+1) {
	SubAccessor ex(r);
	ex.print(std::cout);
	if(_printLevel>pl+2) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }
  }

  bool readVlinkEventData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::be,0);
	CrcLocationData<CrcVlinkEventData> *l(inserter.insert< CrcLocationData<CrcVlinkEventData> >());
	l->location(cl);
	assert(_device[i]->readVlinkEventData(*l->data(),_config.vlinkBlt()));
	inserter.extend(l->data()->numberOfWords()*4);
	if(_printLevel>4) l->print(std::cout);
	if(_printLevel>5) l->data()->print(std::cout,"",true);
      }
    }

    return true;
  }

  bool readSlwStartData(RcdRecord &r) {
    SubInserter inserter(r);
    
    // Loop over all the CRCs
    for(unsigned i(0);i<=21;i++) {
      if(_device[i]!=0) {
	CrcLocation cl(_crateNumber,_device[i]->slot(),CrcLocation::vmeAdm1025,0);

	// Handle VME
	cl.crcComponent(CrcLocation::vme);

	CrcLocationData<CrcAdm1025StartupData> *a(inserter.insert< CrcLocationData<CrcAdm1025StartupData> >());
	a->location(cl);
	assert(_device[i]->readAdm1025Startup(*a->data()));
	if(_printLevel>4) a->print(std::cout);

	CrcLocationData<CrcAdm1025SlowControlsData> *as(inserter.insert< CrcLocationData<CrcAdm1025SlowControlsData> >());
	as->location(cl);
	assert(_device[i]->readAdm1025SlowControls(*as->data()));
	if(_printLevel>4) as->print(std::cout);

	CrcLocationData<CrcLm82StartupData> *d(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	d->location(cl);
	assert(_device[i]->readLm82Startup(CrcLocation::vmeLm82,*d->data()));
	if(_printLevel>4) d->print(std::cout);

	CrcLocationData<CrcLm82SlowControlsData> *ds(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	ds->location(cl);
	assert(_device[i]->readLm82SlowControls(CrcLocation::vmeLm82,*ds->data()));
	if(_printLevel>4) ds->print(std::cout);

	// Handle BE
	cl.crcComponent(CrcLocation::be);

	CrcLocationData<CrcLm82StartupData> *b(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	b->location(cl);
	assert(_device[i]->readLm82Startup(cl.crcComponent(),*b->data()));
	if(_printLevel>4) b->print(std::cout);

	CrcLocationData<CrcLm82SlowControlsData> *bs(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	bs->location(cl);
	assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*bs->data()));
	if(_printLevel>4) bs->print(std::cout);

	// Handle FEs
	for(unsigned f(0);f<8;f++) {
	  if(_feEnable[i][f]) {
	    cl.crcComponent((CrcLocation::CrcComponent)f);

	    CrcLocationData<CrcLm82StartupData> *e(inserter.insert< CrcLocationData<CrcLm82StartupData> >());
	    e->location(cl);
	    //assert(_device[i]->readLm82Startup(cl.crcComponent(),*e->data()));
	    _device[i]->readLm82Startup(cl.crcComponent(),*e->data()); // No assert as prototypes don't respond
	    if(_printLevel>4) e->print(std::cout);

	    CrcLocationData<CrcLm82SlowControlsData> *es(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	    es->location(cl);
	    //assert(_device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()));
	    _device[i]->readLm82SlowControls(cl.crcComponent(),*es->data()); // No assert as prototypes don't respond
	    if(_printLevel>4) es->print(std::cout);
	  }
	}
      }
    }

    return true;
  }


private:
  unsigned char _crateNumber;
  CrcVmeDevice *_device[22];
  bool _feEnable[22][8];
  unsigned _trgSlot;

  CrcReadoutConfigurationData _config;

  unsigned _nEvents;
  RcdRecord* _events[2048];
};

#endif
