#ifndef CrcSerialHeader_HH
#define CrcSerialHeader_HH

#include <iostream>
#include <string>

#include "UtlPrintHex.hh"


class CrcSerialHeader {

public:
  enum Target {
    null,
    fe0=1,fe1,fe2,fe3,fe4,fe5,fe6,fe7,
    beTrg,be,feBroadcast=15,
    endOfTargetEnum
  };

  enum Designator {
    designator00,designator01,designator02,designator03,
    designator04,designator05,designator06,designator07,
    designator08,designator09,designator10,designator11,
    designator12,designator13,designator14,designator15,
    designator16,designator17,designator18,designator19,
    designator20,designator21,designator22,designator23,
    designator24,designator25,designator26,designator27,
    designator28,designator29,designator30,designator31,

    // Target = All
    //resetLevel2=0,
    //resetLevel3,
    //firmwareId,
    //lm82Control,
    //lm82Status,
    
    // Target = BE
    beTriggerSelect=1,
    beMode=2,
    beTest=4,
    beFeDataEnable=5,
    beReadoutControl=6,
    beRunControl=7,
    beSoftTrigger=8,
    beSoftReset=9,
    beDaqId=10,
    beLm82Control=12,
    beFeTrgEnable=15,

    beStatus=16,
    beL1aCounter=17,
    beBxCounter=18,
    beQdrFrameCounter=19,
    beFirmwareId=21,
    beQdrDataCounter=22,
    beTotalFrameCounter=23,
    beLm82Status=25,    


    // Target = BE-trigger
    beTrgInputEnable=1,
    beTrgInputStatus=2,
    beTrgInputCatch=3,
    beTrgGeneralEnable=4,
    beTrgTriggerCounter=5,
    beTrgOscillationPeriod=6,
    beTrgBurstCounter=7,
    beTrgPrebusyTriggerCounter=8,
    beTrgFirmwareId=9,
    beTrgFirmwareDate=10,
    beTrgControl=11,
    beTrgCommands=12,
    beTrgFifo=13,
    beTrgFifoStatus=14,
    beTrgQdrAddress=15, // Obsolete
    beTrgConfiguration=15,
    beTrgBusyTimeout=16,
    beTrgBurstTimeout=17,
    beTrgPrebusyCatch=18,
    beTrgAnd0Enable=19,
    beTrgAnd1Enable=20,
    beTrgAnd2Enable=21,
    beTrgAnd3Enable=22,
    beTrgExtBeamMode=23,
    beTrgInputInvert=24,
    beTrgQdrConfiguration=25,
    beTrgTestLength=26, // Obsolete
    beTrgSignalCatch=27,
    beTrgSequencerControl=28,
    beTrgFifoSquirt=29,
    beTrgRamData=30,
    beTrgRamAddress=31,


    // Target = FE
    feFirmwareId=3,
    feLm82Control=5,
    feSoftTrigger=6,
    feDataOut=7,
    feAddress=8,
    feLinkArray=9,
    feVfeType=10,
    feSoftReset=12,
    feFifoReset=13,
    feLm82Status=27,    
    feSpyRegister=28,    
    feDataIn=29,
    feTriggerCounter=30,

    endOfDesignatorEnum
  };

  CrcSerialHeader() : _length(0), _designator(0x81), _target(0x20) {
  }

  CrcSerialHeader(Target t, Designator d, unsigned l) : _length(0), _designator(0x81), _target(0x20) {
    target(t);
    designator(d);
    length(l);
  }

  unsigned char* packet() {
    return &_designator;
  }

  bool readback() const {
    if((_designator&0x40)==0) return false;
    return true;
  }

  void readback(bool r) {
    if(r) _designator|=  0x40;
    else  _designator&= ~0x40;
  }

  Designator designator() const {
    return static_cast<Designator>((_designator>>1)&0x1f);
  }

  void designator(Designator d) {
    _designator&=0xc1;
    _designator|=((static_cast<unsigned>(d)&0x1f)<<1);
  }

  unsigned short length() const {
    return _length;
  }

  void length(unsigned short l) {
    _length=l;
  }

  void print(std::ostream &o, std::string s="") const {
    o << s << "CrcSerialHeader::print()  Bytes = 0x " << std::hex;

    const unsigned char *b((const unsigned char *const)this);
    for(unsigned i(0);i<sizeof(CrcSerialHeader);i++) {
      if(b[sizeof(CrcSerialHeader)-1-i]<0x10) o << "0";
      o << static_cast<unsigned>(b[sizeof(CrcSerialHeader)-1-i]) << " ";
    }
    o << std::dec << std::endl;

    if(valid()) o << s << " Header bits are valid" << std::endl;
    else        o << s << " Header bits are invalid" << std::endl;

    if(readback()) o << s << " Readback bit set" << std::endl;
    else           o << s << " Readback bit not set" << std::endl;

    unsigned short t((unsigned short)target());
    o << s << " Target     = " << printHex(t,false) << " = ";
    if(t<endOfTargetEnum) o << _targetName[t] << std::endl;
    else                  o << "unknown" << std::endl;

    o << s << " Designator = " << printHex((unsigned short)designator())
      << std::endl;
    o << s << " Length     = " << printHex(_length) << std::endl;
  }

  Target target() const {
    return static_cast<Target>(_target&0x1f);
  }

  static bool feTarget(Target t) {
    if(t==fe0 || t==fe1 || t==fe2 || t==fe3 ||
       t==fe4 || t==fe5 || t==fe6 || t==fe7 ||
       t==feBroadcast) return true;
    return false;
  }

  bool feTarget() const {
    return feTarget(target());
  }

  void target(Target t) {
    _target=0x20;
    _target|=(static_cast<unsigned>(t)&0x1f);
  }

  bool valid() const {
    if((_target&0xe0)==0x20 && (_designator&0x81)==0x81) return true;
    return false;
  }


private:
  unsigned short _length;
  unsigned char _designator;
  unsigned char _target;

  static const std::string _targetName[endOfTargetEnum];
};

const std::string CrcSerialHeader::_targetName[]={
  "null",
  "fe0","fe1","fe2","fe3","fe4","fe5","fe6","fe7",
  "beTrg","be",
  "undefined",
  "undefined",
  "undefined",
  "undefined",
  "feBroadcast"
};

#endif
