#ifndef DaqCounter_HH
#define DaqCounter_HH

#include <vector>
#include <fstream>
#include <iostream>

#include "RcdUserRW.hh"

#include "SubInserter.hh"
#include "SubAccessor.hh"

#include "DaqStateData.hh"
#include "DaqStartUp.hh"
#include "DaqRunStart.hh"
#include "DaqRunEnd.hh"
#include "DaqConfigurationStart.hh"
#include "DaqConfigurationEnd.hh"
#include "DaqSpillStart.hh"
#include "DaqSpillEnd.hh"
#include "DaqBufferEnd.hh"

#include "OnlRunNumber.hh"


class DaqCounter : public RcdUserRW {

public:
  DaqCounter() {
  }

  virtual ~DaqCounter() {
  }

  bool record(RcdRecord &r) {

    // Get time of record
    _headerTime=r.recordTime().seconds();

    // Check record type
    switch (r.recordType()) {

    case RcdHeader::startUp: {
      _count[0]=0;

      SubAccessor accessor(r);
      std::vector<const DaqStartUp*> v(accessor.extract<DaqStartUp>());
      assert(v.size()<=1);

      _maxCount[0]=0xffffffff;
      _runType=DaqRunStart::test;
      _runSubtype=0;

      if(v.size()==1) {
	if(_printLevel>0) {
	  std::cout << "DaqCounter::record()  DaqStartUp subrecord"
		    << std::endl;
	  v[0]->print(std::cout," ");
	}
      
	_maxCount[0]=v[0]->maximumNumberOfRunsInJob();
	_runType=v[0]->runType();
	_runSubtype=v[0]->runSubtype();
      }

      break;
    }
      
    case RcdHeader::runStart: {
      _count[1]=0;
      _count[2]=0;
      _count[3]=0;
      _count[4]=0;
      _count[5]=0;
      _time[0]=_headerTime;

      SubInserter inserter(r);
      DaqRunStart *d(inserter.insert<DaqRunStart>(true));
      
      _runNumber=onlReadRunNumber();
      d->runNumber(_runNumber);
      d->runType(_runType);
      d->runSubtype(_runSubtype);

      setRun(*d);

      _maxCount[1]=d->maximumNumberOfConfigurationsInRun();
      _maxCount[2]=d->maximumNumberOfSpillsInRun();
      _maxCount[3]=d->maximumNumberOfSpillsInRun();
      _maxCount[4]=d->maximumNumberOfEventsInRun();
      _maxCount[5]=d->maximumNumberOfEventsInRun();
      _maxTime[0]=d->maximumTimeOfRun();
      
      if(_printLevel>0) {
	std::cout << "DaqCounter::record()  DaqRunStart subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }
      
    case RcdHeader::runEnd:
    case RcdHeader::runStop: {
      SubInserter inserter(r);
      DaqRunEnd *d(inserter.insert<DaqRunEnd>(true));
      
      d->runNumber(_runNumber);
      d->actualNumberOfConfigurationsInRun(_count[1]);
      d->actualNumberOfSpillsInRun(_count[2]);
      d->actualNumberOfEventsInRun(_count[5]);

      _count[0]++;

      if(_printLevel>0) {
	std::cout << "DaqCounter::record()  DaqRunEnd subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }

    case RcdHeader::configurationStart: {
      _count[6]=0;
      _count[7]=0;
      _count[8]=0;
      _count[9]=0;
      _time[1]=_headerTime;

      SubInserter inserter(r);
      DaqConfigurationStart *d(inserter.insert<DaqConfigurationStart>(true));
      
      d->configurationNumberInRun(_count[1]);
      
      setConfiguration(*d);

      _readoutMode=d->readoutMode();
      _maxCount[6]=d->maximumNumberOfSpillsInConfiguration();
      _maxCount[7]=d->maximumNumberOfSpillsInConfiguration();
      _maxCount[8]=d->maximumNumberOfEventsInConfiguration();
      _maxCount[9]=d->maximumNumberOfEventsInConfiguration();
      _maxTime[1]=d->maximumTimeOfConfiguration();

      if(_printLevel>2) {
	std::cout << "DaqCounter::record()  DaqConfigurationStart subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
     
      // For SM=slow, this acts as a spillStart but with no saved subrecord
      //if(_readoutMode==0) {
      //DaqSpillStart d;
      //spillStart(d);
      //}

      break;
    }

    case RcdHeader::configurationEnd:
    case RcdHeader::configurationStop: {
      SubInserter inserter(r);
      DaqConfigurationEnd *d(inserter.insert<DaqConfigurationEnd>(true));
      
      d->configurationNumberInRun(_count[1]);
      d->actualNumberOfSpillsInConfiguration(_count[6]);
      d->actualNumberOfEventsInConfiguration(_count[9]);

      _count[1]++;
      
      if(_printLevel>2) {
	std::cout << "DaqCounter::record()  DaqConfigurationEnd subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }

    case RcdHeader::spillStart: {
      _count[10]=0;
      _count[11]=0;
      _time[2]=_headerTime;
      _time[3]=_headerTime;
    
      SubInserter inserter(r);
      DaqSpillStart *d(inserter.insert<DaqSpillStart>(true));

      d->spillNumberInRun(_count[2]);
      d->spillNumberInConfiguration(_count[6]);
    
      setSpill(*d);
    
      _maxCount[10]=d->maximumNumberOfEventsInSpill();
      _maxCount[11]=d->maximumNumberOfEventsInSpill();
      _maxTime[2]=d->maximumTimeOfSpill();
      _maxTime[3]=d->maximumTimeOfBuffer();
      
      if(_printLevel>4) {
	std::cout << "DaqCounter::record()  DaqSpillStart subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }
      
    case RcdHeader::spillEnd:
    case RcdHeader::spillStop: {
      // Equivalent to bufferStart

      SubInserter inserter(r);
      DaqSpillEnd *d(inserter.insert<DaqSpillEnd>(true));

      d->spillNumberInRun(_count[2]);
      d->spillNumberInConfiguration(_count[6]);
      d->actualNumberOfPreEventsInSpill(_count[10]);

      _count[2]++;
      _count[6]++;
 
      if(_printLevel>4) {
	std::cout << "DaqCounter::record()  DaqSpillEnd subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }

      break;
    }
      
    case RcdHeader::bufferEnd: 
    case RcdHeader::bufferStop: {
      SubInserter inserter(r);
      DaqBufferEnd *d(inserter.insert<DaqBufferEnd>(true));

      d->bufferNumberInRun(_count[3]);
      d->bufferNumberInConfiguration(_count[7]);
      d->actualNumberOfEventsInBuffer(_count[11]);
      
      _count[3]++;
      _count[7]++;
      
      if(_printLevel>4) {
	std::cout << "DaqCounter::record()  DaqBufferEnd subrecord"
		<< std::endl;
	d->print(std::cout," ");
      }
      
      // For mode=0, this acts as a spillStart but with no saved subrecord
      //if(_readoutMode==0) {
      //DaqSpillStart d;
      //spillStart(d);
      //}

      break;
    }

    case RcdHeader::preEvent: {
      SubInserter inserter(r);
      DaqEvent *d(inserter.insert<DaqEvent>(true));
      
      d->eventNumberInRun(_count[4]);
      d->eventNumberInConfiguration(_count[8]);
      d->eventNumberInSpill(_count[10]);
      
      _count[4]++;
      _count[8]++;
      _count[10]++;

      if(_printLevel>6) {
	std::cout << "DaqCounter::record()  PreEvent DaqEvent subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }
      
    case RcdHeader::event: {
      SubInserter inserter(r);
      DaqEvent *d(inserter.insert<DaqEvent>(true));
      
      d->eventNumberInRun(_count[5]);
      d->eventNumberInConfiguration(_count[9]);
      d->eventNumberInSpill(_count[11]);
      
      _count[5]++;
      _count[9]++;
      _count[11]++;
      
      if(_printLevel>6) {
	std::cout << "DaqCounter::record()  Event DaqEvent subrecord"
		  << std::endl;
	d->print(std::cout," ");
      }
      
      break;
    }
      
    default: {
      break;
    }
    };

    return true;
  }

  bool endJob() {
    bool reply= (_count[0]>=_maxCount[0]);
    if(reply) std::cout << "***** endJob() is true *****" << std::endl;
    return reply;
  }

  bool endRun() {
    bool reply= endJob() ||
      (_count[1]>=_maxCount[1]) ||
      (_count[2]>=_maxCount[2] && _count[3]>=_maxCount[3]) ||
      (_count[4]>=_maxCount[4] && _count[5]>=_maxCount[5]);
    if(reply) std::cout << "***** endRun() is true *****" << std::endl;
    return reply;
  }

  bool stopRun() {
    bool reply= (_headerTime-_time[0]>=_maxTime[0]);
    if(reply) std::cout << "***** stopRun() is true *****" << std::endl;
    return reply;
  }

  bool endConfiguration() {
    bool reply= endRun() ||
      (_count[6]>=_maxCount[6] && _count[7]>=_maxCount[7]) ||
      (_count[8]>=_maxCount[8] && _count[9]>=_maxCount[9]);
    if(reply) std::cout << "***** endConfiguration() is true *****" << std::endl;
    return reply;
  }

  bool stopConfiguration() {
    bool reply= (_headerTime-_time[1]>=_maxTime[1]);
    if(reply) std::cout << "***** stopConfiguration() is true *****" << std::endl;
    return reply;
  }

  bool endSpill() {
    bool reply= endConfiguration() ||
      (_count[10]>=_maxCount[10]);
    if(reply) std::cout << "***** endSpill() is true *****" << std::endl;
    return reply;
  }

  bool stopSpill() {
    bool reply= (_headerTime-_time[2]>=_maxTime[2]);
    if(reply) std::cout << "***** stopSpill() is true *****" << std::endl;
    return reply;
  }

  bool endBuffer() {
    bool reply= endConfiguration() ||
      (_count[11]>=_maxCount[11]);
    if(reply) std::cout << "***** endBuffer() is true *****" << std::endl;
    return reply;
  }

  bool stopBuffer() {
    bool reply= (_headerTime-_time[3]>=_maxTime[3]);
    if(reply) std::cout << "***** stopBuffer() is true *****" << std::endl;
    return reply;
  }

  unsigned readoutMode() const {
    return _readoutMode;
  }

  virtual void setRun(DaqRunStart &d) const {
  }

  virtual void setConfiguration(DaqConfigurationStart &d) const {
  }

  virtual void setSpill(DaqSpillStart &d) const {
  }


protected:
  DaqRunStart::RunType _runType;
  unsigned short _runSubtype;

private:
  unsigned _runNumber;
  unsigned _readoutMode;

  unsigned _count[12];
  unsigned _maxCount[12];

  unsigned _headerTime;
  unsigned _time[4];
  unsigned _maxTime[4];
};

#endif
