//
// $Id: DhcVmeDevice.hh,v 1.1 2008/06/27 10:34:05 meyern Exp $
//

#ifndef DhcVmeDevice_HH
#define DhcVmeDevice_HH

#include <iostream>
#include <vector>

// hal
#include "VMEDevice.hh"
#include "VMEAddressTable.hh"
#include "VMEBusAdapterInterface.hh"

#include "DhcEventData.hh"
#include "DhcBeConfigurationData.hh"
#include "DhcFeConfigurationData.hh"
#include "DhcSerialHeader.hh"
#include "DhcSerialCommand.hh"

const uint32_t CIRCULAR_BUFFER_BASE = 0x0800000;
const uint32_t CIRCULAR_BUFFER_SIZE = 0x1800000;
const uint32_t CIRCULAR_BUFFER_RANGE = CIRCULAR_BUFFER_BASE+CIRCULAR_BUFFER_SIZE;
const uint32_t CIRCULAR_BUFFER_PAGE = 0x4000;


class DhcVmeDevice : public HAL::VMEDevice {

public:
  DhcVmeDevice(HAL::VMEAddressTable &t,
	       HAL::VMEBusAdapterInterface &b, unsigned a)
    : HAL::VMEDevice(t,b,a<<27), _slot(a) {
  }

  unsigned slot() const {
    return _slot;
  }

  bool alive() {
    uint32_t value;
    try {
      // if configured from backup firmware...
      read("CMD_FPGA_backup",&value);
      if (value) {
	// configure from standard firmware
	write("FPGA_prg",0xABCDEF00,HAL::HAL_NO_VERIFY);
	sleep(5);
      }
      // check firmware version
      read("CSR_firmware_rev",&value);
      if (value==0)
	return false;
      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      return false;

    } catch ( std::exception e ) {
      return false;
    }
  }

  bool reset() {

    try {
      // reset board
      write("CMD_reset_all",1,HAL::HAL_NO_VERIFY);
      // reset local buffer pointer
      _local_rp=CIRCULAR_BUFFER_BASE;
      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      return false;

    } catch ( std::exception e ) {
      return false;
    }
  }

  bool serialWrite(const DhcSerialHeader *h) {
    if(h==0) return false;

    uint32_t *b((uint32_t*)h);

    unsigned nWords = (h->len()+4)/4;
    unsigned dLink = h->dlink();
    unsigned width = (h->cmd() == DhcSerialHeader::CMD_Read) ? 1
      : (nWords<3) ? 2 : 3;

    for (unsigned i(0); i<nWords; i++)
      write("SLOW_CTRL_Tx_buffer", b[i], HAL::HAL_NO_VERIFY, 0x80*dLink + 4*i);

    write("SLOW_CTRL_Tx", width << (dLink * 2), HAL::HAL_NO_VERIFY);

    // poll until Tx done, throw exception if timeout
    pollItem("SLOW_CTRL_Tx_Busy",0,5000,&width);

    return true;
  }

  bool serialRead(const DhcSerialHeader *h) {
    if(h==0) return false;

    if (!serialWrite(h))
      return false;

    unsigned nWords = (h->len()+4)/4;
    unsigned dLink = h->dlink();
    unsigned value;

    // poll until Rx done, throw exception if timeout
    pollItem("SLOW_CTRL_Rx",0,5000,&value,HAL::HAL_POLL_UNTIL_DIFFERENT);

    UtlPack *p((UtlPack*)h);
    for (unsigned i(0); i<nWords; i++) {
      read("SLOW_CTRL_Rx_buffer", &value, 0x40*dLink);

      UtlPack r(value);
      if (i<2)
	p[i].halfWord(0, r.halfWord(0));
      if (i>0)
	p[i].halfWord(1, r.halfWord(1));
    }
    return true;
  }

  bool readFeConfigurationData(DhcLocation::DhcComponent c, DhcFeConfigurationData &d) {

    DhcSerialCommand<1> s(DhcSerialHeader::REG_PLSR,
			  DhcSerialHeader::CMD_Read,
			  *d.chipid(), c);

    if (!serialRead(&s)) return false;
    d.plsr(s.data());

    s.reg(DhcSerialHeader::REG_IntD);
    if (!serialRead(&s)) return false;
    d.intd(s.data());

    s.reg(DhcSerialHeader::REG_Shp2D);
    if (!serialRead(&s)) return false;
    d.shp2(s.data());

    s.reg(DhcSerialHeader::REG_Shp1D);
    if (!serialRead(&s)) return false;
    d.shp1(s.data());

    s.reg(DhcSerialHeader::REG_BlrD);
    if (!serialRead(&s)) return false;
    d.blrd(s.data());

    s.reg(DhcSerialHeader::REG_VtnD);
    if (!serialRead(&s)) return false;
    d.vtnd(s.data());

    s.reg(DhcSerialHeader::REG_VtpD);
    if (!serialRead(&s)) return false;
    d.vtpd(s.data());

    s.reg(DhcSerialHeader::REG_DCR);
    if (!serialRead(&s)) return false;
    d.dcr(s.data());

    return true;
  }

  bool writeFeConfigurationData(DhcLocation::DhcComponent c, const DhcFeConfigurationData &d) {

    DhcSerialCommand<1> s(DhcSerialHeader::REG_PLSR,
			  DhcSerialHeader::CMD_Write,
			  *d.chipid(), c);
    s.data(d.plsr());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_IntD);
    s.data(d.intd());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_Shp2D);
    s.data(d.shp2());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_Shp1D);
    s.data(d.shp1());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_BlrD);
    s.data(d.blrd());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_VtnD);
    s.data(d.vtnd());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_VtpD);
    s.data(d.vtpd());
    if (!serialWrite(&s)) return false;

    s.reg(DhcSerialHeader::REG_DCR);
    s.data(d.dcr());
    if (!serialWrite(&s)) return false;

    DhcSerialCommand<8> sl(DhcSerialHeader::REG_Inject,
			  DhcSerialHeader::CMD_Write,
			  *d.chipid(), c);
    sl.data(d.inj());
    if (!serialWrite(&sl)) return false;

    sl.reg(DhcSerialHeader::REG_Kill);
    sl.data(d.kill());
    if (!serialWrite(&sl)) return false;

    return true;
  }

  bool readBeConfigurationData(DhcLocation::DhcComponent c, DhcBeConfigurationData &d) {
    uint32_t value;

    read("CSR",&value);
    d.csr(value);

    read("DCON_enable",&value);
    d.dconEnable(value);

    return true;
  }

  bool writeBeConfigurationData(const DhcBeConfigurationData &d) {

    // disable run mode
    write("CSR_set_run_mode", 1,HAL::HAL_NO_VERIFY);

    // enable selected inputs
    write("DCON_enable",d.dconEnable(),HAL::HAL_NO_VERIFY);

    // set selected options
    write("CSR",d.setOptions(),HAL::HAL_NO_VERIFY);

    return true;
  }

  bool readEventData(DhcEventData &d) {
    return bufferRead(d);
  }

  bool bufferRead(DhcEventData &d) {
    d.numberOfWords(0);
    d.numberOfWords(bufferRead(d.data()));
    return true;
  }

  unsigned bufferRead(unsigned *d) {
    if(d==0) return 0;

    uint32_t Event_wp,  Event_rp;
    int nBytes(0);

    // get write p
    read("Event_wp",&Event_wp);

    // get read p
    read("MEM_PAGE",&Event_rp);
    Event_rp <<= 14;

    // if write p > current p, try to read block
    if (Event_wp > _local_rp) {
      // something to read
      nBytes = Event_wp - _local_rp;
      if (nBytes<0)
	// write p has wrapped around
	nBytes += CIRCULAR_BUFFER_SIZE;

      // do the block transfer
      readBlock("BLT", nBytes, (char*)(d), HAL::HAL_DO_INCREMENT, _local_rp);

      // advance current p
      _local_rp += nBytes;
 
      // advance read p, if necessary
      int nPages((_local_rp - Event_rp)/CIRCULAR_BUFFER_PAGE);
      for (int i(0); i<nPages; i++)
	// advance one page
	write("MEM_PAGE", 0, HAL::HAL_NO_VERIFY);

      if (_local_rp > CIRCULAR_BUFFER_RANGE)
	// pointer has wrapped around
	_local_rp -= CIRCULAR_BUFFER_SIZE;

      // return word count
      return nBytes/sizeof(uint32_t);
    }
    else
      // nothing to read
      return 0;
  }

  void print(std::ostream &o) {
    uint32_t value;
    o << "DhcVmeDevice::print()" << std::endl;

    read("CMD_FPGA_backup",&value);
    o << std::hex << "  FirmwareSector = 0x" << value << std::dec << std::endl;
    read("CSR_firmware_rev",&value);
    o << std::hex << "  FirmwareVer = 0x" << value << std::dec << std::endl;
    read("DCON_enable",&value);
    o << std::hex << "  DCON_EnableMask = 0x" << value << std::dec << std::endl;

    o << std::endl;
  }

private:
  unsigned _printLevel;
  unsigned _slot;
  uint32_t _local_rp;

};

#endif // DhcVmeDevice_HH
