#ifndef DvrEmcVfeDac_HH
#define DvrEmcVfeDac_HH

#include <iostream>
#include <fstream>

// dual/inc/daq
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqSpillStart.hh"

// dual/inc/rcd
#include "RcdHeader.hh"

// dual/inc/sub
#include "SubInserter.hh"

// online/inc/onl
#include "OnlReadout.hh"

// online/inc/crc
#include "CrcVmeDevice.hh"



class DvrEmcVfeDac {

public:
  DvrEmcVfeDac() : _crateNumber(0xec), _slotMask(1<<12) {
    defaults();
  }

  DvrEmcVfeDac(unsigned char c, unsigned s) : _crateNumber(c), _slotMask(s) {
    defaults();
  }

  virtual ~DvrEmcVfeDac() {
  }

  void defaults() {
    // CrcReadoutConfigurationData
    _crcd.vmePeriod(0);
    _crcd.bePeriod(0);
    _crcd.fePeriod(0);
    _crcd.beTrgMode(1);
    _crcd.vlinkMode(3);

    // CrcLocationData<CrcBeConfigurationData>
    _cbcd.crateNumber(_crateNumber);
    _cbcd.slotBroadcast(true);
    _cbcd.crcComponent(CrcLocation::be);
    _cbcd.label(1);

    _cbcd.data()->mode(0x02);
    _cbcd.data()->test(0x00);
    _cbcd.data()->testLength(0);
    _cbcd.data()->trgEnables(0x1ff); // 0x1ff for J0
    //_cbcd.data()->trgEnables(0x2ff); // 0x2ff for internal

    // CrcLocationData<CrcBeTrgConfigurationData>
    _cbtcd.crateNumber(_crateNumber);
    _cbtcd.slotNumber(12);
    _cbtcd.crcComponent(CrcLocation::beTrg);
    _cbtcd.label(1);

    // CrcLocationData<EmcFeConfigurationData>
    _efcd.crateNumber(_crateNumber);
    _efcd.slotBroadcast(true);
    _efcd.crcComponent(CrcLocation::feBroadcast);
    _efcd.label(1);

    _efcd.data()->calibEnable(true);
    _efcd.data()->holdStart(30);

    unsigned nadc=18;
    _efcd.data()->holdWidth(nadc*4*700);
    _efcd.data()->vfeMplexClockPulses(nadc);
    _efcd.data()->frameSyncDelay(16);
    _efcd.data()->qdrDataDelay(3);
    
  }

  void printLevel(unsigned p) {
    _printLevel=p;
  }

  void record(RcdRecord &r) {

    // Print initial record
    unsigned pl(4);

    if(_printLevel>pl) {
      std::cout << std::endl << std::endl << "Initial record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>pl+1) {
	SubAccessor ex(r);
	ex.print(std::cout);
	if(_printLevel>pl+2) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }


    // Check record type
    switch (r.recordType()) {


    // Run start and end are treated the same
    case RcdHeader::runStart: {
      _drs.runType(0x80000);
      _drs.maximumNumberOfConfigurationsInRun(32*6);
      _drs.maximumNumberOfSpillsInRun(0xffffffff);
      _drs.maximumNumberOfEventsInRun(0xffffffff);

      SubInserter inserter(r);
      inserter.insert<DaqRunStart>(_drs);
      if(_printLevel>0) _drs.print(std::cout) << std::endl;

      _dcs.configurationNumberInRun(0);
      _dss.spillNumberInRun(0);
      break;
    }


    case RcdHeader::runEnd:
    case RcdHeader::runStop: {

      break;
    }

    // Configuration start is used to set up system
    case RcdHeader::configurationStart: {
      _dcs.maximumNumberOfSpillsInConfiguration(1);
      _dcs.maximumNumberOfEventsInConfiguration(0xffffffff);

      SubInserter inserter(r);
      inserter.insert<DaqConfigurationStart>(_dcs);
      if(_printLevel>1) _dcs.print(std::cout) << std::endl;

      _dss.spillNumberInConfiguration(0);

      const unsigned iConfiguration(_dcs.configurationNumberInRun());

      // Handle CRC readout control
      inserter.insert<CrcReadoutConfigurationData>(_crcd);
      if(_printLevel>2) _crcd.print(std::cout) << std::endl;
      
      // Handle VME
      // No CrcVmeConfigurationData

      // Handle BE
      inserter.insert< CrcLocationData<CrcBeConfigurationData> >(_cbcd);
      if(_printLevel>2) _cbcd.print(std::cout) << std::endl;

      // Only do BE-Trg for first board
      inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >(_cbtcd);
      if(_printLevel>2) _cbtcd.print(std::cout) << std::endl;

      // Handle FE
      unsigned dac(2*1024*(31-(iConfiguration/6)));
      unsigned enable(iConfiguration%6);

      _efcd.data()->dacData(CrcFeConfigurationData::bot,dac);
      _efcd.data()->vfeEnable(CrcFeConfigurationData::bot,enable,true);

      inserter.insert< CrcLocationData<EmcFeConfigurationData> >(_efcd);
      if(_printLevel>3) _efcd.print(std::cout) << std::endl;

      break;
    }


    // Configuration end reads back setup
    case RcdHeader::configurationEnd:
    case RcdHeader::configurationStop: {
      _dcs.increment();
      break;
    }

    case RcdHeader::spillStart: {
      _dss.maximumNumberOfEventsInSpill(64);

      SubInserter inserter(r);
      inserter.insert<DaqSpillStart>(_dss);
      if(_printLevel>4) _dss.print(std::cout) << std::endl;

      break;
    }

    case RcdHeader::spillEnd:
    case RcdHeader::spillStop: {
      _dss.increment();
      break;
    }

    default: {
      break;
    }
    };


    // Print final record
    if(_printLevel>pl) {
      std::cout << std::endl << std::endl << "Final record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>pl+1) {
	SubAccessor ex(r);
	ex.print(std::cout);
	if(_printLevel>pl+2) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }
  }


protected:
  unsigned _printLevel;

  unsigned char _crateNumber;
  unsigned _slotMask;

  DaqRunStart _drs;
  DaqConfigurationStart _dcs;
  DaqSpillStart _dss;

  CrcReadoutConfigurationData _crcd;
  CrcLocationData<CrcBeConfigurationData> _cbcd;
  CrcLocationData<CrcBeTrgConfigurationData> _cbtcd;
  CrcLocationData<EmcFeConfigurationData> _efcd;
};

#endif
