#ifndef SlwCercReadout_HH
#define SlwCercReadout_HH

#include <iostream>
#include <fstream>

#include "SlwCercLm82StartupData.hh"
#include "SlwCercLm82SlowControlsData.hh"
#include "SlwCercAdm1025StartupData.hh"
#include "SlwCercAdm1025SlowControlsData.hh"

#include "RcdReader.hh"
#include "SubHeader.hh"
#include "SubInserter.hh"
#include "SubExtracter.hh"

#include "CercVmeDevice.hh"

class SlwCercReadout {

public:
  SlwCercReadout(CercVmeDevice &c) : _cvd(c),  _printLevel(0) {
  }

  virtual ~SlwCercReadout() {
  }

  void printLevel(unsigned p) {
    _printLevel=p;
  }

  void readout(RcdRecord &r) {

    // Print initial record
    if(_printLevel>2) {
      std::cout << std::endl << std::endl << "Initial record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>3) {
	SubExtracter ex(r);
	ex.print(std::cout);
	if(_printLevel>4) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }


    // Check record type
    switch (r.recordType()) {


    // Start-up
    case SubHeader::startUp: {

      SubExtracter extracter(r);

      // Handle VME Lm82
      std::vector<const SlwCercLm82StartupData*>
        c(extracter.extract<SlwCercLm82StartupData>());
      assert(c.size()==1);
      if(_printLevel>3) c[0]->print(std::cout);
      _cvd.writeLm82Startup(*c[0]);

      // Handle VME ADM1025A
      std::vector<const SlwCercAdm1025StartupData*>
        d(extracter.extract<SlwCercAdm1025StartupData>());
      assert(d.size()==1);
      if(_printLevel>3) d[0]->print(std::cout);
      _cvd.writeAdm1025Startup(*d[0]);

      // Read back configuration!

      break;
    }


    // Slow controls
    case SubHeader::slowControl: {

      SubInserter inserter(r);

      // Handle VME Lm82
      SlwCercLm82SlowControlsData *b(inserter.insert<SlwCercLm82SlowControlsData>());
      b->crateNumber(0);
      b->slotNumber(7);
      b->cercComponent(CercLocationData::vme);
      _cvd.readLm82SlowControls(*b);
      if(_printLevel>3) b->print(std::cout);

      // Handle VME ADM1025A
      SlwCercAdm1025SlowControlsData *a(inserter.insert<SlwCercAdm1025SlowControlsData>());
      a->crateNumber(0);
      a->slotNumber(7);
      a->cercComponent(CercLocationData::vme);
      _cvd.readAdm1025SlowControls(*a);
      if(_printLevel>3) a->print(std::cout);

      break;
    }

    // Shutdown
    case SubHeader::shutdown: {

      SubInserter inserter(r);

      // Handle VME Lm82
      SlwCercLm82StartupData *b(inserter.insert<SlwCercLm82StartupData>());
      b->crateNumber(0);
      b->slotNumber(7);
      b->cercComponent(CercLocationData::vme);
      _cvd.readLm82Startup(*b);
      if(_printLevel>3) b->print(std::cout);

      // Handle VME ADM1025A
      SlwCercAdm1025StartupData *a(inserter.insert<SlwCercAdm1025StartupData>());
      a->crateNumber(0);
      a->slotNumber(7);
      a->cercComponent(CercLocationData::vme);
      _cvd.readAdm1025Startup(*a);
      if(_printLevel>3) a->print(std::cout);

      break;
    }

    default: {
      break;
    }
    };


    // Print final record
    if(_printLevel>2) {
      std::cout << std::endl << std::endl << "Final record" << std::endl;
      r.RcdHeader::print(std::cout);
      if(_printLevel>3) {
	SubExtracter ex(r);
	ex.print(std::cout);
	if(_printLevel>4) {
	  r.print(std::cout);
	}
      }
      std::cout << std::endl << std::endl;
    }
  }

private:
  CercVmeDevice &_cvd;
  unsigned _printLevel;
};

#endif
