#ifndef OnlMonitor_HH
#define OnlMonitor_HH

#include <iostream>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>

#include <iostream>
#include <fstream>
#include <vector>

#include "RcdTime.hh"
#include "RcdArena.hh"
#include "SubRunStatus.hh"
#include "SubRecord.hh"
#include "DaqSubRecordId.hh"

#include <TROOT.h>
#include "TFile.h"
#include "TH1.h"
#include "TH2.h"
#include "TProfile.h"
#include "TNtuple.h"
#include "TRandom.h"
#include "TMapFile.h"

class OnlMonitor : public DioConsumer<RcdArena> {

public:
  OnlMonitor() {

    // This magic number needs to be got from running a root session
    // interactively using rootn.exe and typing
    //   root [0] m = TMapFile::Create("dummy.map", "recreate", 10000000);
    //   root [1] m.Print()
    // which gives
    //   Memory mapped file:   dummy.map
    //   Title:
    //   Option:               CREATE
    //   Mapped Memory region: 0x41616000 - 0x41fa0000 (9.54 MB)
    //   Current breakval:     0x4161d000
    // where the magic number is the first in the "Mapped Memory region"
    // range. Don't forget to rm dummpy.map afterwards.

    TMapFile::SetMapAddress(0x41616000);
    _mfile = TMapFile::Create("hsimple.map","RECREATE", 1000000,
			      "Demo memory mapped file with histograms");
    _hpx=new TH1F("hpx","Record type",20,0,20);
    _hdt1=new TH1F("hdt1","Time",1000,0.0,1000.0);
    _hdt2=new TH1F("hdt2","Time difference",100,0.0,0.01);
    _mfile->Print();
  }

  ~OnlMonitor() {
    delete _hpx;
    delete _hdt1;
    delete _hdt2;

    delete _mfile;
  }

  virtual DioControl push(DioTypeHolder<RcdArena> &h) {
    RcdRecord *r(h.pointer());

    if(r==0) {
      total++;
      std::cerr << "OnlMonitor::push()  Null pointer to RcdArena" << std::endl;
      _hpx->Fill(19.0);

    } else {
      _hpx->Fill(r->recordType());

      if(r->recordType()==RcdHeader::startRun) {
	std::cout << "OnlMonitor::push()  Start of run" << std::endl;
	total=0;
	for(unsigned i(0);i<RcdHeader::endOfRecordTypeEnum;i++) count[i]=0;
      }

      total++;
      count[r->recordType()]++;

      if((total%1000)==0) std::cout 
	<< "OnlMonitor::push()  Total = " << total << std::endl;

      if(r->recordType()==RcdHeader::endRun ||
	 r->recordType()==RcdHeader::stopRun) {
	std::cout << endl << "OnlMonitor::push()  End of run:  Total = "
	     << total << std::endl;
	for(unsigned i(0);i<RcdHeader::endOfRecordTypeEnum;i++) {
	  std::cout << "  "
	       << RcdHeader::recordTypeName(static_cast<RcdHeader::RecordType>(i))
	       << ": record count[" << i << "] = " << count[i] << std::endl;
	}
      }

      // Histograms

      const std::vector<const RcdTime*> v(SubRecord<const RcdTime>::safePayloadVector(r));
      //for(unsigned i(0);i<v.size();i++) {
      //std::cout << "OnlMonitor::push()  Time " << i << " = "
      //	  << *(v[i]) << std::endl;
      //}

      if(v.size()>0) {
	if(r->recordType()==RcdHeader::startRun) {
	  _runStartTime=*(v[0]);
	}

	_hdt1->Fill((*(v[0])-_runStartTime).deltaTime());

	if(v.size()>1) {
	  RcdTimeDifference d(*(v[v.size()-1])-*(v[0])); // last - first
	  _hdt2->Fill(d.deltaTime());
	}
	
	if(v.size()==1) {
	  RcdTimeDifference d(RcdTime(true)-*(v[0])); // last - first
	  _hdt2->Fill(d.deltaTime());
	}
      }
    }

    h.destroy();

    _mfile->Update();

    DioControl c(h.control());
    if(c.status()!=DioControl::ok) std::cout
      << "OnlMonitor::push() status = " << c.status()
      << " = " << DioControl::statusName[c.status()] << std::endl;
    return c;
  }



protected:
  unsigned total;
  unsigned count[RcdHeader::endOfRecordTypeEnum];

  TMapFile *_mfile;
  TH1F *_hpx;
  TH1F *_hdt1;
  TH1F *_hdt2;

  RcdTime _runStartTime;
};

#endif
