#ifndef ReaderClient_HH
#define ReaderClient_HH
#include <sys/types.h>

#include <iostream>
#include <fstream>
#include <ctime>

#include "Record.hh"
#include "BufferStatus.hh"
#include "VmeInterrupt.hh"
#include "TrgConfigurationSubRecord.hh"
#include "Command.hh"
#include "RecordArray.hh"
#include "ShmSingleton.hh"
#include "ListenerSocket.hh"
#include "DuplexSocket.hh"


class ReaderClient {

public:
  ReaderClient() : _shmC(), _command(_shmC.payload()), 
		   _shmI(), _vmeInterrupt(_shmI.payload()),
		   _startUp(true), _level(0), pin(0) {

    if(!_vmeInterrupt->setConsumerPid()) {
      cerr << " Cannot register with ShmSingleton<VmeInterrupt>" 
	   << flush << endl;      
    }

    _runNumber=readRunNumber();
    cout << " Previous run number " << _runNumber << endl;
    _runNumber++;
  }

  ~ReaderClient() {
    delete pin;
  }

  void recv(bool err) const {
    if(_resetTrigger) {
      _vmeInterrupt->trigger(false);
      _vmeInterrupt->signalProducer();
    }
  }

  Record* send() {
    _resetTrigger=false;

    // Pass through a StartUp record on the first call
    if(_startUp) {
      _startUp=false;
      makeStartUpRecord();
      _record.print(cout);
      return &_record;
    }

    TrgConfigurationSubRecord *tcsr;

    switch(_level) {
    case 0:
      while(true) {
	if(_command->command()==Command::run) {
	  openRunType(_command->runType());
	  _record.read(*pin);
	  _record.print(cout);
	  
	  if(_record.recordType()!=RecordHeader::startOfRun) {
	    cerr << " Record type from file = " << _record.recordType()
		 << " does not match startOfRun = " << RecordHeader::startOfRun << endl;
	  }

	  if(_command->runType()!=0 && _command->runType()!=_record.runType()) {
	    cerr << " Run type from file = " << _record.runType()
		 << " does not match requested type = " << _command->runType() << endl;
	  }

	  if(!writeRunNumber(_runNumber)){
	    cerr << " run number write failed" << endl;
	  } else {
	    cout << " run number " << std::flush;
	    system("cat etc/runnumber.dat");
	  }
	  
	  _runType=_record.runType();
	  _recordNumberInRun=0;
	  _numberOfConfigurationsInRun=_record.numberOfConfigurations();
	  _configurationNumberInRun=0;
	  _spillNumberInRun=0;
	  _eventNumberInRun=0;
	  
	  makeStartOfRunRecord();
	  _record.print(cout);
	  
	  _level=1;
	  _command->command(Command::null);
	  _command->runActive(true);
	  return &_record;
	}
	
	if(_command->command()==Command::kill) {
	  makeShutDownRecord();
	  _record.print(cout);
	  
	  _level=0;
	  _command->command(Command::null);
	  _command->runActive(false);
	  return &_record;
	}

	if(_command->command()==Command::stop) {
	  _command->command(Command::null);
	  _command->runActive(false);
	}

	sleep(1);
      }
      break;


    case 1:
      if(_configurationNumberInRun>=_numberOfConfigurationsInRun) {

	delete pin;
	pin=0;

	makeEndOfRunRecord();
	_record.print(cout);
      
	_level=0;
	
	_command->command(Command::null);
	_command->runActive(false);
	return &_record;
      }


      _record.read(*pin);
      _record.print(cout);
	
      if(_record.recordType()!=RecordHeader::startOfConfiguration) {
	cerr << " Record type from file = " << _record.recordType()
	     << " does not match startOfConfiguration = " << RecordHeader::startOfConfiguration << endl;
      }
      if(_record.configurationNumber()!=_configurationNumberInRun) {
	cerr << " Configuration number from file = " << _record.configurationNumber()
	     << " does not match number seen = " << _configurationNumberInRun << endl;
      }
      
      _configurationType=_record.configurationType();
      _numberOfEventsInConfiguration=_record.numberOfEventsInConfiguration();
      _spillNumberInConfiguration=0;
      _eventNumberInConfiguration=0;
      _numberOfEventsInSpill=0xffffffff; // TO BE FILLED FROM ?
      _eventNumberInSpill=0; // MAY NOT BE START OF SPILL IMMEDIATELY
      
      makeStartOfConfigurationRecord();
      _record.print(cout);
	
      tcsr=(TrgConfigurationSubRecord*)_record.getSubRecord(SubRecord::trigger);
      _ignorTrigger=false;
      if(tcsr!=0) {
	_ignorTrigger=tcsr->ignorTrigger();
      }
      if(_ignorTrigger) cout << "Trigger being ignored" << endl;
      else              cout << "Trigger being looked at" << endl;
      
      _vmeInterrupt->trigger(false);
      
      _level=2;
      _command->command(Command::null);
      _command->runActive(true);
      return &_record;

      break;
    
    case 2:
      while(true) {
	if(_eventNumberInConfiguration>=_numberOfEventsInConfiguration) {

	  makeEndOfConfigurationRecord();
	  _record.print(cout);
	  
	  _level=1;
	  _command->command(Command::null);
	  _command->runActive(true);
	  return &_record;
	}

	if(_command->command()==Command::stop) {

	  makeStopConfigurationRecord();
	  _record.print(cout);
	  
	  _level=1;
	  _command->command(Command::null);
	  _command->runActive(true);
	  return &_record;
	}

	if(_ignorTrigger) {
	  makeEventRecord();
	  //_record.print(cout);
	  
	  //	  _resetTrigger=true;
	  _command->runActive(true);
	  return &_record;
	}

	if(_vmeInterrupt->spill()) {
	  _eventNumberInSpill=0;

	  makeStartOfSpillRecord();
	  _record.print(cout);

	  _level=3;
	  _command->runActive(true);
	  return &_record;
	}

	if(_vmeInterrupt->trigger()) {
	  makeEventRecord();
	  //_record.print(cout);

	  _resetTrigger=true;
	  _command->runActive(true);
	  return &_record;

	} else {
	  _vmeInterrupt->enableConsumerSignal();
	  sleep(1);
	  _vmeInterrupt->disableConsumerSignal();
	}
      }
      break;

    case 3:
      while(true) {
	if(_eventNumberInConfiguration>=_numberOfEventsInConfiguration ||
	   _eventNumberInSpill>=_numberOfEventsInSpill) {

	  makeEndOfSpillRecord();
	  _record.print(cout);

	  _level=2;
	  _command->runActive(true);
	  return &_record;
	}

	if(_command->command()==Command::stop) {

	  makeStopSpillRecord();
	  _record.print(cout);

	  _level=2;
	  //_command->command(Command::null); ?? NO! FORCE DOWN TO LEVEL 1
	  _command->runActive(true);
	  return &_record;
	}

	if(_ignorTrigger) {
	  makeEventRecord();
	  //_record.print(cout);
	  
	  //	  _resetTrigger=true;
	  _command->runActive(true);
	  return &_record;
	}

	if(!_vmeInterrupt->spill()) {
	  makeEndOfSpillRecord();
	  _record.print(cout);

	  _level=2;
	  _command->runActive(true);
	  return &_record;
	}

	if(_vmeInterrupt->trigger()) {
	  makeEventRecord();
	  //_record.print(cout);
	  
	  _resetTrigger=true;
	  _command->runActive(true);
	  return &_record;

	} else {
	  _vmeInterrupt->enableConsumerSignal();
	  sleep(1);
	  _vmeInterrupt->disableConsumerSignal();
	}
      }
      break;

    default:
      break;
    };
  }

  void makeStartUpRecord() {
    _record.recordType(RecordHeader::startUp);
    _record.recordTime(time(0));
    _record.recordNumber(0);
    _record.zeroOthers();
    _record.deleteSubRecords();
  }

  void makeStartOfRunRecord() {
    _record.recordType(RecordHeader::startOfRun);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.runNumber(_runNumber);
    _record.runType(_runType);
    _record.numberOfConfigurations(_numberOfConfigurationsInRun);
  }

  void makeStartOfConfigurationRecord() {
    _record.recordType(RecordHeader::startOfConfiguration);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);	
    _record.configurationNumber(_configurationNumberInRun);
    _record.configurationType(_configurationType);
    _record.numberOfEventsInConfiguration(_numberOfEventsInConfiguration);
  }

  void makeStartOfSpillRecord() {
    _record.recordType(RecordHeader::startOfSpill);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.spillNumber(_spillNumberInRun);
    _record.spillNumberInConfiguration(_spillNumberInConfiguration);
    _record.numberOfEventsInSpill(_numberOfEventsInSpill);
    _record.deleteSubRecords();
  }

  void makeEventRecord() {
    _record.recordType(RecordHeader::event);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.eventNumber(_eventNumberInRun++);
    _record.eventNumberInConfiguration(_eventNumberInConfiguration++);
    _record.eventNumberInSpill(_eventNumberInSpill++);
    _record.deleteSubRecords();
  }

  void makeStopSpillRecord() {
    _record.recordType(RecordHeader::stopSpill);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.spillNumber(_spillNumberInRun++);
    _record.spillNumberInConfiguration(_spillNumberInConfiguration++);
    _record.numberOfEventsInSpill(_eventNumberInSpill);
    _record.deleteSubRecords();
  }

  void makeEndOfSpillRecord() {
    _record.recordType(RecordHeader::endOfSpill);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.spillNumber(_spillNumberInRun++);
    _record.spillNumberInConfiguration(_spillNumberInConfiguration++);
    _record.numberOfEventsInSpill(_eventNumberInSpill);
    _record.deleteSubRecords();
  }

  void makeStopConfigurationRecord() {
    _record.recordType(RecordHeader::stopConfiguration);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.configurationNumber(_configurationNumberInRun++);
    _record.configurationType(_configurationType);
    _record.numberOfEventsInConfiguration(_eventNumberInConfiguration);
    _record.deleteSubRecords();
  }

  void makeEndOfConfigurationRecord() {
    _record.recordType(RecordHeader::endOfConfiguration);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.configurationNumber(_configurationNumberInRun++);
    _record.configurationType(_configurationType);
    _record.numberOfEventsInConfiguration(_eventNumberInConfiguration);
    _record.deleteSubRecords();
  }

  void makeEndOfRunRecord() {
    _record.recordType(RecordHeader::endOfRun);
    _record.recordTime(time(0));
    _record.recordNumber(_recordNumberInRun++);
    _record.runNumber(_runNumber++);
    _record.runType(_runType);
    _record.numberOfConfigurations(_configurationNumberInRun);
    _record.deleteSubRecords();
  }

  void makeShutDownRecord() {
    _record.recordType(RecordHeader::shutDown);
    _record.recordTime(time(0));
    _record.recordNumber(0);
    _record.zeroOthers();
    _record.deleteSubRecords();
  }

private:
  unsigned readRunNumber() {
    ifstream fin("etc/runnumber.dat",ios::in);
    if(!fin) return 0;
    unsigned n(0);
    fin >> n;
    return n;
  }

  bool writeRunNumber(unsigned n) {
    ofstream fout("etc/runnumber.dat",ios::out);
    if(!fout) return false;
    fout << n << endl;
    return true;
  }

  bool openRunType(unsigned nType=0) {
    if(pin!=0) return false;

    char file[128];
    sprintf(file,"typ/typ%06.6u.dat",nType);
    pin=new ifstream(file,ios::in);

    if(pin==0 || !(*pin)) {
      cerr << " Error opening file " << file << endl;
      return true;

    } else {
      cout << " New run: type " << nType
	   << " read from file " << file << endl;
    }
    return true;
  }


  ShmSingleton<Command> _shmC;
  Command *_command;
  ShmSingleton<VmeInterrupt> _shmI;
  VmeInterrupt *_vmeInterrupt;
  bool _startUp;

  unsigned _level;
  bool _resetTrigger,_ignorTrigger;

  unsigned _recordNumberInRun;
  unsigned _runNumber,_runType,_numberOfConfigurationsInRun;
  unsigned _configurationNumberInRun,_configurationType,_numberOfEventsInConfiguration;
  unsigned _spillNumberInRun,_spillNumberInConfiguration,_numberOfEventsInSpill;
  unsigned _eventNumberInRun,_eventNumberInConfiguration,_eventNumberInSpill;

  ifstream *pin;
  RecordArray<MAXRECORDARRAY> _record;
};

#endif
