#ifndef TrgReadout_HH
#define TrgReadout_HH

#include <iostream>
#include <fstream>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "DaqBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubInserter.hh"
#include "SubAccessor.hh"

#include "CrcVmeDevice.hh"
#include "TrgReadoutConfigurationData.hh"


class TrgReadout : public RcdUserRW {

public:
  TrgReadout(unsigned b, unsigned char c, unsigned char s) :
    _location(c,s,CrcLocation::beTrg,0),
    _crateNumber(c), _slotNumber(s),
    _maximumNumberOfBufferedEvents(2000),
    _minimumTriggerSeparation(0.0002),
    _device(0), _busAdapter(0),
    _addressTableReader("online/hal/CrcVmeAddress.hal"),
    _addressTable("CRC VME Address Table",_addressTableReader) {
    
    // Catch exceptions locally here in case PCI card not installed
    try {
      _busAdapter=new DaqBusAdapter(b);
    } catch ( std::exception e ) {
      _busAdapter=0;
    }
    
    // Try to make the device object
    if(_busAdapter!=0 && _slotNumber>=1 && _slotNumber<=21) {
      _device=new CrcVmeDevice(_addressTable,*_busAdapter,_slotNumber);
      if(_device->alive()) {
	std::cout << "TrgReadout::ctor()  PCI card " << b
		  << " slot " << std::setw(2)
		  << (unsigned)_slotNumber << " found alive" << std::endl;
      } else {
	delete _device;
	_device=0;
      }
    }
    
    // Allow it to be null; no trigger for separate crates!
    //assert(_device!=0);

    // Define the poll objects
    _poll.location(_location);
    _pollSpill.location(_location);

    // Define the label for the multi-timer
    _tid.halfWord(1,SubHeader::trg);
    _tid.byte(2,_location.crateNumber());
  }
  
  virtual ~TrgReadout() {
    /* Causes a core dump???
    if(_busAdapter!=0) delete _busAdapter;
    if(_device!=0) delete _device;
    */
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "TrgReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    // Initialise the multi-timer
    SubInserter inserter(r);
    _daqMultiTimer=inserter.insert<DaqMultiTimer>(true);
    _daqMultiTimer->timerId(_tid.word());
    inserter.extend(16*sizeof(UtlTime));

    // Catch exceptions from HAL code
    try {

      // Check record type
      switch (r.recordType()) {

      case RcdHeader::startUp: {

	// Reset the BE-Trg
	if(_device!=0) assert(_device->beTrgSoftReset());

	// Set this once only
	if(_device!=0) assert(_device->beTrgForceBusy(false));

	break;
      }

      case RcdHeader::runStart: {

        // Access the DaqRunStart to get print level
        SubAccessor accessor(r);
        std::vector<const DaqRunStart*> v(accessor.access<DaqRunStart>());
        assert(v.size()==1);
	_printLevel=v[0]->runType().printLevel();

	_daqRun=*(v[0]);
	if(doPrint(r.recordType(),1)) _daqRun.print(std::cout," ") << std::endl;

	break;
      }

      case RcdHeader::runEnd:
      case RcdHeader::runStop: {

	break;
      }

	// Configuration start is used to set up system
      case RcdHeader::configurationStart: {

	// Get the configuration information needed
	SubAccessor accessor(r);

	std::vector<const DaqConfigurationStart*>
	  dc(accessor.access<DaqConfigurationStart>());
	assert(dc.size()==1);
	_daqConfiguration=*(dc[0]);
	if(doPrint(r.recordType(),1)) _daqConfiguration.print(std::cout," ") << std::endl;
      
	std::vector<const TrgReadoutConfigurationData*>
	  bc(accessor.access<TrgReadoutConfigurationData>());

	if(bc.size()>1) {
	  for(unsigned i(0);i<bc.size();i++) bc[i]->print(std::cout," ERROR ");
	  assert(false);
	}

	if(bc.size()==1) _config=*(bc[0]);
	if(doPrint(r.recordType(),1)) _config.print(std::cout," ") << std::endl;

	break;
      }

      case RcdHeader::configurationEnd:
      case RcdHeader::configurationStop: {

	break;
      }

      case RcdHeader::acquisitionStart: {
	_nEvents=0;

	if(_config.enable()) {
	  _spillStartSeen=false;

	  if(_device!=0) {

	    // Loops to show input status every second
	    //assert(_device->checkSpillSignal(40));	

	    // Zero counters and flush fifos
	    //assert(_device->beTrgForceBusy(false));
	    assert(_device->beTrgSoftTrigger());
	    assert(_device->clearBeTrgFifos());

	  }
	}

	// Read out trigger data after clearing
	assert(readTriggerData(r,true));

	_inSpill=false;
	_numberOfBufferedEvents=0;

	break;
      }

      case RcdHeader::acquisitionEnd:
      case RcdHeader::acquisitionStop: {
	  
	// Read out trigger data before clearing
	assert(readTriggerData(r,true));

	if(_device!=0 && _config.enable()) {

	  /*
	  // Zero counters, force busy, etc
	    //assert(_device->beTrgForceBusy(false));
	    assert(_device->beTrgSoftTrigger());
	    assert(_device->clearBeTrgFifos());
	  }
	  */
	}

	break;
      }

      case RcdHeader::spillStart: {
	_inSpill=true;

	if(_config.enable()) {
	  _spillStartSeen=true;
	  _spillStartTime=r.recordTime();

	  if(_config.beTrgSoftSpill()) {
	    if(doPrint(r.recordType(),1)) std::cout << " Be-Trg soft spill" << std::endl;
	    if(_device!=0) assert(_device->beTrgSoftSpill(true));

	  } else {
	    _poll.data()->maximumTime(_daqConfiguration.maximumTimeOfSpill());
	    if(doPrint(r.recordType(),1)) _poll.print(std::cout," ") << std::endl;
	    if(_device!=0) assert(_device->pollBeTrgSpillStart(*(_poll.data()),_config.spillInvert()));
	    if(doPrint(r.recordType(),1)) _poll.print(std::cout," ") << std::endl;
	    
	    SubInserter inserter(r);
	    inserter.insert< CrcLocationData<CrcBeTrgPollData> >(_poll);
	  }
	}

	break;
      }

      case RcdHeader::spillEnd: {
	_inSpill=false;

	if(_config.enable()) {
	  if(_device!=0 && _config.beTrgSoftSpill()) assert(_device->beTrgSoftSpill(false));
	}
	break;
      }

	/*
      case RcdHeader::spill: {
	if(_config.enable()) {
	  *(_pollSpill.data())=TrgSpillPollData();
	  
	  if(_config.beTrgSoftSpill()) {
	    if(_printLevel>8) std::cout << "BeTrg soft spill on" << std::endl;
 
	    if(_device!=0) assert(_device->beTrgSoftSpill(true));
	    _pollSpill.data()->updateStartSpillTime();
	  }

	  if(_config.beTrgSoftTrigger()) {
	    if(_printLevel>8) std::cout << "BeTrg soft triggers" << _daqConfiguration.maximumNumberOfEventsInAcquisition() << std::endl;
	    if(_device!=0) assert(_device->beTrgSoftTriggers(_daqConfiguration.maximumNumberOfEventsInAcquisition()));

	  } else {
	    _pollSpill.data()->maximumSpillTime(_daqConfiguration.maximumTimeOfSpill());
	    _pollSpill.data()->maximumPollTime(_daqConfiguration.maximumTimeOfSpill());
	    _pollSpill.data()->maximumNumberOfEvents(_daqConfiguration.maximumNumberOfEventsInAcquisition());

	    if(_device!=0) assert(_device->trgSpillPoll(*(_pollSpill.data())));
	    if(doPrint(r.recordType(),1)) _pollSpill.print(std::cout," ");
	  
	    SubInserter inserter(r);
	    inserter.insert< CrcLocationData<TrgSpillPollData> >(_pollSpill);
	  }

	  if(_config.beTrgSoftSpill()) {
	    if(_printLevel>8) std::cout << "Be-Trg soft spill off" << std::endl;

	    if(_device!=0) assert(_device->beTrgSoftSpill(false));
	    _pollSpill.data()->updateEndSpillTime();
	  }

	  assert(takeTrigger(r));
	  assert(readTriggerData(r));
	}

	break;
      }
	*/

      case RcdHeader::trigger: {
	if(_config.enable()) {

	  // Get control data
	  SubAccessor accessor(r);

	  std::vector<const DaqEvent*>
	    de(accessor.access<DaqEvent>());
	  assert(de.size()==1);
	  if(doPrint(r.recordType(),1)) de[0]->print(std::cout," ") << std::endl;

	  // Take a trigger and (occasionally) read out trigger data
	  _nEvents++;
	  _numberOfBufferedEvents++;

	  assert(takeTrigger(r));
	  assert(readTriggerData(r));

	  // Put control data into record
	  SubInserter inserter(r);

	  DaqTrigger *dt(inserter.insert<DaqTrigger>(true));

	  dt->triggerNumberInRun(          de[0]->eventNumberInRun());
	  dt->triggerNumberInConfiguration(de[0]->eventNumberInConfiguration());
	  dt->triggerNumberInAcquisition(  de[0]->eventNumberInAcquisition());

	  // Set flags; do not change buffer full flag if already set
	  if(_numberOfBufferedEvents>=_maximumNumberOfBufferedEvents) dt->crcBufferFull(true); // CRC limit
	  dt->inSpill(_inSpill);

	  if(doPrint(r.recordType(),1)) dt->print(std::cout," ") << std::endl;
	}

	break;
      }

      case RcdHeader::triggerBurst: {
	if(_config.enable()) {

	  // Get control data
	  SubAccessor accessor(r);

	  std::vector<const DaqBurstStart*>
	    dbs(accessor.access<DaqBurstStart>());
	  assert(dbs.size()==1);

	  if(doPrint(r.recordType(),1)) dbs[0]->print(std::cout," ") << std::endl;

	  // Take one trigger
	  _numberOfBufferedEvents++;
	  assert(takeTrigger(r));
	  UtlTime lastTrigger(true);

	  // Find the maximum number of extra triggers to take
	  unsigned maximumExtras(dbs[0]->maximumNumberOfExtraTriggersInBurst());
	  if(_config.readPeriod()>0 && _config.readPeriod()<maximumExtras)
	    maximumExtras=_config.readPeriod()-1;

	  // Loop over extra triggers
	  unsigned nExtras(0);
	  //CrcBeTrgEventData x;
	  UtlTimeDifference timeOfBurst(UtlTime(true)-r.recordTime());




#ifdef CHECK_THE_TIMING
	  DaqMultiTimer tempTimer[500];
	  tempTimer[0].addTimer();
#endif




	  for(unsigned i(0); _inSpill && i<maximumExtras &&
		timeOfBurst<dbs[0]->maximumTimeOfBurst() &&
		_numberOfBufferedEvents<_maximumNumberOfBufferedEvents; i++) {
	    
#ifdef CHECK_THE_TIMING
	  	tempTimer[0].addTimer();
#endif

#ifdef SPILL_INPUT

	    	// Check spill setting while waiting for spill
		//while(_inSpill && ((UtlTime(true)-lastTrigger).deltaTime()<_minimumTriggerSeparation)) {
		//if(timeOfBurst.deltaTime()>0.1) {

		/*
		  assert(_device->readBeTrgEventData(x,false,true));
		  if(doPrint(r.recordType(),1)) x.print(std::cout," ") << std::endl;
		  
		  if(_config.spillInvert()) {
		    _inSpill=(x.inputStatus()&(1<<SPILL_INPUT))==0;
		  } else {
		    _inSpill=(x.inputStatus()&(1<<SPILL_INPUT))!=0;
		  }
		*/

		UtlPack x(_device->readBeTrgInputStatus());
		if(_config.spillInvert()) {
		  _inSpill=!x.bit(SPILL_INPUT);
		} else {
		  _inSpill= x.bit(SPILL_INPUT);
		}

	        //}
		
#endif
	    
#ifdef CHECK_THE_TIMING
	  tempTimer[0].addTimer();
#endif

	    // If still spilling, then take a trigger
	    if(_inSpill) {
	      nExtras++;
	      _numberOfBufferedEvents++;
	      
	      assert(takeTrigger(r));
	      lastTrigger.update();
	    }
	    
#ifdef CHECK_THE_TIMING
	  tempTimer[0].addTimer();
#endif

	    // Update time taken
	    timeOfBurst=UtlTime(true)-r.recordTime();
	  }
	  

#ifdef CHECK_THE_TIMING
	  tempTimer[0].print(std::cout," TEMPTIMER ") << std::endl;
#endif


	  // Now force a read of the trigger data
	  _nEvents+=1+nExtras;
	  assert(readTriggerData(r,true));

	  // Set up control data
	  DaqBurstEnd *dbe(inserter.insert<DaqBurstEnd>(true));

	  dbe->burstNumberInRun(          dbs[0]->burstNumberInRun());
	  dbe->burstNumberInConfiguration(dbs[0]->burstNumberInConfiguration());
	  dbe->burstNumberInAcquisition(  dbs[0]->burstNumberInAcquisition());

	  dbe->lastTriggerNumberInRun(          dbs[0]->firstTriggerNumberInRun()          +nExtras);
	  dbe->lastTriggerNumberInConfiguration(dbs[0]->firstTriggerNumberInConfiguration()+nExtras);
	  dbe->lastTriggerNumberInAcquisition(  dbs[0]->firstTriggerNumberInAcquisition()  +nExtras);

	  dbe->actualNumberOfExtraTriggersInBurst(nExtras);
	  dbe->actualTimeOfBurst(timeOfBurst);

	  // Set flags; do not change buffer full flag if already set
	  if(_numberOfBufferedEvents>=_maximumNumberOfBufferedEvents) dbe->crcBufferFull(true); // CRC limit
	  dbe->inSpill(_inSpill);

	  if(doPrint(r.recordType(),1)) dbe->print(std::cout," ") << std::endl;
	}
	break;
      }

      case RcdHeader::event: {
	if(_config.enable()) {
	  //assert(takeTrigger(r));
	  //assert(readTriggerData(r));
	  /*
	  unsigned char array[16*1024];
	  assert(_device->readVlinkEventData(*((CrcVlinkEventData*)array),true));
	  ((CrcVlinkEventData*)array)->print(std::cout,"TEST ",true);
	  */
	  assert(_numberOfBufferedEvents>0);
	  _numberOfBufferedEvents--;
	}

	break;
      }

      default: {

	break;
      }
      };

      _daqMultiTimer->addTimer(); // Last
      if(doPrint(r.recordType(),1)) _daqMultiTimer->print(std::cout," ") << std::endl;

      return true;

    } catch ( HAL::HardwareAccessException& e ) {
      std::cout << "*** Exception occurred : " << e.what() << std::endl;
      
    } catch ( std::exception e ) {
      std::cout << "*** Unknown exception occurred" << std::endl;
    }
    return false;
  }

  ////////////////////////////////////////////////////////////////////////////////////////

  // This is now done for all CRCs in CrcReadout
  /*
  bool readRunData(RcdRecord &r) {
    SubInserter inserter(r);

    CrcLocationData<CrcBeTrgRunData> *t(inserter.insert< CrcLocationData<CrcBeTrgRunData> >());
    t->location(_location);
    if(_device!=0) assert(_device->readBeTrgRunData(*t->data()));
    if(doPrint(r.recordType(),1)) t->print(std::cout," ") << std::endl;
    
    return true;
  }

  bool readConfigurationData(RcdRecord &r) {
    SubInserter inserter(r);

    CrcLocationData<CrcBeTrgConfigurationData> *b(inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >());
    b->location(_location);
    if(_device!=0) assert(_device->readBeTrgConfigurationData(*b->data()));
    if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;
    
    return true;
  }
  */

  bool takeTrigger(RcdRecord &r) {
    SubInserter inserter(r);

    if(_config.clearBeTrgTrigger()) {
      if(doPrint(r.recordType(),1)) std::cout << " Be-Trg clear trigger" << std::endl;
      if(_device!=0) assert(_device->clearBeTrgTrigger());
    }

    //_daqMultiTimer->addTimer(); // 1

    if(_config.beTrgSoftTrigger()) {
      if(doPrint(r.recordType(),1)) std::cout << " Be-Trg soft trigger" << std::endl;
      if(_device!=0) assert(_device->beTrgSoftTrigger());

    } else {

      _poll.data()->maximumTime(_daqConfiguration.maximumTimeOfEvent());

      /* Do we need to cut on other time limits?
	 UtlTimeDifference remainingRunTime(_daqRun.maximumTimeOfRun()-(r.recordTime()-_runStartTime));

      UtlTimeDifference remainingTime(_daqConfiguration.maximumTimeOfSpill()-(r.recordTime()-_spillStartTime));
      if(remainingTime<_daqConfiguration.maximumTimeOfEvent()) _poll.data()->maximumTime(remainingTime);
      */

      //if(doPrint(r.recordType(),1)) _poll.print(std::cout," ") << std::endl;
      if(_device!=0) assert(_device->pollBeTrgTrigger(*(_poll.data())));
      if(doPrint(r.recordType(),1)) _poll.print(std::cout," ") << std::endl;

      inserter.insert< CrcLocationData<CrcBeTrgPollData> >(_poll);
    }

    //_daqMultiTimer->addTimer(); // 2
    
    return true;
  }

  bool readTriggerData(RcdRecord &r, bool forceRead=false, bool forceReadc=false) {
    if(_device!=0) {

      // Do full readout
      if(forceRead || (_config.readPeriod()>0 && (_nEvents%_config.readPeriod())==0)) {
	SubInserter inserter(r);
	CrcLocationData<CrcBeTrgEventData>
	  *p(inserter.insert< CrcLocationData<CrcBeTrgEventData> >());
	p->location(_location);
	
	//assert(_device->readBeTrgEventData(*p->data()));
	_device->readBeTrgEventData(*p->data(),_config.beTrgSquirt()); // Not assert because of FIFO
	inserter.extend(p->data()->numberOfFifoWords()*4); // For Be-Trg FIFO
	if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;

#ifdef SPILL_INPUT
	if(_config.spillInvert()) {
          _inSpill=(p->data()->inputStatus()&(1<<SPILL_INPUT))==0;
        } else {
	  _inSpill=(p->data()->inputStatus()&(1<<SPILL_INPUT))!=0;
	}
#endif
	
      // Do just trigger counter and spill indicator
      } else if(forceReadc || (_config.readcPeriod()>0 && (_nEvents%_config.readcPeriod())==0)) {
	SubInserter inserter(r);
	CrcLocationData<CrcBeTrgEventData>
	  *p(inserter.insert< CrcLocationData<CrcBeTrgEventData> >());
	p->location(_location);
	
	assert(_device->readBeTrgEventData(*p->data(),false,true));
	if(doPrint(r.recordType(),1)) p->print(std::cout," ") << std::endl;

#ifdef SPILL_INPUT
	if(_config.spillInvert()) {
          _inSpill=(p->data()->inputStatus()&(1<<SPILL_INPUT))==0;
        } else {
          _inSpill=(p->data()->inputStatus()&(1<<SPILL_INPUT))!=0;
        }
#endif

      }
    }

    //_daqMultiTimer->addTimer(); // 3
    
    return true;
  }
  

protected:
  const CrcLocation _location;
  const unsigned char _crateNumber;
  const unsigned char _slotNumber;

  unsigned _numberOfBufferedEvents;
  const unsigned _maximumNumberOfBufferedEvents;
  const double _minimumTriggerSeparation;

  CrcVmeDevice *_device;
  DaqBusAdapter *_busAdapter;
  HAL::VMEAddressTableASCIIReader _addressTableReader;
  HAL::VMEAddressTable _addressTable;

  DaqRunStart _daqRun;
  DaqConfigurationStart _daqConfiguration;
  TrgReadoutConfigurationData _config;

  bool _spillStartSeen;
  UtlTime _spillStartTime;

  UtlPack _tid;
  DaqMultiTimer *_daqMultiTimer;

  bool _inSpill;

  CrcLocationData<CrcBeTrgPollData> _poll;
  CrcLocationData<TrgSpillPollData> _pollSpill;

  unsigned _nEvents;
};

#endif
