#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//#include "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

#include "UtlTime.hh"
#include "UtlArguments.hh"
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "DaqRunStart.hh"

#include "SubAccessor.hh"

#include "CrcLocationData.hh"
#include "EmcReadout.hh"
#include "CrcReadoutConfigurationData.hh"
#include "OnlCounter.hh"
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

using std::cin;
using std::exception;
using namespace std;

//efine CERCVMEADDRESSTABLE "CercAddressMap.dat"
#define CERCVMEADDRESSTABLE "online/hal/CrcVmeAddress.hal"
#define SEQUENCE_SETTINGS "Sequences.dat"


bool continueJob=true;
bool continueRun=true;
bool continueConfiguration=true;

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;
  continueJob=false;
}

int main(int argc, const char **argv) {
  try{

    UtlArguments argh(argc,argv);
    //argh.print(cout);

    const bool doCheck(argh.option('c',"Run data check"));
    const bool doHistograms(argh.option('s',"Display histograms"));
    const bool doGraphics(argh.option('g',"Display graphics"));
    const bool useWriteDmy(argh.option('w',"Dummy output file"));
    
    const unsigned printLevel(argh.optionArgument('p',5,"Print level"));
    const unsigned nEvents(argh.optionArgument('n',10,"Number of events between updates"));
    
    std::string logFile(argh.optionArgument('l',"log/msg.log","RS232 message log file"));
    
    if(argh.help()) return 0;
    
    if(doCheck) cout << "Data check selected" << endl;
    else           cout << "Data check not selected" << endl;
    if(doHistograms) cout << "Histograms display selected" << endl;
    else           cout << "Histograms display not selected" << endl;
    if(doGraphics) cout << "Graphics display selected" << endl;
    else           cout << "Graphics display not selected" << endl;
    if(useWriteDmy) cout << "Dummy output selected" << endl;
    else              cout << "File output selected" << endl;
    
    cout << "Print level set to " << printLevel << endl;
    cout << "Number of events set to " << nEvents << endl;
    
    // Define the CERC locations
    unsigned char theCrate(0xec);

    //const unsigned nSlots(4);
    //unsigned theSlots[nSlots]={12,5,7,17};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader( CERCVMEADDRESSTABLE );
    VMEAddressTable addressTable( "Test address table", addressTableReader );
    
    unsigned nSlots(0);
    unsigned theSlots[22];
    
    // Find CRCs automatically
    CrcVmeDevice *dev[22];
    for(unsigned i(4);i<=21;i++) {
      unsigned j(i);
      if(i==4) j=12;
      if(i!=12) { // Special trigger slot
	dev[nSlots]=new CrcVmeDevice(addressTable,busAdapter,j);
	if(dev[nSlots]->alive()) {
	  theSlots[nSlots]=j;
	  cout << "Slot " << j << " found alive" << endl;
	  nSlots++;
	} else {
	  delete dev[nSlots];
	}
      }
    }

    assert(nSlots>0);

    // Register CERCs
    EmcReadout oe(theCrate);
    oe.printLevel(printLevel);
    for(unsigned i(0);i<nSlots;i++) oe.device(dev[i],i==0);
    
    HstBase *hn(0);
    //if(doHistograms) hn=new HstNoise(theSlots[0]);
    //if(doHistograms) hn=new HstChipNoise(true);
    if(doHistograms) hn=new HstChanNoise(true);

    RcdArena arena;

    // Send Startup record
    OnlCounter od;
    od.readout(arena,RcdHeader::startUp);
    oe.record(arena);

    // Send Slow controls record
    od.readout(arena,RcdHeader::slowControl);
    SubInserter inserter(arena);

    for(unsigned i(0);i<nSlots;i++) {
      CrcLocationData<CrcAdm1025SlowControlsData>
	*b(inserter.insert< CrcLocationData<CrcAdm1025SlowControlsData> >());
      b->crateNumber(theCrate);
      b->slotNumber(theSlots[i]);
      b->crcComponent(CrcLocation::vme);
      b->label(1);

      *b->data()=CrcAdm1025SlowControlsData();
      b->print(std::cout) << std::endl;

      CrcLocationData<CrcLm82SlowControlsData>
	*v(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
      v->crateNumber(theCrate);
      v->slotNumber(theSlots[i]);
      v->crcComponent(CrcLocation::vmeLm82);
      v->label(1);
      
      *v->data()=CrcLm82SlowControlsData();
      v->print(std::cout) << std::endl;

      CrcLocationData<CrcLm82SlowControlsData>
	*l(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
      l->crateNumber(theCrate);
      l->slotNumber(theSlots[i]);
      l->crcComponent(CrcLocation::be);
      l->label(1);
      
      *l->data()=CrcLm82SlowControlsData();
      l->print(std::cout) << std::endl;

      for(unsigned f(0);f<8;f++) {
	CrcLocationData<CrcLm82SlowControlsData>
	  *l(inserter.insert< CrcLocationData<CrcLm82SlowControlsData> >());
	l->crateNumber(theCrate);
	l->slotNumber(theSlots[i]);
	l->crcComponent((CrcLocation::CrcComponent)f);
	l->label(1);
	
	*l->data()=CrcLm82SlowControlsData();
	l->print(std::cout) << std::endl;
      }
    }
  
    oe.record(arena);

    // Create the poll object in advance
    CrcLocationData<CrcBeTrgPollData> poll;
    poll.crateNumber(theCrate);
    poll.slotNumber(theSlots[0]);
    poll.crcComponent(CrcLocation::beTrg);
    poll.label(0);
    poll.print(std::cout) << std::endl;

    signal(SIGINT,signalHandler);

    while(continueJob) {
      // Send Run start
      od.readout(arena,RcdHeader::runEnd);
    
      // Readout and write
      oe.record(arena);
      sleep(5);
    }    

    if(hn!=0) hn->record(arena);
    if(hn!=0) delete hn;

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }
}
