#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

// hal
#include "VMEAddressTable.hh"
#include "VMEAddressTableASCIIReader.hh"
#include "SBS620x86LinuxBusAdapter.hh"
//nclude "VMEDummyBusAdapter.hh"
#include "HardwareAccessException.hh"

// dual/inc/utl
#include "UtlTime.hh"
#include "UtlArguments.hh"

// dual/inc/rcd
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "RcdMultiUserRW.hh"
#include "RcdMultiUserRO.hh"

// dual/inc/sub
#include "SubAccessor.hh"

// dual/inc/daq
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqSpillStart.hh"
#include "DaqWriter.hh"

// dual/inc/crc
#include "CrcLocationData.hh"
#include "CrcReadoutConfigurationData.hh"

// dual/inc/hst
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

// online/inc/daq
#include "DaqCounter.hh"

// online/inc/emc
#include "EmcReadout.hh"

// online/inc/dvr
#include "DvrEmcIntDac.hh"
#include "DvrCrcSlw.hh"

#include "DaqSlwControl.hh"
#include "ShmObject.hh"

using namespace std;

unsigned continueFlag(4);

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;

  continueFlag=0;
  if(signal==SIGUSR1) continueFlag=1;
  if(signal==SIGUSR2) continueFlag=2;
}

void initialiseRecord(RcdRecord &r, RcdHeader::RecordType t) {
  r.updateRecordTime();
  r.recordType(t);
  r.deleteData();
}

int main(int argc, const char **argv) {
  unsigned eTime(CALICE_DAQ_TIME);
  cout << argv[0] << " compiled at " << ctime((const time_t*)&eTime) << endl;

  UtlArguments argh(argc,argv);

  const bool useWriteDmy(argh.option('w',"Dummy output file"));
  const bool useWriteAsc(argh.option('a',"Ascii output file"));
  const bool doHistograms(argh.option('s',"Display histograms"));
  
  const unsigned nRun(argh.optionArgument('r',1,"Number of runs"));
  const unsigned printLevel(argh.optionArgument('p',0,"Print level"));
  
  if(argh.help()) return 0;
  
  if(doHistograms)  cout << "Histograms display selected" << endl;
  else              cout << "Histograms display not selected" << endl;
  if(useWriteDmy)   cout << "Dummy output selected" << endl;
  else {
    if(useWriteAsc) cout << "Ascii output selected" << endl;
    else            cout << "Binary output selected" << endl;
  }
  
  cout << "Print level set to " << printLevel << endl;
  cout << "Number of runs set to " << nRun<< endl;
  cout << endl;

  // Stop slow data run
  ShmObject<DaqSlwControl> shmSlwControl(45454547);
  DaqSlwControl *pDsc(shmSlwControl.payload());
  assert(pDsc!=0);
  pDsc->command(DaqSlwControl::wait);

  try{

    // Define the CERC locations
    unsigned char theCrate(0xec);

    //const unsigned nSlots(4);
    //unsigned theSlots[nSlots]={12,5,7,17};
    
    SBS620x86LinuxBusAdapter busAdapter(0);
    VMEAddressTableASCIIReader addressTableReader("online/hal/CrcVmeAddress.hal");
    VMEAddressTable addressTable("CRC VME Address Table",addressTableReader);
    
    unsigned nSlots(0);
    unsigned theSlots[22];
    
    // Find CRCs automatically
    CrcVmeDevice *dev[22];
    for(unsigned i(4);i<=21;i++) {
      unsigned j(i);
      if(i==4) j=12;
      if(i!=12) { // Special trigger slot
	dev[nSlots]=new CrcVmeDevice(addressTable,busAdapter,j);
	//if(dev[nSlots]->alive() && (j==7 || j==12)) {
	if(dev[nSlots]->alive()) {
	  theSlots[nSlots]=j;
	  cout << "Slot " << j << " found alive" << endl;
	  nSlots++;
	} else {
	  delete dev[nSlots];
	}
      }
    }

    assert(nSlots>0);

    // Define lists of user modules
    //vector<RcdUserBase*> vrub;
    RcdMultiUserRW vRrw;
    vRrw.printLevel(printLevel);
    RcdMultiUserRO vrub;
    vrub.printLevel(printLevel);

    // Add DAQ counter module
    DaqCounter dc(!useWriteDmy);
    vRrw.addUser(dc);

    // Add driver module
    DvrEmcIntDac deid(theCrate);
    vRrw.addUser(deid);

    // Add CRC readout module and register CRCs
    EmcReadout er(theCrate);
    for(unsigned i(0);i<nSlots;i++) er.device(dev[i],i==0);
    vRrw.addUser(er);
    
    // Add histogram module
    HstChipNoise *hcn(new HstChipNoise(true));
    if(doHistograms) vrub.addUser(*hcn);

    // Add writer module
    DaqWriter dw(useWriteDmy,!useWriteAsc);
    vrub.addUser(dw);

    // Set print levels
    //for(unsigned i(0);i<vrub.size();i++) vrub[i]->printLevel(printLevel);

    RcdArena arena;
    SubAccessor accessor(arena);
    RcdArena *aArray(new RcdArena[256]);

    /*
    // Send Startup record
    initialiseRecord(arena,RcdHeader::startUp);
    vRrw.record(arena);
    vrub.record(arena);

    DvrCrcSlw dcs(theCrate);

    // Send slow controls record
    initialiseRecord(arena,RcdHeader::slowControl);
    dcs.record(arena);
    vRrw.record(arena);
    vrub.record(arena);
    */

    // Ignor Ctrl^C for a while
    signal(SIGINT,SIG_IGN);

    // Loop over runs
    for(unsigned iRun(0);iRun<nRun && continueFlag>0;iRun++) {
      while(pDsc->status()>DaqSlwControl::waiting) sleep(1);

      // Now catch Ctrl^C
      signal(SIGINT,signalHandler);
      signal(SIGTERM,signalHandler);
      signal(SIGUSR1,signalHandler);
      signal(SIGUSR2,signalHandler);

      initialiseRecord(arena,RcdHeader::runStart);
      vRrw.record(arena);
      vrub.record(arena);
    
      // Access the DaqRunStart
      std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
      assert(v.size()==1);

      // Loop over configurations
      const unsigned nCfg(v[0]->maximumNumberOfConfigurationsInRun());
      if(continueFlag==1) continueFlag=2;

      for(unsigned iCfg(0);iCfg<nCfg && continueFlag>1;iCfg++) {

	initialiseRecord(arena,RcdHeader::configurationStart);
	vRrw.record(arena);
	vrub.record(arena);

	// Access the DaqConfigurationStart
	std::vector<const DaqConfigurationStart*> v(accessor.extract<DaqConfigurationStart>());
	assert(v.size()==1);

	// Loop over spills
	const unsigned nSpl(v[0]->maximumNumberOfSpillsInConfiguration());
	if(continueFlag==2) continueFlag=3;

	for(unsigned iSpl(0);iSpl<nSpl && continueFlag>2;iSpl++) {

	  initialiseRecord(arena,RcdHeader::spillStart);
	  vRrw.record(arena);
	  vrub.record(arena);

	  // Access the DaqSpillStart
	  std::vector<const DaqSpillStart*> v(accessor.extract<DaqSpillStart>());
	  assert(v.size()==1);

	  // Loop over events
	  const unsigned nEvt(v[0]->maximumNumberOfEventsInSpill());
	  if(continueFlag==3) continueFlag=4;

	  unsigned iEvt(0);
	  for(iEvt=0;iEvt<nEvt && continueFlag>3;iEvt++) {

	    // Do event for some modules
	    initialiseRecord(aArray[iEvt],RcdHeader::event);
	    vRrw.record(aArray[iEvt]);
	  }
      
	  // Do spill end for some modules
	  if(continueFlag>2) initialiseRecord(arena,RcdHeader::spillEnd);
	  else               initialiseRecord(arena,RcdHeader::spillStop);
	  vRrw.record(arena);

	  // Do event for the rest of the modules
	  for(unsigned jEvt(0);jEvt<iEvt;jEvt++) vrub.record(aArray[jEvt]);

	  // Do spill end for the rest of the modules
	  vrub.record(arena);
	  hcn->update();

	  // Do a slow readout
	  initialiseRecord(arena,RcdHeader::slowReadout);
	  vRrw.record(arena);
	  vrub.record(arena);
	}

	if(continueFlag>1) initialiseRecord(arena,RcdHeader::configurationEnd);
	else               initialiseRecord(arena,RcdHeader::configurationStop);
	vRrw.record(arena);
	vrub.record(arena);
      }

      if(continueFlag>0) initialiseRecord(arena,RcdHeader::runEnd);
      else               initialiseRecord(arena,RcdHeader::runStop);
      vRrw.record(arena);
      vrub.record(arena);

      // Ignor Ctrl-C again
      signal(SIGINT,SIG_IGN);
    }

  } catch ( HardwareAccessException& e ) {
    cout << "*** Exception occurred : " << e.what() << endl;
    
  } catch ( exception e ) {
    cout << "*** Unknown exception occurred" << endl;
  }

  pDsc->command(DaqSlwControl::run);
}
