#ifndef MpsLaserReadout_HH
#define MpsLaserReadout_HH

#include <iostream>
#include <fstream>

// dual/inc/utl
#include "UtlPack.hh"

// dual/inc/rcd
#include "RcdUserRW.hh"
#include "RcdHeader.hh"

// dual/inc/sub
#include "SubInserter.hh"
#include "SubAccessor.hh"

// dual/inc/skt
#include "DuplexSocket.hh"


class MpsLaserReadout : public RcdUserRW {

public:
  MpsLaserReadout(std::string ipAddress="localhost",
		  unsigned port=15000, unsigned tries=10) :
    RcdUserRW(), _socket(ipAddress.c_str(),port,tries) {

    std::cout << std::endl << "MpsLaserReadout::ctor()  SOCKET ESTABLISHED"
	      << std::endl << std::endl;
  }

  virtual ~MpsLaserReadout() {
  }


  // The main driver for handling records
  bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "MpsLaserReadout::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
    
    // Check record type

    // Start and end of runs
    if(r.recordType()==RcdHeader::runStart ||
       r.recordType()==RcdHeader::runEnd) {

      // Send just the message ID = 3
      //             012345678901234567890123456789012345
      std::string s("300000000000000000000000000000000000");
      assert(sendAndRecv(s));

      // Convert string into the data object
      MpsLaserRunData x;
      assert(x.parse(s));

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsLaserReadout::record()"
		  << "  Received a MpsLaserRunData subrecord" << std::endl;
	if(doPrint(r.recordType(),2)) {
	  x.print(std::cout," ") << std::endl;
	}
      }

      // Add data to the record
      SubInserter inserter(r);
      inserter.insert<MpsLaserRunData>(x);
    }


    // Start of configuration 
    if(r.recordType()==RcdHeader::configurationStart) {

      // Look for data in the record to write to the laser
      SubAccessor accessor(r);
      std::vector<const MpsLaserConfigurationData*>
	v(accessor.access<MpsLaserConfigurationData>());
      
      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsLaserReadout::record()"
		  << "  Found " << v.size()
		  << " MpsLaserConfigurationData subrecords" << std::endl;
      }

      for(unsigned i(0);i<v.size();i++) {
	if(doPrint(r.recordType(),2)) {
	  v[i]->print(std::cout," ") << std::endl;
	}

	// Check for write flag to be set
	if(v[i]->write()) {
	  std::string s(v[i]->parse());
	  sendAndRecv(s);

	  // Save configuration
	  _configurationData=*(v[i]);
	}
      }

      // Read back the configuration as a cross-check
      assert(readConfigurationData(r));
    }
    

    // End of configuration
    if(r.recordType()==RcdHeader::configurationEnd) {

      // Read back the configuration as a cross-check
      assert(readConfigurationData(r));
    }


    // Bunch train is for data taking
    if(r.recordType()==RcdHeader::bunchTrain) {

      // Decide from configuration whether to fire the laser
      if(_configurationData.triggerMode()==1) { // Random value for now

	// Fire the laser using message ID = 4
	//             012345678901234567890123456789012345
	std::string s("400000000000000000000000000000000000");
	assert(sendAndRecv(s));
      }
    }

    return true;
  }
  

  // Read configuration data
  bool readConfigurationData(RcdRecord &r) {

    // Request data with message ID = 2
    //             012345678901234567890123456789012345
    std::string s("200000000000000000000000000000000000");
    assert(sendAndRecv(s));

    // Convert string into the data object
    MpsLaserConfigurationData x;
    assert(x.parse(s));
    
    if(doPrint(r.recordType(),1)) {
      std::cout << "MpsLaserReadout::readConfigurationData()"
		<< "  Received a MpsLaserConfigurationData subrecord" << std::endl;
      if(doPrint(r.recordType(),2)) {
	x.print(std::cout," ") << std::endl;
      }
    }
    
    // Add it to the record
    SubInserter inserter(r);
    inserter.insert<MpsLaserConfigurationData>(x);

    return true;
  }


  // Handle the socket interactions  
  bool sendAndRecv(std::string &s) {
    const int nBytes(s.size());

    // Send the string
    if(!_socket.send(s.c_str(),nBytes)) {
      std::cerr << "MpsLaserReadout::sendAndRecv()"
		<< "  Error writing to socket;"
		<< " number of bytes written < " << nBytes << std::endl;
      perror(0);
      return false;
    }

    // Read back the return string
    int n(-1);
    n=_socket.recv(&(s[0]),nBytes);
    if(n!=nBytes) {
      std::cerr << "MpsLaserReadout::sendAndRecv()"
		  << "  Error reading from socket;"
		<< " number of bytes written = " << n << " != " << nBytes
		<< std::endl;
      perror(0);
      return false;
    }

    return true;
  }
    

private:

  DuplexSocket _socket;
  MpsLaserConfigurationData _configurationData;
};

#endif
