#ifndef UsbDaqDevice_HH
#define UsbDaqDevice_HH

#include <unistd.h>
#include <sys/wait.h>

#include <cmath>
#include <cstdlib>
#include <iostream>

#include "HardwareAccessException.hh"
#include "CaliceMaps1TestSystemInterface.hh"

#include "ShmObject.hh"

#include "MpsUsbDaqRunData.hh"
#include "MpsUsbDaqConfigurationData.hh"
#include "MpsUsbDaqBunchTrainData.hh"
#include "MpsPcb1ConfigurationData.hh"
#include "MpsPcb1SlowReadoutData.hh"
#include "MpsSensor1ConfigurationData.hh"
#include "MpsSensor1BunchTrainData.hh"


class UsbDaqDevice {

public:
  class MpsSensor1BunchTrainArena : public MpsSensor1BunchTrainData {
  public:
    unsigned char _arena[64*1024-sizeof(MpsSensor1BunchTrainData)];
  };


  UsbDaqDevice() : _pUsbDaq(0), _alive(false), _readAheadDAQ(0) {

    try {
      _pUsbDaq=new CALICEMAPS1::CaliceMaps1TestSystemInterface;
      _alive=true;

    } catch ( USBDAQ::USBDAQException& e ) {
      _pUsbDaq=0;
    }

    // Reset everything
    if(_alive) {
      try {
	//_pUsbDaq->LogicReset();
      } catch ( USBDAQ::USBDAQException & e  ) {
	std::cout << e << std::endl;
      }

      // Redo it as lost by above reset
      //_pUsbDaq=new CALICEMAPS1::CaliceMaps1TestSystemInterface;

      // Make shared memory area for bunch train read-ahead
      unsigned shmKey(0xbeef0000+_pUsbDaq->ReadSensorId());
      _shmBuffer=new ShmObject<MpsSensor1BunchTrainArena>(shmKey);
      //_sensorBunchTrainBuffer=(MpsSensor1BunchTrainData*)(_shmBuffer->payload());
      _sensorBunchTrainBuffer=_shmBuffer->payload();
    }

    _print=false;
  }

  virtual ~UsbDaqDevice() {
    delete _pUsbDaq;
    delete _shmBuffer;
  }

  CALICEMAPS1::CaliceMaps1TestSystemInterface* usbDaq() {
    return _pUsbDaq;
  }

  bool alive() {
    return _alive;
  }

  // Run

  bool runReset() {
    _pUsbDaq->EnableLVDSBuffers();
    _pUsbDaq->IssueTopConfigReset();
    _pUsbDaq->IssueSlowConfigReset();
    _pUsbDaq->IssueReadbackConfigReset();

    memset(_sensor1ConfigurationStripCheckBuffer,0xaa,3*840);

    return true;
  }

  bool readMpsUsbDaqRunData(MpsUsbDaqRunData &m) {
    m.zeroIds();

    m.master(_pUsbDaq->IsMaster());
    m.usbDaqAddress(_pUsbDaq->ReadDAQAddress());
    m.sensorId(_pUsbDaq->ReadSensorId());
    m.firmwareVersion(_pUsbDaq->ReadFirmwareRevision());
    m.firmwareDate(_pUsbDaq->ReadFirmwareRevisionUTC());

    return true;
  }

  // Configuration

  bool writeMpsUsbDaqConfigurationData(const MpsUsbDaqConfigurationData &m) {
    _usbDaqConfigurationData=m;

    try {

      _pUsbDaq->SetSpillCycleCount(m.spillCycleCount());
      
      for(unsigned i(0);i<3;i++) _pUsbDaq->SetClockPhase(i,m.clockPhase(i));
      
      _pUsbDaq->SetDebugReset200Duration(m.debugReset200Duration());
      _pUsbDaq->SetDebugReset600Duration(m.debugReset600Duration());
      _pUsbDaq->SetDebugDiodeResetDuration(m.debugDiodeResetDuration());
      _debugResetDuration[0]=m.debugReset200Duration();
      _debugResetDuration[1]=m.debugReset600Duration();
      _debugResetDuration[2]=m.debugDiodeResetDuration();
      
      _pUsbDaq->SetDebugReset200Hold(m.debugReset200Hold());
      _pUsbDaq->SetDebugReset600Hold(m.debugReset600Hold());
      _pUsbDaq->SetDebugDiodeResetHold(m.debugDiodeResetHold());
      
      _pUsbDaq->SetDebugTrim(m.debugTrim());
      _debugTrim=m.debugTrim();

      _pUsbDaq->SetHitOverride(m.hitOverride());
      _pUsbDaq->SetMonoPOR(m.monoPOR());
      
      _pUsbDaq->SetSenseEnable(m.senseEnable());
      _pUsbDaq->SetDebugHitInEnable(m.debugHitInEnable(),m.debugHitInEnableValue());
      _hitInEnableValue=m.debugHitInEnableValue();
      
      _pUsbDaq->SetFastPhiOverride(m.fastPhiOverride());
      _pUsbDaq->SetDiscriminatorThresholds(m.discriminatorThreshold(0),
					   m.discriminatorThreshold(1));
      _discriminatorThreshold[0]=m.discriminatorThreshold(0);
      _discriminatorThreshold[1]=m.discriminatorThreshold(1);
      
      _pUsbDaq->SetSenseDelay(m.senseDelay());
      _pUsbDaq->SetDiodeResetDuration(m.diodeResetDuration());
      _pUsbDaq->SetShaperResetDuration(m.shaperResetDuration());
      _pUsbDaq->SetPreAmpResetDuration(m.sampleResetDuration());
      _pUsbDaq->SetTimeStampPrescale(m.timeStampPrescale());
      
      _pUsbDaq->SetSpillMode(m.spillMode());
      _readSensor=!m.spillModeInhibitReadout();
      _readHistory=m.spillModeHistoryEnable();
      
      _pUsbDaq->SetPixelEnable12(m.pixelEnable12());
      _pUsbDaq->SetPixelEnable34(m.pixelEnable34());
      
      _pUsbDaq->SetMCResetDuration(m.masterClockResetDuration());
      _pUsbDaq->SetSramFillClockSleep(m.sramFillClockSleep());
      _pUsbDaq->SetReadEnableOverride(m.readEnableOverride());
      
      _pUsbDaq->SetReadoutColumnOrder(m.readoutColumnOrder()); 
      _pUsbDaq->SetStaticTimeStamp(m.staticTimeStamp());
      
      _pUsbDaq->SetInitBPolarity(m.initBPolarity());
      _pUsbDaq->SetRstBPolarity(m.rstBPolarity());
      _pUsbDaq->SetFwdBPolarity(m.fwdBPolarity());
      
      _pUsbDaq->SetSlowSpillPhi(m.slowSpillPhi());
      
      _pUsbDaq->SetTimeStampOverride(m.timeStampOverride());
      
      _pUsbDaq->SetPostPixResetSleepDuration(m.postPixResetSleepDuration());
      
      _pUsbDaq->SetReadoutStartIndex(m.readoutStartIndex());
      _pUsbDaq->SetStackPointerOffset(m.stackPointerOffset());
      
      _pUsbDaq->SetTestTrigConfig(m.testTriggerMode());
      _pUsbDaq->SetTestTrigStart(m.testTriggerStart());
      _pUsbDaq->SetTestTrigStop(m.testTriggerStop());
      
      _pUsbDaq->SetSpillCycleStart(m.spillCycleStart());

      _pUsbDaq->SetTriggerMask(m.triggerMask());
      _pUsbDaq->SetTriggerSource(m.triggerSource());
      _pUsbDaq->SetTriggerEnable(m.triggerEnable());

      _pUsbDaq->SetDiodeReset12Hold(m.diodeResetHold());
      _pUsbDaq->SetSampleReset34Hold(m.sampleResetHold());
      _pUsbDaq->SetShaperReset34Hold(m.shaperResetHold());

    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }

    return true;
  }

  bool readMpsUsbDaqConfigurationData(MpsUsbDaqConfigurationData &m) {
    m=_usbDaqConfigurationData;

    try {
      m.zeroBools();

      m.spillCycleCount(_pUsbDaq->ReadSpillCycleCount());

      for(unsigned i(0);i<3;i++) m.clockPhase(i,_pUsbDaq->ReadClockPhase(i));

      //?????
      //m.debugReset200Duration(_pUsbDaq->ReadDebugReset200Duration());
      //m.debugReset200Duration(_pUsbDaq->ReadDebugReset600Duration());
      //m.debugDiodeResetDuration(_pUsbDaq->ReadDebugDiodeResetDuration());
      m.debugReset200Duration(_debugResetDuration[0]);
      m.debugReset600Duration(_debugResetDuration[1]);
      m.debugDiodeResetDuration(_debugResetDuration[2]);
      
      m.debugReset200Hold(_pUsbDaq->ReadDebugReset200Hold());
      m.debugReset600Hold(_pUsbDaq->ReadDebugReset600Hold());
      m.debugDiodeResetHold(_pUsbDaq->ReadDebugDiodeResetHold());

      //???
      m.debugTrim(_pUsbDaq->ReadDebugTrim());
      //m.debugTrim(_debugTrim);

      m.hitOverride(_pUsbDaq->ReadHitOverride());
      m.monoPOR(_pUsbDaq->ReadMonoPOR());
      m.pixelEnable12(_pUsbDaq->ReadPixelEnable12());
      m.pixelEnable34(_pUsbDaq->ReadPixelEnable34());
      m.senseEnable(_pUsbDaq->ReadSenseEnable());
      
      m.debugHitInEnable(_pUsbDaq->ReadDebugHitInEnable());
      // ???
      m.debugHitInEnableValue(_hitInEnableValue);
      
      m.fastPhiOverride(_pUsbDaq->ReadFastPhiOverride());
      
      //???
      m.discriminatorThreshold(0,_pUsbDaq->ReadDiscriminatorThreshold(0));
      m.discriminatorThreshold(1,_pUsbDaq->ReadDiscriminatorThreshold(1));
      //m.discriminatorThreshold(0,_discriminatorThreshold[0]);
      //m.discriminatorThreshold(1,_discriminatorThreshold[1]);
      
      m.senseDelay(_pUsbDaq->ReadSenseDelay());
      m.diodeResetDuration(_pUsbDaq->ReadDiodeResetDuration());
      m.shaperResetDuration(_pUsbDaq->ReadShaperResetDuration());
      m.sampleResetDuration(_pUsbDaq->ReadPreAmpResetDuration());
      m.timeStampPrescale(_pUsbDaq->ReadTimeStampPrescale());
      m.spillMode(_pUsbDaq->ReadSpillMode());
      
      m.masterClockResetDuration(_pUsbDaq->ReadMCResetDuration());
      m.sramFillClockSleep(_pUsbDaq->ReadSramFillClockSleep());
      m.readEnableOverride(_pUsbDaq->ReadReadEnableOverride());
      
      m.readoutColumnOrder(_pUsbDaq->ReadReadoutColumnOrder());
      m.staticTimeStamp(_pUsbDaq->ReadStaticTimeStamp());
      
      m.initBPolarity(_pUsbDaq->ReadInitBPolarity());
      m.rstBPolarity(_pUsbDaq->ReadRstBPolarity());
      m.fwdBPolarity(_pUsbDaq->ReadFwdBPolarity());
      
      m.slowSpillPhi(_pUsbDaq->ReadSlowSpillPhi());
      
      m.timeStampOverride(_pUsbDaq->ReadTimeStampOverride());
      
      m.postPixResetSleepDuration(_pUsbDaq->ReadPostPixResetSleepDuration());
      
      m.readoutStartIndex(_pUsbDaq->ReadReadoutStartIndex());
      m.stackPointerOffset(_pUsbDaq->ReadStackPointerOffset());
      
      m.testTriggerMode(_pUsbDaq->ReadTestTrigConfig());
      m.testTriggerStart(_pUsbDaq->ReadTestTrigStart());
      m.testTriggerStop(_pUsbDaq->ReadTestTrigStop());

      m.spillCycleStart(_pUsbDaq->ReadSpillCycleStart());

      m.triggerMask(_pUsbDaq->ReadTriggerMask());
      m.triggerSource(_pUsbDaq->ReadTriggerSource());
      m.triggerEnable(_pUsbDaq->ReadTriggerEnable());

      m.diodeResetHold(_pUsbDaq->ReadDiodeReset12Hold());
      m.sampleResetHold(_pUsbDaq->ReadSampleReset34Hold());
      m.shaperResetHold(_pUsbDaq->ReadShaperReset34Hold());

    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }

    return true;
  }

  bool writeMpsPcb1ConfigurationData(const MpsPcb1ConfigurationData &m) {
    try {
      _pUsbDaq->SetDebugVthPos(m.vDebugThPos());
      _pUsbDaq->SetDebugVthNeg(m.vDebugThNeg());
      _pUsbDaq->SetI34DebugSFBias(m.iDebugSFBias());
      
      _pUsbDaq->SetVth12Pos(m.v12ThPos());
      _pUsbDaq->SetVth12Neg(m.v12ThNeg());
      _pUsbDaq->SetVth34Pos(m.v34ThPos());
      _pUsbDaq->SetVth34Neg(m.v34ThNeg());
      
      _pUsbDaq->SetISenseColRef(m.iSenseColRef());
      _pUsbDaq->SetPreAmp34VCasc(m.v34PreAmpCasc());
      _pUsbDaq->SetISenseBias(m.iSenseBias());
      _pUsbDaq->SetI12Comp1Bias(m.i12CompBias1());
      _pUsbDaq->SetI12Comp2Bias(m.i12CompBias2());
      _pUsbDaq->SetISenseIoutBias(m.iSenseOutBias());
      _pUsbDaq->SetISenseCompBias(m.iSenseCompBias());
      _pUsbDaq->SetI12IoutBias(m.i12OutBias());
      _pUsbDaq->SetI12PreAmpBias(m.i12PreAmpBias());
      _pUsbDaq->SetI12ShaperBias(m.i12ShaperBias());
      _pUsbDaq->SetI12CompBiasTrim(m.i12CompBiasTrim());
      _pUsbDaq->SetI34CompBiasTrim(m.i34CompBiasTrim());
      
      _pUsbDaq->SetVRst(m.v34Reset());
      _pUsbDaq->SetI34SFBias(m.i34SFBias());
      _pUsbDaq->SetI34CompBias1(m.i34CompBias1());
      _pUsbDaq->SetI34CompBias2(m.i34CompBias2());
      _pUsbDaq->SetI34MSOBias1(m.i34MSOBias1());
      _pUsbDaq->SetI34MSOBias2(m.i34MSOBias2());
      _pUsbDaq->SetI34PreAmpBias(m.i34PreAmpBias());
      _pUsbDaq->SetI34OutBias(m.i34OutBias());
      _pUsbDaq->SetI34OutSFBias(m.i34OutSFBias());
      _pUsbDaq->SetI12MSOBias1(m.i12MSOBias1());
      
      _pUsbDaq->SetShaper12VCasc(m.v12ShaperCasc());
      _pUsbDaq->SetPreAmp12VCasc(m.v12PreAmpCasc());
      
      _unconnected=m.unconnected();
      
    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }

    return true;
  }
  
  bool readMpsPcb1ConfigurationData(MpsPcb1ConfigurationData &m) {
    try {
      m.vDebugThPos(_pUsbDaq->ReadDebugVthPos());
      m.vDebugThNeg(_pUsbDaq->ReadDebugVthNeg());
      m.iDebugSFBias(_pUsbDaq->ReadI34DebugSFBias());
      
      m.v12ThPos(_pUsbDaq->ReadVth12Pos());
      m.v12ThNeg(_pUsbDaq->ReadVth12Neg());
      m.v34ThPos(_pUsbDaq->ReadVth34Pos());
      m.v34ThNeg(_pUsbDaq->ReadVth34Neg());
      
      m.iSenseColRef(_pUsbDaq->ReadISenseColRef());
      //m.iSenseColRef(_pUsbDaq->ReadSenseColRef());
      m.v34PreAmpCasc(_pUsbDaq->ReadPreAmp34VCasc());
      m.iSenseBias(_pUsbDaq->ReadISenseBias());
      m.i12CompBias1(_pUsbDaq->ReadI12Comp1Bias());
      m.i12CompBias2(_pUsbDaq->ReadI12Comp2Bias());
      m.iSenseOutBias(_pUsbDaq->ReadISenseIoutBias());
      m.iSenseCompBias(_pUsbDaq->ReadISenseCompBias());
      m.i12OutBias(_pUsbDaq->ReadI12IoutBias());
      m.i12PreAmpBias(_pUsbDaq->ReadI12PreAmpBias());
      m.i12ShaperBias(_pUsbDaq->ReadI12ShaperBias());
      m.i12CompBiasTrim(_pUsbDaq->ReadI12CompBiasTrim());
      m.i34CompBiasTrim(_pUsbDaq->ReadI34CompBiasTrim());
      
      m.v34Reset(_pUsbDaq->ReadVRst());
      m.i34SFBias(_pUsbDaq->ReadI34SFBias());
      m.i34CompBias1(_pUsbDaq->ReadI34CompBias1());
      m.i34CompBias2(_pUsbDaq->ReadI34CompBias2());
      m.i34MSOBias1(_pUsbDaq->ReadI34MSOBias1());
      m.i34MSOBias2(_pUsbDaq->ReadI34MSOBias2());
      m.i34PreAmpBias(_pUsbDaq->ReadI34PreAmpBias());
      m.i34OutBias(_pUsbDaq->ReadI34OutBias());
      m.i34OutSFBias(_pUsbDaq->ReadI34OutSFBias());
      m.i12MSOBias1(_pUsbDaq->ReadI12MSOBias1());
      
      m.v12ShaperCasc(_pUsbDaq->ReadShaper12VCasc());
      m.v12PreAmpCasc(_pUsbDaq->ReadPreAmp12VCasc());
      
      m.unconnected(_unconnected);
      
    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }

    return true;
  }

  bool loadMpsSensor1ConfigurationData(const MpsSensor1ConfigurationData &w,
				       MpsSensor1ConfigurationData &r) {
    try {

    //_pUsbDaq->LoadConfig((unsigned char*)w.data(),r.data(),w.numberOfColumns(),w.numberOfRows());

    std::cout << "CALLED" << std::endl;
    const unsigned *p((const unsigned*)w.data());
    /*
    for(unsigned i(0);i<6*840;i++) {
      std::cout << "Word p[" << i << "] = " << printHex(p[i]) << std::endl;
    }
    */
    r=w;
    _pUsbDaq->LoadConfig((const USBDAQ::u8*)w.data(),(USBDAQ::u8*)_sensor1ConfigurationStripCheckBuffer,
			 w.numberOfBytes(),w.numberOfStrips());
    r.stripCheckData(_sensor1ConfigurationStripCheckBuffer);

    _pUsbDaq->ReadBackConfig((USBDAQ::u8*)_sensor1ConfigurationMaskTrimBuffer,r.numberOfMaskTrimBytes(),r.numberOfStrips());
    r.maskTrimData(_sensor1ConfigurationMaskTrimBuffer);

    const unsigned *q((const unsigned*)r.data());
    for(unsigned i(0);i<6*840;i++) {
      if(q[i]!=p[i]) {
	std::cout << "Word p[" << i << "] = " << printHex(p[i]) << std::endl;
	std::cout << "Word q[" << i << "] = " << printHex(q[i]) << std::endl;
      }
    }

    

    std::cout << "DONE" << std::endl;

    /*
    unsigned p[1024],q[1024];
    for(unsigned i(0);i<6;i++) {
      p[i]=i;q[i]=0;
      std::cout << "Word p[" << i << "] = " << printHex(p[i]) << std::endl;
    }
    _pUsbDaq->LoadTopConfigRow((const USBDAQ::u8*)p,(USBDAQ::u8*)q,24);
    for(unsigned i(0);i<6;i++) {
      std::cout << "Word q[" << i << "] = " << printHex(q[i]) << std::endl;
    }
    std::cout << "DONE" << std::endl;
    */

    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }
    return true;
  }

  bool writeMpsSensor1ConfigurationData(const MpsSensor1ConfigurationData &m) {
    try {

      // Set true here to not actually write values (for debug)
      if(false) {
	unsigned n(0);
	for(unsigned i(0);i<840;i++) {
	  UtlPack b(m.stripCheckBits(i));
	  _sensor1ConfigurationStripCheckBuffer[n]=b.byte(0);
	  n++;
	  _sensor1ConfigurationStripCheckBuffer[n]=b.byte(1);
	  n++;
	  _sensor1ConfigurationStripCheckBuffer[n]=b.byte(2);
	  n++;
	}
	
      } else {
	_pUsbDaq->LoadConfig((const USBDAQ::u8*)m.data(),(USBDAQ::u8*)_sensor1ConfigurationStripCheckBuffer,
			     m.numberOfBytes(),m.numberOfStrips());
      }
      
    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      std::cerr << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
      std::cout << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
      exit(1);
    }
    return true;
  }

  bool readMpsSensor1ConfigurationData(MpsSensor1ConfigurationData &m) {
    try {
      m.stripCheckData(_sensor1ConfigurationStripCheckBuffer);
      _pUsbDaq->ReadBackConfig((USBDAQ::u8*)_sensor1ConfigurationMaskTrimBuffer,m.numberOfMaskTrimBytes(),m.numberOfStrips());
      m.maskTrimData(_sensor1ConfigurationMaskTrimBuffer);

    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      std::cerr << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
      std::cout << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
      exit(1);
    }
    return true;
  }


  // Bunch train

  bool bunchTrainReset() {
    if(_print) std::cout << "bunchTrainReset()" << std::endl;
    _readAheadDAQ=0;
    return true;
  }

  bool takeBunchTrain() {
    _readAheadDAQ=0;

    //_pUsbDaq->SetConfigReadoutMode();

    //usleep(1000000);

    try {
      _pUsbDaq->StartSpill();

    } catch ( USBDAQ::USBDAQException& e ) {
      std::cerr << e << std::endl;
      std::cout << e << std::endl;
      exit(1);
    }

    //sleep(1);
    //_pUsbDaq->ClearConfigReadoutMode();
    return true;
  }

  bool readMpsUsbDaqBunchTrainData(MpsUsbDaqBunchTrainData &m) {
    m.numberOfTries(1);

    unsigned nTags(0);

    if(_readHistory) {

      MpsUsbDaqBunchTrainDatum *d(m.data());

      //unsigned nBuffer(*((const unsigned*)_buffer));
      unsigned nBuffer(0);
      
      try {
	nBuffer=_pUsbDaq->ReadTriggerHistory(_buffer,8192);
	
      } catch ( USBDAQ::USBDAQException& e ) {
	std::cerr << e << std::endl;
	std::cout << e << std::endl;
	std::cerr << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
	std::cout << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
	exit(1);
      }
      
      // Currently values are inverted
      for(unsigned i(0);i<nBuffer;i++) {


        //////////////////////////////////////////////////////////////////////////////////

        // MASSIVE HACK!!!!! 07/12/07
        // BIT 5 IS ALWAYS ON!!!

        //_buffer[i]&=0xdf;

	// REMOVE ALL UPPER BITS FOR NOW!!!

        _buffer[i]&=0x03;

        //////////////////////////////////////////////////////////////////////////////////


	//if(i<16) std::cout << "i = " << i << " _buffer[i] = " << (unsigned)_buffer[i] << std::endl;
	if(_buffer[i]!=3) {
	  d[nTags].channels(_buffer[i]);
	  d[nTags].channel(0,!d[nTags].channel(0));
	  d[nTags].channel(1,!d[nTags].channel(1));
	  d[nTags].timeStamp(i);
	  nTags++;
	}
      }
    }

    m.numberOfTags(nTags);

    return true;
  }

  bool readMpsPcb1SlowReadoutData(MpsPcb1SlowReadoutData &m) {
    m.temperature(_pUsbDaq->ReadSensorCardTemperature());
    return true;
  }

  bool transferMpsSensor1BunchTrainData() {
    _readAheadDAQ=fork();

    if(_readAheadDAQ==0) {
      //std::cout << "Child process for sensor " << _pUsbDaq->ReadSensorId() << " started" << std::endl;
      readMpsSensor1BunchTrainData(*_sensorBunchTrainBuffer);
      //_sensorBunchTrainBuffer->print(std::cout,"CHILD  ") << std::endl;
      //std::cout << "Child process for sensor " << _pUsbDaq->ReadSensorId() << " completed" << std::endl;
      exit(0);
    }

    return true;
  }

  bool readMpsSensor1BunchTrainData(MpsSensor1BunchTrainData &m) {
    if(_readAheadDAQ!=0) {
      //std::cout << "Parent process waiting" << std::endl;
      waitpid(_readAheadDAQ,0,0);
      //std::cout << "Parent process copying" << std::endl;
      memcpy(&m,_sensorBunchTrainBuffer,sizeof(MpsSensor1BunchTrainData)+4*_sensorBunchTrainBuffer->totalNumberOfHits());
      //m.print(std::cout,"PARENT ") << std::endl;
      return true;
    }

    UtlPack status;
    status.halfWord(0,_pUsbDaq->ReadSensorStatusFlags());

    /*
    unsigned short x[4];
    _pUsbDaq->ReadSensorWordCount(x);
    for(unsigned i(0);i<4;i++) m.numberOfRegionHits(i,x[i]);
    */

    /*
    USBDAQ::u32 p[4*4096];

    try {
      _pUsbDaq->DAQ(p,4*4096);
    
    } catch ( USBDAQ::USBDAQException& e ) {
      std::cout << e << std::endl;
      exit(1);
    }

    for(unsigned i(0);i<64;i++) {
      std::cout << printHex((unsigned)p[i]) << std::endl;
    }
    */

    if(_readSensor) {

      try {
	_pUsbDaq->DAQ((USBDAQ::u32*)m.daqData(),64*1024);
	//*(m.daqData())=rand()%12768;
	//memset(m.daqData()+1,0,4*(2+m.totalNumberOfHits()));
	
      } catch ( USBDAQ::USBDAQException& e ) {
	std::cerr << e << std::endl;
	std::cout << e << std::endl;
	std::cerr << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
	std::cout << "Exception from USB_DAQ address " << _pUsbDaq->ReadDAQAddress() << std::endl;
	exit(1);
      }

    } else {
      *(m.daqData()  )=0;
      *(m.daqData()+1)=0;
      *(m.daqData()+2)=0;
    }

    status.halfWord(1,_pUsbDaq->ReadSensorStatusFlags());
    m.finalBits(status.word());

    return true;
  }

private:
  CALICEMAPS1::CaliceMaps1TestSystemInterface* _pUsbDaq;
  bool _alive;
  bool _print;

  unsigned char _buffer[8*1024];

  unsigned char _sensor1ConfigurationStripCheckBuffer[3*840];
  unsigned char _sensor1ConfigurationMaskTrimBuffer[21*840];

  pid_t _readAheadDAQ;
  ShmObject<MpsSensor1BunchTrainArena> *_shmBuffer;
  MpsSensor1BunchTrainData *_sensorBunchTrainBuffer;
					       

  bool _readSensor;
  bool _readHistory;

  // TEMP!!!!
  unsigned short _debugResetDuration[3];
  unsigned short _debugTrim;
  unsigned short _discriminatorThreshold[2];
  unsigned short _hitInEnableValue;
  unsigned short _unconnected;
  MpsUsbDaqConfigurationData _usbDaqConfigurationData;
};

#endif
