#ifndef HstGeneric_HH
#define HstGeneric_HH

#include <cassert>

#include "TMapFile.h"

#include "RcdMultiUserRO.hh"

#include "HstRecord.hh"
#include "HstCrcNoise.hh"
#include "HstCrcFeNoise.hh"
#include "HstCrcSignal.hh"
#include "HstTriggerPoll.hh"
#include "HstSpill.hh"
#include "HstBeTrgHistory.hh"
#include "HstCheckConfiguration.hh"
#include "HstCheckSubrecords.hh"
#include "HstLc1176.hh"


class HstGeneric : public RcdMultiUserRO {

public:
  HstGeneric(const std::string mapFile="HstGeneric.map", unsigned mBytes=64) {
    
    // The following magic number needs to be got from running a root session
    // interactively using rootn.exe and typing
    //   root [0] m = TMapFile::Create("dummy.map","recreate",64*1024*1024);
    //   root [1] m->Print()
    // which gives
    //
    //    Memory mapped file:   dummy.map
    //      Title:                
    //  Option:               CREATE
    //      Mapped Memory region: 0xb35c2000 - 0xb75c2000 (64.00 MB)
    //      Current breakval:     0xb35c9000
    //
    // where the magic number is the first in the "Mapped Memory region"
    // range. Don't forget to rm dummy.map afterwards.
    //
    // The code below will then only work for sizes up to the size
    // used (64MBytes here). Larger sizes will cause a crash.

    assert(mBytes<=64);
    TMapFile::SetMapAddress(0xb35c2000);

    // Create a new memory mapped file. The memory mapped file can be
    // opened in an other process on the same machine and the objects
    // stored in it can be accessed.
    
    _mfile = TMapFile::Create(mapFile.c_str(),"RECREATE",mBytes*1024*1024,
			     "DAQ Memory Mapped Histogram File");

    // Print status of mapped file
    _mfile->Print();

    // Enable histogramming
    _doHists=true;
    _ignorRunType=false;
    _nEvents=0;
    _rEvents=10;
    _rEvents=1000; // TEMP slow updating between acquisitions

    // Add modules
    _hstRecord=new HstRecord;
    addUser(*_hstRecord);

    _hstCrcNoise=new HstCrcNoise;
    addUser(*_hstCrcNoise);
    
    _hstCrcFeNoise=new HstCrcFeNoise;
    addUser(*_hstCrcFeNoise);
    
    _hstCrcSignal=new HstCrcSignal;
    addUser(*_hstCrcSignal);

    _hstTriggerPoll=new HstTriggerPoll;
    addUser(*_hstTriggerPoll);

    _hstSpill=new HstSpill;
    addUser(*_hstSpill);

    _hstBeTrgHistory=new HstBeTrgHistory;
    addUser(*_hstBeTrgHistory);

    _hstCheckConfiguration=new HstCheckConfiguration;
    addUser(*_hstCheckConfiguration);

    _hstCheckSubrecords=new HstCheckSubrecords;
    addUser(*_hstCheckSubrecords);

    _hstLc1176=new HstLc1176;
    addUser(*_hstLc1176);
    
    // Print status of mapped file
    //_mfile->Update();
    _mfile->Print();
    //_mfile->ls();
  }

  virtual ~HstGeneric() {
    delete _hstRecord;
    delete _hstCrcNoise;
    delete _hstCrcFeNoise;
    delete _hstCrcSignal;
    delete _hstTriggerPoll;
    delete _hstSpill;
    delete _hstBeTrgHistory;
    delete _hstCheckConfiguration;
    delete _hstCheckSubrecords;
    delete _hstLc1176;
    
    _mfile->Close();
 }

  virtual bool record(const RcdRecord &r) {

    // Possibly disable histogramming
    if(r.recordType()==RcdHeader::runStart) {
      _doHists=true;
      if(!_ignorRunType) {
	SubAccessor accessor(r);
	std::vector<const DaqRunStart*>
	  v(accessor.extract<DaqRunStart>());

	if(v.size()==1) {
	  _doHists=v[0]->runType().histogramRun();
	  printLevel(v[0]->runType().printLevel());
	}
      }
    }

    if(_doHists) {
      if(!RcdMultiUserRO::record(r)) return false;

      // updates all objects in shared memory
      if(r.recordType()!=RcdHeader::trigger &&
      	 r.recordType()!=RcdHeader::event) _mfile->Update();

      // Never update for a trigger, only an event
      if(r.recordType()==RcdHeader::event) {
	if(_rEvents>0 && (_nEvents%_rEvents)==0) _mfile->Update();
	_nEvents++;
      }
    }
    
    // Re-enable histogramming
    if(r.recordType()==RcdHeader::runEnd) _doHists=true;

    return true;
  }

  void ignorRunType(bool b) {
    _ignorRunType=b;
  }

  void updatePeriod(unsigned r) {
    _rEvents=r;
  }


private:
  TMapFile *_mfile;

  bool _ignorRunType;
  bool _doHists;
  unsigned _nEvents;
  unsigned _rEvents;

  HstRecord *_hstRecord;
  HstCrcNoise *_hstCrcNoise;
  HstCrcFeNoise *_hstCrcFeNoise;
  HstCrcSignal *_hstCrcSignal;
  HstTriggerPoll *_hstTriggerPoll;
  HstSpill *_hstSpill;
  HstBeTrgHistory *_hstBeTrgHistory;
  HstCheckConfiguration *_hstCheckConfiguration;
  HstCheckSubrecords *_hstCheckSubrecords;
  HstLc1176 *_hstLc1176;
};

#endif
