#ifndef RunRunner_HH
#define RunRunner_HH

#include <sstream>
#include <fstream>
#include <iostream>

#include "DaqRunType.hh"
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqAcquisitionStart.hh"

#include "RcdMultiUserRW.hh"
#include "RcdMultiUserRO.hh"

#include "SubAccessor.hh"

#include "DaqRunStart.hh"

#include "RunControl.hh"
#include "ShmObject.hh"


class RunRunner {

public:
  RunRunner() :
    _shmRunControl(RunControl::shmKey), _pRc(_shmRunControl.payload()) {
    assert(_pRc!=0);
  }

  virtual ~RunRunner() {
  }

  /*
  unsigned continuationFlag() const {
    return _continuationFlag;
  }
  */

  void continuationFlag(unsigned c) {
    if(c==0) _pRc->flag(RunControl::shutDown);
    if(c==1) _pRc->flag(RunControl::runEnd);
    if(c==2) _pRc->flag(RunControl::configurationEnd);
  }

  //bool run(unsigned nr, DaqRunType rt, RcdMultiUserRW rw, RcdMultiUserRO ro) {
  bool run(RcdMultiUserRW rw, RcdMultiUserRO ro) {
    _pRc->flag(RunControl::daqContinue);

    // Define record memory
    //RcdArenas *pArena(new RcdArenas); // WAY TOO BIG!!!
    RcdArena *pArena(new RcdArena);
    assert(pArena!=0);

    RcdRecord &arena(*pArena);
    SubAccessor accessor(arena);

    // Outer loop
    while(_pRc->flag()>RunControl::shutDown) {

      // Get run information from shared memory
      _pRc->print(std::cout,"NEW RUN ==>> ");
      unsigned nRun(_pRc->numberOfRuns());
      //unsigned pl(_pRc->printLevel());
      DaqRunStart rs(_pRc->runStart());
      DaqRunType runType(rs.runType());
      
      rw.printLevel(runType.printLevel());
      ro.printLevel(runType.printLevel());

      // Reset to default (slow monitoring) after copying contents
      //_pRc->reset();
     
      // Set run type for DAQ configuration module
      //dc.runStart(rs);

      if(_pRc->flag()==RunControl::sequenceEnd) _pRc->flag(RunControl::daqContinue);


      // Loop over runs
      for(unsigned iRun(0);(iRun<nRun || nRun==0) && _pRc->flag()>RunControl::sequenceEnd;iRun++) {
	arena.initialise(RcdHeader::runStart);
	rw.record(arena);
	ro.record(arena);
	
	// Initialise time of last slow record 
	unsigned lastSlowRecord(0);
	
	// If last run was stopped using the flag, allow the next
	if(_pRc->flag()==RunControl::runEnd) _pRc->flag(RunControl::daqContinue);
	//while(_pRc->flag()==RunControl::1) sleep(1); //???
	

	// Loop over configurations
	while(_pRc->flag()>RunControl::runEnd) {
	  arena.initialise(RcdHeader::configurationStart);
	  rw.record(arena);
	  ro.record(arena);
	  
	  // Access the DaqConfigurationStart to get slow readout information
	  std::vector<const DaqConfigurationStart*> v(accessor.extract<DaqConfigurationStart>());
	  assert(v.size()==1);
	  const double slwDt(v[0]->minimumTimeBeforeSlowReadout().deltaTime());
	  
	  // If last configuration was stopped using the flag, allow the next
	  if(_pRc->flag()==RunControl::configurationEnd) _pRc->flag(RunControl::daqContinue);
	  //while(_pRc->flag()==RunControl::2) sleep(1); //???
	  

	  // Loop over acquisitions and slow readouts
	  while(_pRc->flag()>RunControl::configurationEnd) {
	    
	    // If slow run, then no acquisitions, so sleep for a short while
	    // Cannot use interupts because of driver so don't sleep for total time
	    if(runType.majorType()==DaqRunType::slow) {
	      sleep(1);
	      
	    // Standard run
	    } else {
	      
	      arena.initialise(RcdHeader::acquisitionStart);
	      rw.record(arena);
	      ro.record(arena);
	      
	      // If last acquisition was stopped using the flag, allow the next
	      if(_pRc->flag()==RunControl::acquisitionEnd) _pRc->flag(RunControl::daqContinue);
	      //while(_pRc->flag()==RunControl::3) sleep(1); //???
	      

	      // Switch depending on spill state machine mode; buffer mode
	      if(runType.bufferRun()) {
		
		// Initialise event counter
		unsigned iEvt(0);

		// Switch if spill mode
		if(runType.spillRun()) {
		  arena.initialise(RcdHeader::spill);
		  rw.record(arena);
		  ro.record(arena);
		  
		  // Access the DaqSpillEnd
		  std::vector<const DaqSpillEnd*> v(accessor.extract<DaqSpillEnd>());
		  assert(v.size()==1);
		  
		  // Get information on number of events to be read out
		  iEvt=v[0]->actualNumberOfEventsInSpill();
		  

		// Spill-with-triggers mode
		} else {

		  if(_pRc->flag()==RunControl::spillEnd) _pRc->flag(RunControl::daqContinue);

		  // Take a first trigger to get spill status
		  arena.initialise(RcdHeader::trigger);
		  rw.record(arena);
		  ro.record(arena);

		  iEvt++;

		  // Loop over triggers and events until spill signal is seen
		  while(_pRc->flag()>RunControl::spillStart) {
		    arena.initialise(RcdHeader::event);
		    rw.record(arena);
		    ro.record(arena);

		    arena.initialise(RcdHeader::trigger);
		    rw.record(arena);
		    ro.record(arena);
		  }      

		  arena.initialise(RcdHeader::spillStart);
		  rw.record(arena);
		  ro.record(arena);

		  if(_pRc->flag()==RunControl::spillStart) _pRc->flag(RunControl::daqContinue);

		  // Loop over triggers
		  while(_pRc->flag()>RunControl::spillEnd) {
		    arena.initialise(RcdHeader::trigger);
		    rw.record(arena);
		    ro.record(arena);

		    // Count number of events to be read out
		    iEvt++;
		  }
		  
		  arena.initialise(RcdHeader::spillEnd);
		  rw.record(arena);
		  ro.record(arena);
		}
		
		// Switch if transfer mode
		if(runType.transferRun()) {
		  arena.initialise(RcdHeader::transfer);
		  rw.record(arena);
		  ro.record(arena);
		  

		// Transfer-with-event run
		} else {
		  arena.initialise(RcdHeader::transferStart);
		  rw.record(arena);
		  ro.record(arena);
		  
		  // Must always read out events if triggered
		  // so do not check RunControl flag here
		  for(unsigned jEvt(0);jEvt<iEvt;jEvt++) {
		    arena.initialise(RcdHeader::event);
		    rw.record(arena);
		    ro.record(arena);
		  }
		  
		  arena.initialise(RcdHeader::transferEnd);
		  rw.record(arena);
		  ro.record(arena);
		}
		
	      // Single event mode
	      } else {
		
		// Loop over triggers and events
		while(_pRc->flag()>RunControl::acquisitionEnd) {
		  arena.initialise(RcdHeader::trigger);
		  rw.record(arena);
		  ro.record(arena);
		  
		  arena.initialise(RcdHeader::event);
		  rw.record(arena);
		  ro.record(arena);
		}      
	      }
	      
	      // Do acquisition end
	      arena.initialise(RcdHeader::acquisitionEnd);
	      rw.record(arena);
	      ro.record(arena);
	    }
	    
	    // Do a slow readout if enough time since last
	    if((time(0)-lastSlowRecord)>=slwDt) {
	      arena.initialise(RcdHeader::slowReadout);
	      rw.record(arena);
	      ro.record(arena);
	      
	      lastSlowRecord=arena.recordTime().seconds();
	    }
	  }
	
	  arena.initialise(RcdHeader::configurationEnd);
	  rw.record(arena);
	  ro.record(arena);
	}
	
	arena.initialise(RcdHeader::runEnd);
	rw.record(arena);
	ro.record(arena);
	
	// Ensure if run aborted that slow files have different timestamps
	sleep(1);
      }
    }
    
    // Clear up memory space
    delete pArena;
    
    return true;
  }
  

private:
  ShmObject<RunControl> _shmRunControl;
  RunControl *_pRc;
};

#endif
