#ifndef TrgConfiguration_HH
#define TrgConfiguration_HH

#include <iostream>
#include <fstream>

// dual/inc/daq
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqSpillStart.hh"
#include "DaqRunType.hh"

// dual/inc/rcd
#include "RcdUserRW.hh"

// dual/inc/sub
#include "SubInserter.hh"
#include "SubAccessor.hh"


class TrgConfiguration : public RcdUserRW {

public:
  TrgConfiguration(unsigned char c, unsigned char s) :
    _location(c,s,CrcLocation::beTrg,1) {
  }

  virtual ~TrgConfiguration() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << std::endl << "TrgConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    // Check record type
    switch (r.recordType()) {


    // Run start 
    case RcdHeader::runStart: {

      // Access the DaqRunStart
      SubAccessor accessor(r);
      std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
      assert(v.size()==1);
      if(doPrint(r.recordType(),1)) v[0]->print(std::cout," ") << std::endl;

      _runType=v[0]->runType();

      break;
    }


    // Configuration start is used to set up system
    case RcdHeader::configurationStart: {

      // Access the DaqConfigurationStart
      SubAccessor accessor(r);
      std::vector<const DaqConfigurationStart*>
	v(accessor.extract<DaqConfigurationStart>());
      assert(v.size()==1);
      if(doPrint(r.recordType(),1)) v[0]->print(std::cout," ") << std::endl;

      // Define vectors for configuration data
      std::vector<TrgReadoutConfigurationData> vTrd;
      std::vector< CrcLocationData<CrcBeTrgConfigurationData> > vTcd;

      // Get the configuration number and write default for first configuration
      const unsigned iCfg(v[0]->configurationNumberInRun());

      if(iCfg==0) {
	vTrd.push_back(TrgReadoutConfigurationData());
	vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_location));
      }

      // Now set up depending on run type
      //const DaqRunType::Type t(_runType.type());
      //const DaqRunType::MajorType m(_runType.majorType());
      //const unsigned char v(_runType.version());
      const UtlPack u(_runType.version());

      // Turn off trigger for several major run types
      switch(_runType.majorType()) {
      case DaqRunType::daq:
      case DaqRunType::crc:
      case DaqRunType::slow: {
	if(iCfg==0) {
	  vTrd[0].enable(false);
	  vTcd.clear();
	}
	break;
      }
      default: {
	break;
      }
      }; // switch(majorType)


      // Set trigger for other run types
      switch(_runType.type()) {

      case DaqRunType::trgTest: {
	break;
      }
      case DaqRunType::trgReadout: {
	if(iCfg==0) {
	  vTrd[0].enable(true);
	  vTrd[0].clearBeTrgTrigger(true);
	  vTrd[0].beTrgSoftTrigger((u.word()%2)==1);
	  vTrd[0].readPeriod(1);
	  vTrd[0].beTrgSquirt(((u.word()/2)%2)==1);
	}
	break;
      }
      case DaqRunType::trgParameters: {
	if(iCfg==0) {
	  vTrd[0].enable(true);
	  vTrd[0].clearBeTrgTrigger(true);
	  vTrd[0].beTrgSoftTrigger((u.word()%2)==0);
	  vTrd[0].readPeriod(1);
	  vTrd[0].beTrgSquirt(false);
	}
	break;
      }
      case DaqRunType::trgNoise: {
	if(iCfg==0) {
	  vTrd[0].enable(true);
	  vTcd[0].data()->inputEnable(1<<24); // Trg osc
	  vTcd[0].data()->oscillationPeriod(4000); // 0.0001 sec
	}
	break;
      }
      case DaqRunType::trgSpill: {
	if(iCfg==0) {
	  vTrd[0].beTrgSquirt(true);
	  vTrd[0].beTrgPollNumber(10000);
	  vTrd[0].beTrgSpillNumber(500);
	  vTrd[0].beTrgSpillTime(UtlTimeDifference(1,0));
	  vTrd[0].enable(true);
	  vTrd[0].clearBeTrgTrigger(false);
	  vTrd[0].readPeriod(1);
	  
	  vTcd[0].data()->busyTimeout(8000); // 0.0002 sec
	  vTcd[0].data()->inputEnable(1<<24); // Trg osc
	  vTcd[0].data()->oscillationPeriod(4000); // 0.0001 sec
	}
	break;
      }

      case DaqRunType::cosmicsNoise: {
	if(iCfg==0) {
	  vTrd[0].enable(true);
	  vTcd[0].data()->inputEnable(1<<24); // Trg osc
	  vTcd[0].data()->oscillationPeriod(4000); // 0.0001 sec
	}
	break;
      }

      default: {
	break;
      }
      }; // switch(type)
	
      /*
      case DaqRunType::crc: {
	trd->enable(false);

	switch(s) {
	case DaqRunType::crcTest: {
	  break;
	}
	case DaqRunType::crcNoise: {
      
	  trd->beTrgSquirt(true);
	  trd->beTrgPollNumber(10000);
	  trd->beTrgSpillNumber(500);

	  tcd->data()->busyTimeout(10000);

	  if(u.halfByte(1)<3) {
	    trd->enable(true);
	    trd->clearBeTrgTrigger(true);
	    trd->readPeriod(1);
	    tcd->data()->generalEnable(1); // Bit 0 enables J0 fanout

	    if(u.halfByte(1)==0) {
	      trd->beTrgSoftTrigger(false);
	      tcd->data()->inputEnable(1<<24); // Trg osc
	    //tcd->data()->oscillationPeriod(4000); // 0.0001 sec
	      tcd->data()->oscillationPeriod(40000); // 0.001 sec
	    }
	    if(u.halfByte(1)==1) {
	      trd->beTrgSoftTrigger(false);
	      tcd->data()->inputEnable(0xffff);
	    }
	    if(u.halfByte(1)==2) {
	      trd->beTrgSoftTrigger(true);
	      tcd->data()->inputEnable(0);
	    }

	  } else {
	    trd->enable(false);
	    trd->clearBeTrgTrigger(false);
	    trd->readPeriod(0);
	    tcd->data()->generalEnable(0);
	  }
	  
	  break;
	}
	case DaqRunType::crcIntDac: {
	  trd->enable(false);
	  break;
	}
	case DaqRunType::crcExtDac: {
	  trd->enable(false);
	  break;
	}

	default: {
	  break;
	}
	};
	break;
      }
      */

      // Load configuration into record
      SubInserter inserter(r);

      if(doPrint(r.recordType(),1)) std::cout 
	<< " Number of TrgReadoutConfigurationData subrecords inserted = "
	<< vTrd.size() << std::endl;
      assert(vTrd.size()<=1);

      for(unsigned i(0);i<vTrd.size();i++) {
	if(doPrint(r.recordType(),1)) vTrd[i].print(std::cout,"  ") << std::endl;
	inserter.insert<TrgReadoutConfigurationData>(vTrd[i]);
      }

      if(doPrint(r.recordType(),1)) std::cout 
	<< " Number of CrcBeTrgConfigurationData subrecords inserted = "
	<< vTcd.size() << std::endl;
      assert(vTcd.size()<=1);

      for(unsigned i(0);i<vTcd.size();i++) {
	if(doPrint(r.recordType(),1)) vTcd[i].print(std::cout,"  ") << std::endl;
	inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >(vTcd[i]);
      }

      break;
    }

    default: {

      break;
    }
    }; // switch(recordtype)

    return true;
  }

protected:
  const CrcLocation _location;
  DaqRunType _runType;
};

#endif
