#include "runnerDefine.icc"

#include <signal.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <unistd.h>

#include <iostream>

#include "UtlArguments.hh"
#include "RcdArena.hh"
#include "RcdIoSkt.hh"
#include "EmcReadout.hh"
#include "AhcReadout.hh"
#include "TrgReadout.hh"
#include "BmlLc1176Readout.hh"

#include "RcdMultiUserRW.hh"
#include "RunLock.hh"

#define SKT_READ_AHEAD

using namespace std;


int main(int argc, const char **argv) {
  
  unsigned eTime(CALICE_DAQ_TIME);
  cout << argv[0] << " compiled at " << ctime((const time_t*)&eTime);
  
  UtlArguments argh(argc,argv);
  
  const unsigned pciCard(argh.optionArgument('p',0,"PCI card"));
  
  if(argh.help()) return 0;
  
  RunLock lock(argv[0]);

  cout << "PCI card set to " << pciCard << endl;
  
  RcdIoSkt s;

  RcdMultiUserRW vRrw;

  // Add trigger readout module
#ifdef TRG_PCI
  TrgReadout   trgr(TRG_PCI,TRG_CRATE,TRG_SLOT);
  vRrw.addUser(trgr);
#endif

  // Add ECAL and AHCAL readout modules
#ifdef EMC_PCI
  EmcReadout er(EMC_PCI);
  vRrw.addUser(er);
#endif

#ifdef AHC_PCI
  AhcReadout ar(AHC_PCI);
  //CrcReadout ar(0,0xac);
  vRrw.addUser(ar);
#endif

#ifdef BML_LC1176_PCI
  BmlLc1176Readout bl(BML_LC1176_PCI,BML_LC1176_VME);
  vRrw.addUser(bl);
#endif

  RcdArena r;
  r.initialise(RcdHeader::startUp);
  
  RcdArena ahead;
  ahead.initialise(RcdHeader::event);
  bool inSpill(false),inTransfer(false);
  bool eventAhead(false);
  
  //assert(s.open(1124+pciCard));
  assert(s.open(1124));
  while(r.recordType()!=RcdHeader::shutDown) {
    
    // Read next record from socket
    assert(s.read(r));
    
    // Check for startUp record
    if(r.recordType()==RcdHeader::startUp) {
      
      // Put in software information
      SubInserter inserter(r);
      DaqSoftware *dsw(inserter.insert<DaqSoftware>(true));
      dsw->message(argh.command());
      dsw->setVersions();
      dsw->print(std::cout);
    }
    
    // Check for runStart record
    if(r.recordType()==RcdHeader::runStart) {

      // Access the DaqRunStart to get print level
      SubAccessor accessor(r);
      std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
      if(v.size()>0)  vRrw.printLevel(v[0]->runType().printLevel());
    }

    // Check for spillStart/End records
#ifdef SKT_READ_AHEAD
    if(r.recordType()==RcdHeader::spillStart) inSpill=true;
    if(r.recordType()==RcdHeader::spillEnd)   inSpill=false;

    // Check for transferStart/End records
    if(r.recordType()==RcdHeader::transferStart) inTransfer=true;
    if(r.recordType()==RcdHeader::transferEnd)   inTransfer=false;

    // Check if have an event already read
    if(eventAhead) {
      if(r.recordType()!=RcdHeader::event) {
	if(r.recordType()!=RcdHeader::transferEnd) {
	  std::cerr << "Event read ahead but next record is not event" << std::endl;
	  r.RcdHeader::print(std::cerr," AHEAD ") << std::endl;
	}
      } else {
	r.extend(ahead.numberOfWords(),ahead.data());	
      }
    }
#endif

    // Process record and send back
    assert(vRrw.record(r));
    assert(s.write(r));

    // Check for possibility of readahead, i.e. predict next is event
#ifdef SKT_READ_AHEAD
    eventAhead=false;
    if((!inSpill && r.recordType()==RcdHeader::trigger) || inTransfer) {
      eventAhead=true;
      ahead.initialise(RcdHeader::event);
      assert(vRrw.record(ahead));
    }
#endif
  }

  assert(s.close());

  return 0;
}
