#ifndef AhcSlowReadoutData_HH
#define AhcSlowReadoutData_HH

#include <string>
#include <iostream>

class AhcModuleSlowReadoutData;

class AhcSlowReadoutData {

public:
  enum {
    versionNumber=0
  };

  AhcSlowReadoutData();

  bool parse(std::string r);
  
  time_t timeStamp() const;
  void   timeStamp(time_t t);

  unsigned numberOfModules() const;
  void     numberOfModules(unsigned n);
  
  const AhcModuleSlowReadoutData* moduleData(unsigned i) const;
        AhcModuleSlowReadoutData* moduleData(unsigned i);

  std::ostream& print(std::ostream &o, std::string s="") const;


private:
  time_t _timeStamp;
  unsigned _numberOfModules;
};


#ifdef CALICE_DAQ_ICC

#include <time.h>
#include <cstring>

#include "AhcModuleSlowReadoutData.hh"


AhcSlowReadoutData::AhcSlowReadoutData() {
  memset(this,0,sizeof(AhcSlowReadoutData));
}

bool AhcSlowReadoutData::parse(std::string r) {
  std::istringstream sin(r);
  sin >> _timeStamp;
  if(!sin) return false;

  _numberOfModules=0;
  unsigned x(0);
  sin >> x;

  while(sin) {
    _numberOfModules++;

    AhcModuleSlowReadoutData *p(moduleData(_numberOfModules-1));
    p->moduleNumber(x);

    for(unsigned i(0);i<11;i++) {
      sin >> x;
      if(!sin) return false;
      p->temperature(i,x);
    }

    for(unsigned i(0);i<2;i++) {
      sin >> x;
      if(!sin) return false;
      p->highVoltage(i,x);
    }

    for(unsigned i(0);i<5;i++) {
      sin >> x;
      if(!sin) return false;
      p->current(i,x);
    }

    sin >> x;
  }

  return true;
}

time_t AhcSlowReadoutData::timeStamp() const {
  return _timeStamp;
}

void AhcSlowReadoutData::timeStamp(time_t t) {
  _timeStamp=t;
}

unsigned AhcSlowReadoutData::numberOfModules() const {
  return _numberOfModules;
  }

void AhcSlowReadoutData::numberOfModules(unsigned n) {
  _numberOfModules=n;
}

const AhcModuleSlowReadoutData* AhcSlowReadoutData::moduleData(unsigned i) const {
  assert(i<_numberOfModules);
  return ((const AhcModuleSlowReadoutData*)(this+1))+i;
}

AhcModuleSlowReadoutData* AhcSlowReadoutData::moduleData(unsigned i) {
  assert(i<_numberOfModules);
  return ((AhcModuleSlowReadoutData*)(this+1))+i;
}

std::ostream& AhcSlowReadoutData::print(std::ostream &o, std::string s) const {
  o << s << "AhcSlowReadoutData::print()" << std::endl;
  o << s << " Timestamp = " << _timeStamp << " = " << ctime(&_timeStamp);
  o << s << " Number of modules = " << _numberOfModules << std::endl;

  for(unsigned i(0);i<_numberOfModules;i++) {
    o << s << " Module " << std::setw(2) << i << " ";
    moduleData(i)->print(o,s+" ");
  }

  return o;
}

#endif
#endif
