#ifndef AhcConfiguration_HH
#define AhcConfiguration_HH

#include <iostream>
#include <sstream>
#include <string>
#include <fstream>

// dual/inc/rcd
#include "RcdHeader.hh"
#include "RcdUserRW.hh"

// dual/inc/sub
#include "SubInserter.hh"
#include "SubAccessor.hh"
//#include "SubModifier.hh"

// online/inc/crc
#include "CrcConfiguration.hh"
#include "CrcReadoutConfigurationData.hh"
#include "AhcVfeSlowControlsFile.hh"

// daquser/inc/ahc
#include "AhcConfigReader.hh"

class AhcConfiguration : public CrcConfiguration {

public:
  AhcConfiguration() : CrcConfiguration(0xac), _holdEndEdge(0), _holdCmAsic(11), _holdPmAsic(55), _holdCmSipm(32), _holdPmSipm(60), _holdBeamSipm(12), _holdPin(0) {
    for (unsigned fe(0);fe<8;fe++) config[fe] = 0;
    _readoutConfiguration.crateNumber(0xac);
  }


  virtual ~AhcConfiguration() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "AhcConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
    
    // Do underlying CRC configuration
    assert(CrcConfiguration::record(r));
    
    // Check record type
    switch (r.recordType()) {
      
      // Start up
    case RcdHeader::startUp: {
      assert(readDacValues(r));
      assert(ahcDacConfiguration(r));
      assert(readStagePositions(r));
      break;
    }

      
      // Run start 
    case RcdHeader::runStart: {

      // Enable/disable CRCs here
      //for(unsigned s(2);s<=21;s++) _readoutConfiguration.slotEnable(s,false);


       if(doPrint(r.recordType(),1))
	_readoutConfiguration.print(std::cout," ") << std::endl;

      break;

      
    }
      
      
      // Configuration start 
    case RcdHeader::configurationStart: {

     
      // expert run: read in from .cfg files
      if(_runType.type()==DaqRunType::ahcExpert) {
	
	//	for(unsigned s(2);s<=21;s++) _readoutConfiguration.slotEnable(s,false);
	//	_readoutConfiguration.slotEnable(12,true);        // only enable slot 12 = HCAL

	char configFilename[128];
	for (unsigned fe(0);fe<8;fe++) 
	  {
	    if(config[fe]!=0) delete config[fe];
	    sprintf(configFilename,"AhcExpertFE%d.cfg",fe); // hardcoded one fe file readin
	    config[fe] = new AhcConfigReader(configFilename,6);
	    //	    if (config[fe]->getHoldStart()+config[fe]->getHoldWidth() > _holdEndEdge) _holdEndEdge = config[fe]->getHoldStart()+config[fe]->getHoldWidth();
	  }
      }
      //else {
	_holdEndEdge = 0x4000; // standard value from CrcFeConfiguration.hh
	//}
      //  _holdEndEdge = 0xFA00;

      if(_runType.type()==DaqRunType::ahcDacScan) assert(ahcDacConfiguration(r));
      if(_runType.type()==DaqRunType::ahcTest) assert(ahcDacConfiguration(r));

      // Handle the ones which need work
      if(_runType.majorType()==DaqRunType::ahc ||
	 _runType.majorType()==DaqRunType::beam ||
	 _runType.majorType()==DaqRunType::cosmics) {

	ahcReadoutConfiguration(r);
	ahcBeConfiguration(r);
	ahcBeTrgConfiguration(r);
	ahcFeConfiguration(r);
	ahcVfeConfiguration(r);
	if(_configurationNumber%3==0) ahcSlowConfiguration(r);
      }
      
     
      break;
    }
      
      /*
    case RcdHeader::slowConfiguration: {
      if( (_runType.type()==DaqRunType::ahcBeamStage) || (_runType.type()==DaqRunType::ahcBeamStageScan) ) {
	  std::cout << "STAGE RUN" << std::endl;
	  ahcSlowConfiguration(r);
	}

     
      break;
    }
      */
      
      // Run end
    case RcdHeader::runEnd: {
      if(_runType.type()==DaqRunType::ahcDacScan) assert(ahcDacConfiguration(r));
      if(_runType.type()==DaqRunType::ahcTest) assert(ahcDacConfiguration(r));
      break;
    }

      // Shut down
    case RcdHeader::shutDown: {
      assert(ahcDacConfiguration(r));
      break;
    }
      
    default: {
      break;
    }
    };
    
    return true;
  }
  
  
  virtual bool readDacValues(const RcdRecord &r) {

    // NEED TO GENERALISE TO MORE THAN JUST SLOT 12
    //_location.slotNumber(12);

    // Clear vector for DAC data
    _vDac.clear();

    // Disable all slots initially
    for(unsigned i(2);i<=21;i++) _readoutConfiguration.slotEnable(i,false);
    
    // count number of PIN readout configured frontends
    nPin=0;
 
    // Set crate number
    _readoutConfiguration.crateNumber(0xac);

    // Enable trigger slot but no FEs except FE0 (for fast trigger readout)
    if(_trgLocation.slotNumber()>0) {
      _readoutConfiguration.slotEnable(_trgLocation.slotNumber(),true);
      _readoutConfiguration.slotFeEnable(_trgLocation.slotNumber(),0,true);
    }
    //_readoutConfiguration.slotEnable(12,true);

    // Beni's code to read in DAC values
    std::ifstream configFile;
    char line[512];
    
    std::string* detectorType[21][8];
    std::string* dacFileName[21][8];
    for (int slot =0;slot<21;slot++)
      {
	for (int fe =0;fe<8;fe++) detectorType[slot][fe] = new std::string("");
	for (int fe =0;fe<8;fe++) dacFileName[slot][fe] = new std::string("");
      }
    std::cout << "loading of DAC values" << std::endl;
    
    configFile.open("DAC.cfg");
    while (configFile.good())
      {
	//configFile >> buffer;
	configFile.getline(line,sizeof(line));
	int _SLOT(0);
	int _FE(0);
	std::string dummy1;
	std::string dummy2;

	if (line[0] != '#')
	  {
	    // std::istringstream buffer(line, std::stringstream::in || std::stringstream::out );
	    std::istringstream buffer(line);
	    //buffer.str(line);
	    //std::stringstream buffer(std::stringstream::in || std::stringstream::out ); 
	    //buffer << std::string(line) << std::endl;
	    std::cout << "line: "<< line << std::endl;
	    //std::cout << "buffer: "<< buffer.str() << std::endl;
	    std::cout << "---"<< std::endl;

	    buffer >> _SLOT;
	    //      std::cout << _SLOT << std::endl;
	    
	    buffer >> _FE;
	    //      std::cout << _FE << std::endl;

	    buffer >> dummy1;
	    *(detectorType[_SLOT][_FE]) = dummy1;
	    //      std::cout << dummy1 << std::endl;
	    
	    if (dummy1 != "PIN")
	      {
		buffer >> dummy2;
		*(dacFileName[_SLOT][_FE]) = dummy2;
		//	std::cout << dummy2 << std::endl;
	      }

	    std::cout << "DAC.cfg read in" << std::endl;
	    std::cout << _SLOT <<" "<< _FE <<" "<< *(detectorType[_SLOT][_FE]) <<" "<< *(dacFileName[_SLOT][_FE]) << std::endl;
	
	  }
      }

    // Look for FEs with a file and hence DAC values
    for(unsigned slot(5);slot<21;slot++) {
      for(unsigned f(0);f<8;f++) {
	if(*(detectorType[slot][f])=="AHCAL" || *(detectorType[slot][f])=="TCMT") {
	  if(*(dacFileName[slot][f])!="") {
	    AhcVfeStartUpData data;
	    AhcVfeSlowControlsFile file;
	    if(file.read(*(dacFileName[slot][f]),data,f)) {

	      // If loading DAC, also enable slot & FE
	      _readoutConfiguration.slotEnable(slot,true);
	      _readoutConfiguration.slotFeEnable(slot,f,true);
	      _location.slotNumber(slot);

	      // Now put DAC values into vector
	      if(f==0) _location.crcComponent(CrcLocation::fe0);
	      if(f==1) _location.crcComponent(CrcLocation::fe1);
	      if(f==2) _location.crcComponent(CrcLocation::fe2);
	      if(f==3) _location.crcComponent(CrcLocation::fe3);
	      if(f==4) _location.crcComponent(CrcLocation::fe4);
	      if(f==5) _location.crcComponent(CrcLocation::fe5);
	      if(f==6) _location.crcComponent(CrcLocation::fe6);
	      if(f==7) _location.crcComponent(CrcLocation::fe7);
	      _vDac.push_back(CrcLocationData<AhcVfeStartUpData>(_location,data));
	    }
	  }
	} // end of AHCAL module setting
	

	if(*(detectorType[slot][f])=="PIN") {
	  
	  if (nPin<maxPin)
	    {
	      _readoutConfiguration.slotFeEnable(slot,f,true);
	      
	      _pinLocation[nPin].crateNumber(0xac);
	      _pinLocation[nPin].slotNumber(slot);
	      
	      if(f==0) _pinLocation[nPin].crcComponent(CrcLocation::fe0);
	      if(f==1) _pinLocation[nPin].crcComponent(CrcLocation::fe1);
	      if(f==2) _pinLocation[nPin].crcComponent(CrcLocation::fe2);
	      if(f==3) _pinLocation[nPin].crcComponent(CrcLocation::fe3);
	      if(f==4) _pinLocation[nPin].crcComponent(CrcLocation::fe4);
	      if(f==5) _pinLocation[nPin].crcComponent(CrcLocation::fe5);
	      if(f==6) _pinLocation[nPin].crcComponent(CrcLocation::fe6);
	      if(f==7) _pinLocation[nPin].crcComponent(CrcLocation::fe7);
	      
	      _readoutConfiguration.slotFeEnable(_pinLocation[nPin].slotNumber(),
						 _pinLocation[nPin].crcComponent(),true);
	      
	      // max two frontends are configured for the PIN
	      nPin++;
	      std::cout <<"There are "<< nPin <<" frontend(s) for Pin readout configured" << std::endl;
	    }
	  else
	    {
	      std::cout << "There are more than two frontends configured in the DAC.cfg file for PIN" << std::endl;
	    }
	}

	if(*(detectorType[slot][f])=="TCMT") {
	  // at the moment no special treatment! same as AHCAL! see L217 and if needed change it!

	}

	

      }
    }

   
    if(doPrint(r.recordType(),1))
      _readoutConfiguration.print(std::cout," ") << std::endl;
    
    return true;
  }


  virtual bool ahcDacConfiguration(RcdRecord &r) {

    const unsigned char v(_runType.version());  

    SubInserter inserter(r);
    
    if(doPrint(r.recordType(),1)) std::cout
      << " Number of AhcVfeStartUpData subrecords inserted = "
      << _vDac.size() << std::endl << std::endl;
    
    for(unsigned i(0);i<_vDac.size();i++) {
      CrcLocationData<AhcVfeStartUpData> data(_vDac[i]);

      if(_runType.type()==DaqRunType::ahcDacScan) {
	if(_configurationNumber>0) {

	  for(unsigned i(0);i<12;i++) {
	    for(unsigned j(0);j<18;j++) {
	      int dac(data.data()->dac(i,j));
	      //if((_configurationNumber%2)==1) { // Increment
		dac+=(_configurationNumber+1)/2;
		if(dac>255) dac=255;
		data.data()->dac(i,j,dac);
		/*
	      } else {                          // Decrement
		dac-=(_configurationNumber+1)/2;
		if(dac<0) dac=0;
		data.data()->dac(i,j,dac);
	      }
		*/
	    }
	  }
	}
      }

      if(_runType.type()==DaqRunType::ahcTest) {
        if(_configurationNumber>0) {
	  
          for(unsigned i(0);i<12;i++) {
            for(unsigned j(0);j<18;j++) {
              int dac;
              dac+=256/(v+1);
            }
          }
        }
      }

      if(doPrint(r.recordType(),1)) data.print(std::cout,"  ") << std::endl;
      inserter.insert< CrcLocationData<AhcVfeStartUpData> >(data);
    }

    return true;
  }


  virtual bool readStagePositions(const RcdRecord &r) {

    char* _STAGEfile="StagePositions.dat";
    string _strBuffer;
    char _charBuffer[128];
    int _STAGExposition;
    int _STAGEyposition;
    unsigned _rochip;
    unsigned _rochan;

    ifstream _stageFile;
    

    std::cout << "trying to load STAGE positions from " << _STAGEfile <<endl;
    if (_STAGEfile == "") 
      { 
	cout << "syntax error: emtpy STAGE file string. In STAGE mode multi you must supply a valid STAGEfile value" << endl;
	exit(1);
      }

    _stageFile.open(_STAGEfile);

    if (!_stageFile.good() || !_stageFile.is_open()) 
      {
	cout << "file error: error while opening "<< _STAGEfile << " please check if file exist and is readable" << endl;	  
	exit(1);
      }

    unsigned STAGEentries(0);
    unsigned connector;
    int xmovement, ymovement;
    bool dataStart(false);
    while (_stageFile.good())
      {
	if (!dataStart)
	  {
	    _stageFile >> _strBuffer;
	      
	    if (_strBuffer=="#")
	      {
		_stageFile.getline(_charBuffer,sizeof(_charBuffer));
		std::cout << "comment: "<< _charBuffer  << endl;
	      }
	    if (_strBuffer=="--data--") dataStart = true;
	      
	  }
	else
	  {
	    _stageFile >> _rochip >> _rochan >> connector >> _STAGExposition >> _STAGEyposition >> xmovement >> ymovement;
	    if ( (_STAGExposition > 0) && (_STAGExposition < 800) && (_STAGEyposition > 0) && (_STAGEyposition < 800) )  
	      {
		_STAGEchipArray.push_back(_rochip);
		_STAGEchanArray.push_back(_rochan);
		_STAGExArray.push_back(xmovement);
		_STAGEyArray.push_back(ymovement);
	      }
	    else 
	      {
		cout << "-- range error -- chip: " << _rochip << "  channel: " << _rochan << "  STAGE x-position: "<< _STAGExposition << "  STAGE y-position: "<< _STAGEyposition   <<endl;
		exit(1);
	      }
	    STAGEentries++;
	    std::cout << "  chip: " << _rochip << "  channel: " << _rochan << "  STAGE x-position: "<< _STAGExposition << "  STAGE y-position: "<< _STAGEyposition   <<endl;
	  }
      }
    _stageFile.close();
    cout << "found " << STAGEentries << " STAGE setting entries in file" << endl;
    if (_STAGExArray.size() != _STAGEyArray.size()) 
      {
	cout << " different number of x and y positions check position file  " << endl;
	exit(1);
      }
      
 
    


    if(doPrint(r.recordType(),1))
      _readoutConfiguration.print(std::cout," ") << std::endl;
    
    return true;
  }






  virtual bool ahcReadoutConfiguration(RcdRecord &r) {

    SubInserter inserter(r);
    CrcReadoutConfigurationData
      *b(inserter.insert<CrcReadoutConfigurationData>(true));
    *b=_readoutConfiguration;
    
    // Now do the readout periods, soft triggers and modes
    const unsigned char v(_runType.version());

    switch(_runType.type()) {

    case DaqRunType::ahcTest: {
      b->beSoftTrigger(true);
      b->vlinkBlt(true);
      break;
    }
      
    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcPmNoise: {
     if((v%4)==1) {
        b->beSoftTrigger(true);
      }
      if((v%4)==2) {
        b->feBroadcastSoftTrigger(true);
      }
      if((v%4)==3) {
        b->feSoftTrigger(0,true);
        b->feSoftTrigger(1,true);
        b->feSoftTrigger(2,true);
        b->feSoftTrigger(3,true);
        b->feSoftTrigger(4,true);
        b->feSoftTrigger(5,true);
        b->feSoftTrigger(6,true);
        b->feSoftTrigger(7,true);
      }

      b->vlinkBlt(((v/4)%2)!=1);

      b->vmePeriod(v/8);
      b->bePeriod(v/8);
      b->fePeriod(v/8);
      break;
    }

    case DaqRunType::ahcAnalogOut:
    case DaqRunType::ahcDacScan:
    case DaqRunType::ahcCmAsic:
    case DaqRunType::ahcCmAsicVcalibScan:
    case DaqRunType::ahcCmAsicHoldScan:
    case DaqRunType::ahcPmAsic:
    case DaqRunType::ahcPmAsicVcalibScan:
    case DaqRunType::ahcPmAsicHoldScan:
    case DaqRunType::ahcCmLed:
    case DaqRunType::ahcCmLedVcalibScan:
    case DaqRunType::ahcCmLedHoldScan:
    case DaqRunType::ahcPmLed:
    case DaqRunType::ahcPmLedVcalibScan:
    case DaqRunType::ahcPmLedHoldScan: {
      b->beSoftTrigger(true);
      b->vlinkBlt(true);
      break;
    }

    case DaqRunType::ahcBeam:
      //case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:
    case DaqRunType::ahcCosmicsHoldScan:

    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:

    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {
      if((_configurationNumber%3)<2) {
	//b->beSoftTrigger(true);
	b->becPeriod(1);
	b->bePeriod(8);
	b->fePeriod(1);
      } else {
	b->becPeriod(1);
	b->bePeriod(16);
	b->fePeriod(8);
	//b->bePeriod(1); // TEMP
	//b->fePeriod(1); // TEMP
      }
      b->vlinkBlt(true);
      break;
    }

    case DaqRunType::ahcBeamHoldScan:{
      b->becPeriod(16);
      b->fePeriod(16);
      b->bePeriod(1); // TEMP
      b->fePeriod(1); // TEMP
     
      b->vlinkBlt(true);
      break;
      
    }

    case DaqRunType::ahcScintillatorHoldScan: {
      b->vlinkBlt(true);
      break;
    }

    case DaqRunType::ahcExpert: {
      if(config[0]->getTrigger()==24) //hardcoded to use triggerinformation only from FE 0 file
	{
	  b->beSoftTrigger(true);
	  b->bePeriod(1); // TEMP
	  b->fePeriod(1); // TEMP
	} else {
	  b->becPeriod(16);
	  b->fePeriod(16);
	  b->bePeriod(1); // TEMP
	  b->fePeriod(1); // TEMP
	}
      b->vlinkBlt(true);
      break;
    }

    default: {
      break;
    }
    };

    if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;

    return true;
  }

  virtual bool ahcBeConfiguration(RcdRecord &r) {

    // Define vector for configuration data
    std::vector< CrcLocationData<CrcBeConfigurationData> > vBcd;

    _location.slotBroadcast(true);
    _location.crcComponent(CrcLocation::be);
    vBcd.push_back(CrcLocationData<CrcBeConfigurationData>(_location));

    const unsigned char v(_runType.version());

    switch(_runType.type()) {
    case DaqRunType::ahcTest: {
      vBcd[0].data()->j0TriggerEnable(false);
      vBcd[0].data()->j0BypassEnable(false);
       break;
    }
      
    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcPmNoise: {
      vBcd[0].data()->j0TriggerEnable((v%4)==0);
      vBcd[0].data()->j0BypassEnable(false);
      break;
    }
      
    case DaqRunType::ahcAnalogOut:
    case DaqRunType::ahcDacScan:
    case DaqRunType::ahcCmAsic:
    case DaqRunType::ahcCmAsicVcalibScan:
    case DaqRunType::ahcCmAsicHoldScan:
    case DaqRunType::ahcCmLed:
    case DaqRunType::ahcCmLedVcalibScan:
    case DaqRunType::ahcCmLedHoldScan:
    case DaqRunType::ahcPmAsic:
    case DaqRunType::ahcPmAsicVcalibScan:
    case DaqRunType::ahcPmAsicHoldScan:
    case DaqRunType::ahcPmLed:
    case DaqRunType::ahcPmLedVcalibScan:
    case DaqRunType::ahcPmLedHoldScan: {
      vBcd[0].data()->j0TriggerEnable(false);
      vBcd[0].data()->j0BypassEnable(false);
      break;
    }
      
    case DaqRunType::ahcScintillatorHoldScan: {
      vBcd[0].data()->j0TriggerEnable(true);
      vBcd[0].data()->j0BypassEnable(false);
      break;
    }
      
    case DaqRunType::ahcBeam:
      // case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:
    case DaqRunType::ahcCosmicsHoldScan:

    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:

    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {
      //vBcd[0].data()->j0TriggerEnable((_configurationNumber%3)==2);
      vBcd[0].data()->j0TriggerEnable(true);
      vBcd[0].data()->j0BypassEnable(false);

      if(_trgLocation.slotNumber()>0) {
	_location.slotNumber(_trgLocation.slotNumber());
	vBcd.push_back(CrcLocationData<CrcBeConfigurationData>(_location));
	vBcd[1].data()->j0TriggerEnable(true);
	vBcd[1].data()->j0BypassEnable(false);
	vBcd[1].data()->trgDataFe0Enable(true);
	vBcd[1].data()->feDataEnable(1);
	vBcd[1].data()->feTriggerEnable(1);
      }
      break;
    }
      
    case DaqRunType::ahcBeamHoldScan: {
      vBcd[0].data()->j0TriggerEnable(1);
      vBcd[0].data()->j0BypassEnable(false);
      break;
    }
      

   case DaqRunType::ahcExpert: {     
     if(config[0]->getTrigger()==24) //hardcoded to use triggerinformation only from FE 0 file 
       {
	 vBcd[0].data()->j0TriggerEnable(false);
	 vBcd[0].data()->j0BypassEnable(false);
       } else {
	 vBcd[0].data()->j0TriggerEnable(true);
	 vBcd[0].data()->j0BypassEnable(false);
       }
     break;
    }

    default: {
      vBcd[0].data()->j0TriggerEnable(true);
      vBcd[0].data()->j0BypassEnable(false);
      break;
    }
    };

    // Load configuration into record
    SubInserter inserter(r);
    
    if(doPrint(r.recordType(),1)) std::cout
      << " Number of CrcBeConfigurationData subrecords inserted = "
      << vBcd.size() << std::endl << std::endl;
    
    for(unsigned i(0);i<vBcd.size();i++) {
      if(doPrint(r.recordType(),1)) vBcd[i].print(std::cout,"  ") << std::endl;
      inserter.insert< CrcLocationData<CrcBeConfigurationData> >(vBcd[i]);
    }
    
    return true;
  }
  
  virtual bool ahcBeTrgConfiguration(RcdRecord &r) {

    // Set location
    _location.crcComponent(CrcLocation::beTrg);
 
    // Define vectors for configuration data
    std::vector<TrgReadoutConfigurationData> vTrd;
    std::vector< CrcLocationData<CrcBeTrgConfigurationData> > vTcd;

    // Turn trigger off for all CRCs
    if(_configurationNumber==0) {
      _location.slotBroadcast(true);
      vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_location));
      vTcd[0].data()->inputEnable(0);
      vTcd[0].data()->generalEnable(0);
    }

    // Override if trigger slot is in this crate
    if(_trgLocation.slotNumber()>0) {
      vTrd.push_back(TrgReadoutConfigurationData());

      const unsigned char v(_runType.version());

      switch(_runType.type()) {

      case DaqRunType::ahcTest: {
	break;
      }
      
      case DaqRunType::ahcCmNoise:
      case DaqRunType::ahcPmNoise: {
	vTrd[0].beTrgSquirt(true);
	vTrd[0].readPeriod(v/8);

	if((v%4)==0) {
	  unsigned t(vTcd.size());
	  vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	  vTcd[t].data()->inputEnable(0);
	  vTcd[t].data()->generalEnable(1);

	  vTrd[0].clearBeTrgTrigger(true);
	  vTrd[0].beTrgSoftTrigger(true);
	}
	break;
      }
      
      case DaqRunType::ahcScintillatorHoldScan: {
	vTrd[0].clearBeTrgTrigger(true);
	vTrd[0].beTrgSquirt(true);
	
	unsigned t(vTcd.size());
	vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	vTcd[t].data()->generalEnable(1);
	
	vTcd[t].data()->inputEnable(1<<12);
	//vTcd[t].data()->inputEnable(0x100ffff);
	//vTcd[t].data()->inputEnable(0x3001000);
	//vTcd[t].data()->andEnable(0,0x0100);
	break;
      }

      case DaqRunType::ahcBeam:
	// case DaqRunType::ahcBeamHoldScan:
      case DaqRunType::ahcBeamStage:
      case DaqRunType::ahcBeamStageScan:
      case DaqRunType::ahcCosmics:
      case DaqRunType::ahcCosmicsHoldScan:
	
      case DaqRunType::beamTest:
      case DaqRunType::beamNoise:
      case DaqRunType::beamData:
      case DaqRunType::beamHoldScan:
	
      case DaqRunType::cosmicsTest:
      case DaqRunType::cosmicsNoise:
      case DaqRunType::cosmicsData:
      case DaqRunType::cosmicsHoldScan: {
	if((_configurationNumber%3)!=2) {
	  //vTrd[0].enable(false);
	  vTrd[0].clearBeTrgTrigger(true);
	  //vTrd[0].beTrgSoftTrigger(true);
	  vTrd[0].beTrgSquirt(false);
	  vTrd[0].readPeriod(8);
	  vTrd[0].readcPeriod(1);

	  unsigned t(vTcd.size());
	  vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	  vTcd[t].data()->generalEnable(1);
	  vTcd[t].data()->fifoIdleDepth(240);
	  vTcd[t].data()->oscillatorEnable(true);
	  vTcd[t].data()->oscillationPeriod(40000); // 1kHz
#ifdef SPILL_INPUT
	  vTcd[t].data()->inputInvert(1<<SPILL_INPUT); // Input spill signal
#endif

	} else {
	  vTrd[0].clearBeTrgTrigger(true);
	  vTrd[0].beTrgSquirt(false);
	  vTrd[0].readPeriod(16);
	  vTrd[0].readcPeriod(1);
	  //vTrd[0].readPeriod(1); // temp
	  
	  unsigned t(vTcd.size());
	  vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	  vTcd[t].data()->generalEnable(1);
	  vTcd[t].data()->fifoIdleDepth(240);

	  if(v<16) {
	    //if((v&0xc0)==0) { // Normal cosmic trigger
	    vTcd[t].data()->inputEnable(1<<v);
	    //vTcd[t].data()->inputEnable(0x0002000);
	    //vTcd[t].data()->inputEnable(0x100ffff);
	    //vTcd[t].data()->inputEnable(0x3001000);
	    //vTcd[t].data()->andEnable(0,0x0100);

	  } else if(v<31) { // use internal coincidence
	    vTcd[t].data()->inputEnable(1<<25);
	    vTcd[t].data()->generalEnable(1+(1<<8));
	    vTcd[t].data()->andEnable(0,3<<(v-16));

	  } else if(v==253) { // use internal coincidence
	    // vTcd[t].data()->inputEnable((1<<10) + (1<<11));
	    vTcd[t].data()->inputEnable(3<<10);
	    vTcd[t].data()->generalEnable(1<<8);
	    vTcd[t].data()->andEnable(0,3<<10);

	  } else if(v==254) { // Soft trigger
	    vTrd[0].beTrgSoftTrigger(true);
	    vTcd[t].data()->inputEnable(0);

	  } else if(v==255) { // Oscillator
	    vTcd[t].data()->oscillatorEnable(true);
	    vTcd[t].data()->oscillationPeriod(40000); // 1kHz
	  }
	}
	break;
      }

      case DaqRunType::ahcBeamHoldScan:{
	
	vTrd[0].clearBeTrgTrigger(true);
	vTrd[0].beTrgSquirt(true);
	vTrd[0].readPeriod(16);
	vTrd[0].readcPeriod(16);
	vTrd[0].readPeriod(1); // temp
	  
	unsigned t(vTcd.size());
	vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	vTcd[t].data()->generalEnable(1);
	
	if(v<16) {
	  //if((v&0xc0)==0) { // Normal cosmic trigger
	  vTcd[t].data()->inputEnable(1<<v);
	  //vTcd[t].data()->inputEnable(0x0002000);
	  //vTcd[t].data()->inputEnable(0x100ffff);
	  //vTcd[t].data()->inputEnable(0x3001000);
	  //vTcd[t].data()->andEnable(0,0x0100);
	} else if(v==253) { // use internal coincidence
	  vTcd[t].data()->inputEnable((1<<10) | (1<<11));
	  vTcd[t].data()->andEnable(0,(1<<10) | (1<<11));
	  //	  vTcd[t].data()->andEnable(1,1<<11);
	} else if(v==254) { // Soft trigger
	  vTrd[0].beTrgSoftTrigger(true);
	  vTcd[t].data()->inputEnable(0);
	} else if(v==255) { // Oscillator
	  vTcd[t].data()->oscillatorEnable(true);
	  vTcd[t].data()->oscillationPeriod(40000); // 1kHz
	  
	}
     
	break;

      }

	
      case DaqRunType::ahcExpert: {
	if(config[0]->getTrigger()==24) //hardcoded to use triggerinformation only from FE 0 file
	  {
	    vTrd[0].enable(false);
	    vTrd[0].readPeriod(1); // temp
	  } else {
	    vTrd[0].clearBeTrgTrigger(true);
	    vTrd[0].beTrgSquirt(true);
	    vTrd[0].readPeriod(16);
	    vTrd[0].readcPeriod(16);
	    vTrd[0].readPeriod(1); // temp
	  
	    unsigned t(vTcd.size());
	    vTcd.push_back(CrcLocationData<CrcBeTrgConfigurationData>(_trgLocation));
	    vTcd[t].data()->generalEnable(1);

	    cout<<"WE SET THE TRIGGER TO: "<<config[0]->getTrigger()<<endl;
	    vTcd[t].data()->inputEnable(1<<config[0]->getTrigger());
	    vTcd[t].data()->oscillationPeriod(40000); // 1kHz
	  }
	break;
      }


      default: {
	break;
      }
      };
    }

    // Load configuration into record
    SubInserter inserter(r);

    if(doPrint(r.recordType(),1)) std::cout
      << " Number of TrgReadoutConfigurationData subrecords inserted = "
      << vTrd.size() << std::endl << std::endl;
    assert(vTrd.size()<=1);

    for(unsigned i(0);i<vTrd.size();i++) {
      if(doPrint(r.recordType(),1)) vTrd[i].print(std::cout,"  ") << std::endl;
      inserter.insert<TrgReadoutConfigurationData>(vTrd[i]);
    }

    if(doPrint(r.recordType(),1)) std::cout
      << " Number of CrcBeTrgConfigurationData subrecords inserted = "
      << vTcd.size() << std::endl << std::endl;
    assert(vTcd.size()<=2);

    for(unsigned i(0);i<vTcd.size();i++) {
      if(doPrint(r.recordType(),1)) vTcd[i].print(std::cout,"  ") << std::endl;
      inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >(vTcd[i]);
    }

    return true;
  }
  
  virtual bool ahcFeConfiguration(RcdRecord &r) {


    if (_runType.type() == DaqRunType::ahcExpert) {
     
      SubInserter inserter(r);
      
      std::vector< CrcLocationData<CrcFeConfigurationData> > vFcd;
      
      for(unsigned f(0);f<8;f++) {

	_location.slotBroadcast(true);

	if(f==0) _location.crcComponent(CrcLocation::fe0);
	if(f==1) _location.crcComponent(CrcLocation::fe1);
	if(f==2) _location.crcComponent(CrcLocation::fe2);
	if(f==3) _location.crcComponent(CrcLocation::fe3);
	if(f==4) _location.crcComponent(CrcLocation::fe4);
	if(f==5) _location.crcComponent(CrcLocation::fe5);
	if(f==6) _location.crcComponent(CrcLocation::fe6);
	if(f==7) _location.crcComponent(CrcLocation::fe7);

	_location.label(1);

	unsigned t(vFcd.size());
	vFcd.push_back(CrcLocationData<CrcFeConfigurationData>(_location));
	
	vFcd[t].data()->calibEnable(config[f]->getTcalib());
	//vFcd[t].data()->calibWidth(config[f]->getCalibWidth());
	vFcd[t].data()->calibWidth(228);  // chance from 128 to 228 for pin debug
	
	vFcd[t].data()->dacData(CrcFeConfigurationData::boardB,config[f]->getVcalib());
	vFcd[t].data()->dacData(CrcFeConfigurationData::boardA,config[f]->getVcalib());
	
	vFcd[t].data()->holdStart(config[f]->getHoldStart());
       	//vFcd[t].data()->holdWidth(_holdEndEdge - config[f]->getHoldStart());
	//vFcd[t].data()->sequenceDelay(123);
	vFcd[t].data()->holdEnd(_holdEndEdge); 
	
	AhcVfeControl vc(0);
	vc.ledSel(false);
	
	vc.ledSel(config[f]->getLEDSel());
	vc.swHoldIn(!config[f]->getHoldEnable());
	
	if(!config[f]->getHoldEnable()) vFcd[t].data()->vfeMplexClockPulses(config[f]->getAnalogOutChan()+1);     
	vFcd[t].data()->vfeControl(vc.data());

      }

      for(unsigned i(0);i<vFcd.size();i++) {
	if(doPrint(r.recordType(),1)) vFcd[i].print(std::cout,"  ") << std::endl;
	std::cout << "insert frontend settings for fe " << i << std::endl; 
	inserter.insert< CrcLocationData<CrcFeConfigurationData> >(vFcd[i]);
      }

      std::cout << "end of FE expert configuration" << std::endl;
      return true; //ignore all the rest in expert mode
    }
    



    std::cout << "standard mode configuration" << std::endl;

    SubInserter inserter(r);
    
    CrcLocationData<CrcFeConfigurationData>
      *d(inserter.insert< CrcLocationData<CrcFeConfigurationData> >(true));

    _location.slotBroadcast(true);
    _location.crcComponent(CrcLocation::feBroadcast);
    d->location(_location);

    AhcVfeControl vc(0);
    vc.ledSel(false);

    const unsigned char v(_runType.version());

    switch(_runType.type()) {

    case DaqRunType::ahcTest: {
      //      d->data()->calibEnable(true);
      break;
    }

    case DaqRunType::ahcAnalogOut: {
      d->data()->calibEnable(true);
      vc.ledSel(true);
      vc.swHoldIn(true);

      if((v&0x80)==0) { //CM
	//d->data()->holdStart(83);
	//d->data()->dacData(CrcFeConfigurationData::boardA,20000);
	d->data()->dacData(CrcFeConfigurationData::boardB,9500);
      } else { // PM
	//d->data()->holdStart(113);
	//d->data()->dacData(CrcFeConfigurationData::boardA,25000);
	d->data()->dacData(CrcFeConfigurationData::boardB,30000);
      }
      //d->data()->calibWidth(d->data()->holdStart()+128);

      d->data()->vfeMplexClockPulses((v+1)%32);
      break;
    }

    case DaqRunType::ahcCmAsic: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdCmAsic);
      d->data()->calibWidth(d->data()->holdStart()+128);


      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,0);

      if((v&0x40)==0) d->data()->dacData(CrcFeConfigurationData::boardA,64*(v%64));
      if((v&0x80)==0) d->data()->dacData(CrcFeConfigurationData::boardB,64*(v%64));
      break;
    }

    case DaqRunType::ahcCmAsicVcalibScan: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdCmAsic);
      d->data()->calibWidth(d->data()->holdStart()+128);

      unsigned steps(v+1);
      if((_configurationNumber%2)==0) {
	d->data()->dacData(CrcFeConfigurationData::boardA,(_configurationNumber/2)*(4096/steps));
	d->data()->dacData(CrcFeConfigurationData::boardB,0);
      } else {
	d->data()->dacData(CrcFeConfigurationData::boardA,0);
	d->data()->dacData(CrcFeConfigurationData::boardB,(_configurationNumber/2)*(4096/steps));
      }
      break;
    }

    case DaqRunType::ahcCmAsicHoldScan: {
      d->data()->calibEnable(true);
      d->data()->dacData(CrcFeConfigurationData::boardA,2048);
      d->data()->dacData(CrcFeConfigurationData::boardB,2048);

      unsigned steps(v+1);
      d->data()->holdStart(_configurationNumber*(256/steps));
      d->data()->calibWidth(d->data()->holdStart()+128);
      break;
    }

    case DaqRunType::ahcPmAsic: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdPmAsic);
      d->data()->calibWidth(d->data()->holdStart()+128);

      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,0);

      if((v&0x40)==0) d->data()->dacData(CrcFeConfigurationData::boardA,128*(v%64));
      if((v&0x80)==0) d->data()->dacData(CrcFeConfigurationData::boardB,128*(v%64));
      break;
    }

    case DaqRunType::ahcPmAsicVcalibScan: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdPmAsic);
      d->data()->calibWidth(d->data()->holdStart()+128);

      unsigned steps(v+1);
      if((_configurationNumber%2)==0) {
	d->data()->dacData(CrcFeConfigurationData::boardA,(_configurationNumber/2)*(8192/steps));
	d->data()->dacData(CrcFeConfigurationData::boardB,0);
      } else {
	d->data()->dacData(CrcFeConfigurationData::boardA,0);
	d->data()->dacData(CrcFeConfigurationData::boardB,(_configurationNumber/2)*(8192/steps));
      }
      break;
    }

    case DaqRunType::ahcPmAsicHoldScan: {
      d->data()->calibEnable(true);
      d->data()->dacData(CrcFeConfigurationData::boardA,4096);
      d->data()->dacData(CrcFeConfigurationData::boardB,4096);

      unsigned steps(v+1);
      d->data()->holdStart(_configurationNumber*(512/steps));
      d->data()->calibWidth(d->data()->holdStart()+128);
      break;
    }

    case DaqRunType::ahcCmLed: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdCmSipm); 
      d->data()->calibWidth(d->data()->holdStart()+128);

      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,0);

      if((v&0x80)==0) d->data()->dacData(CrcFeConfigurationData::boardB,512*(v%128));

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcCmLedVcalibScan: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdCmSipm);
      d->data()->calibWidth(d->data()->holdStart()+128);

      unsigned steps(v+1);
      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      //d->data()->dacData(CrcFeConfigurationData::boardB,_configurationNumber*(65536/steps));
      d->data()->dacData(CrcFeConfigurationData::boardB,_configurationNumber*(10000/steps)+10000); // 65536/steps = maximal vcalib

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcCmLedHoldScan: {
      d->data()->calibEnable(true);
      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,8500);

      d->data()->holdStart(20+_configurationNumber);
      d->data()->calibWidth(d->data()->holdStart()+128);

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcPmLed: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdPmSipm);
      //d->data()->holdStart(28);
      d->data()->calibWidth(d->data()->holdStart()+128);
      //d->data()->calibWidth(156);

      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,0);

      if((v&0x80)==0) d->data()->dacData(CrcFeConfigurationData::boardB,512*(v%128));

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcPmLedVcalibScan: {
      d->data()->calibEnable(true);
      d->data()->holdStart(_holdPmSipm);
      d->data()->calibWidth(d->data()->holdStart()+128);

      unsigned steps(v+1);
      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,_configurationNumber*65536/steps);//(10000/steps)+10000); // 65536/steps = maximal vcalib

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcPmLedHoldScan: {
      d->data()->calibEnable(true);

      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,25000);

      d->data()->holdStart(20+2*_configurationNumber);
      d->data()->calibWidth(d->data()->holdStart()+128);

      vc.ledSel(true);
      break;
    }

    case DaqRunType::ahcBeam:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:
    case DaqRunType::ahcCosmicsHoldScan:

    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:

    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {
      if((_configurationNumber%3)!=1) {
	//d->data()->sequenceDelay(8);
	d->data()->holdStart(_holdBeamSipm);
	d->data()->calibWidth(d->data()->holdStart()+128);
      } else {
	d->data()->calibEnable(true);
	d->data()->holdStart(_holdPmSipm);
	d->data()->calibWidth(d->data()->holdStart()+128);
	//d->data()->sequenceDelay(112);
	d->data()->dacData(CrcFeConfigurationData::boardA,0);
	d->data()->dacData(CrcFeConfigurationData::boardB,18000);
	vc.ledSel(true);
      }
      break;
    }

    case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::beamHoldScan: {
      //      if((_configurationNumber%3)==0) {
      //	d->data()->holdStart(_holdBeamSipm);
      // } else if((_configurationNumber%3)==2) {
	unsigned short hold[22]={1,2,3,4,5,6,7,8,9,10,11,12,13,14,15,20,30,40,50,100,150,200};
	//d->data()->sequenceDelay(hold[_configurationNumber%22]);
	d->data()->holdStart(hold[_configurationNumber%22]);
	//d->data()->sequenceDelay(hold[(_configurationNumber/3)%16]-1);
	// } else {
	//	d->data()->calibEnable(true);
	//	d->data()->sequenceDelay(112);
	//	d->data()->dacData(CrcFeConfigurationData::boardA,0);
	//	d->data()->dacData(CrcFeConfigurationData::boardB,25000);
	//	vc.ledSel(true);
    
	break;
    }

    case DaqRunType::ahcScintillatorHoldScan: {
      d->data()->calibEnable(true);
      d->data()->dacData(CrcFeConfigurationData::boardA,0);
      d->data()->dacData(CrcFeConfigurationData::boardB,25000);

      vc.ledSel(true);

      unsigned short hold[16]={1,2,3,4,5,6,7,8,9,10,15,20,30,40,50,100};
      d->data()->sequenceDelay(hold[_configurationNumber%16]-1);
      break;
    }
  
    default: {


      break;
    }
    };
  



    //    d->data()->holdWidth(_holdEndEdge - d->data()->holdStart()); // equalize hold end
    //d->data()->sequenceDelay(123);     // shift the whole timing sequence
    d->data()->holdEnd(_holdEndEdge);  // equalize hold end
    //d->data()->holdWidth(_holdEndEdge);  // equalize hold width!!!
    
    // Write in VFE control
    d->data()->vfeControl(vc.data());

    //PIN setting
    if(_runType.type()!=DaqRunType::ahcExpert )
      {
	// Now do the PIN diodes; hold and so on
	for(unsigned i(0);i<nPin;i++) {
	  std::cout <<"PIN HOLD" << std::endl;
	  //	  std::cout << _pinLocation[i].crateNumber() << " " << _pinLocation[i].crcComponent() << std::endl; 
	  
	  CrcLocationData<CrcFeConfigurationData>
	    *d(inserter.insert< CrcLocationData<CrcFeConfigurationData> >(true));
	  d->location(_pinLocation[i]);
	  
	  vc.ledSel(true);
	  d->data()->calibEnable(false);
	  d->data()->dacData(CrcFeConfigurationData::boardA,0);
	  d->data()->dacData(CrcFeConfigurationData::boardB,0);
	  if (_runType.type()==DaqRunType::ahcCmLedHoldScan )
	    {
	      d->data()->holdStart(20+_configurationNumber);
	      d->data()->calibWidth(d->data()->holdStart()+128);
	    }
	    else if ( _runType.type()==DaqRunType::ahcPmLedHoldScan )
	      {
		d->data()->holdStart(20+_configurationNumber);
		d->data()->calibWidth(d->data()->holdStart()+128+_configurationNumber);
	      }
	    else 
	      {
		d->data()->holdStart(_holdPin);
		d->data()->calibWidth(d->data()->holdStart()+128);
	      }

	  //std::cout << _holdEndEdge <<" "<< d->data()->holdStart() << std::endl;
	  //d->data()->holdWidth(_holdEndEdge - d->data()->holdStart());
	  //d->data()->sequenceDelay(123);     // shift the whole timing sequence
	  d->data()->holdEnd(_holdEndEdge);  // equalize hold end
	  //d->data()->holdWidth(_holdEndEdge);  // equalize hold width!!!
	  d->data()->vfeControl(vc.data());
	  d->label(1);
	  d->print(std::cout," ");
	  
	}
	
      }
    

    if(doPrint(r.recordType(),2)) {
      d->print(std::cout," ");
      vc.print(std::cout," ") << std::endl;
    }
        
    return true;
  }



  virtual bool ahcVfeConfiguration(RcdRecord &r) {


    if (_runType.type() == DaqRunType::ahcExpert) {

      SubInserter inserter(r);
      
      std::vector< CrcLocationData<AhcVfeConfigurationData> > vVcd;
      
      for(unsigned f(0);f<8;f++) {

	_location.slotBroadcast(true);

	if(f==0) _location.crcComponent(CrcLocation::fe0);
	if(f==1) _location.crcComponent(CrcLocation::fe1);
	if(f==2) _location.crcComponent(CrcLocation::fe2);
	if(f==3) _location.crcComponent(CrcLocation::fe3);
	if(f==4) _location.crcComponent(CrcLocation::fe4);
	if(f==5) _location.crcComponent(CrcLocation::fe5);
	if(f==6) _location.crcComponent(CrcLocation::fe6);
	if(f==7) _location.crcComponent(CrcLocation::fe7);

	_location.label(1);

	unsigned t(vVcd.size());
	vVcd.push_back(CrcLocationData<AhcVfeConfigurationData>(_location));
	
	config[f]->insertSettings(vVcd[t].data());


      }

      for(unsigned i(0);i<vVcd.size();i++) {
	if(doPrint(r.recordType(),1)) vVcd[i].print(std::cout,"  ") << std::endl;
	inserter.insert< CrcLocationData<AhcVfeConfigurationData> >(vVcd[i]);
      }

      std::cout << "end of VFE expert configuration" << endl;
      return true; //ignore all the rest in expert mode
    }
    




    SubInserter inserter(r);
 
    CrcLocationData<AhcVfeConfigurationData>
      *d(inserter.insert< CrcLocationData<AhcVfeConfigurationData> >(true));

    d->crateNumber(_location.crateNumber());
    d->slotBroadcast(true);
    d->crcComponent(CrcLocation::feBroadcast);
    d->label(1);

    AhcVfeShiftRegister calibMode; // Default

    AhcVfeShiftRegister physicsMode;
    physicsMode.shapingCapacitor(0,true);
    physicsMode.shapingCapacitor(1,true);
    physicsMode.shapingCapacitor(2,true);
    physicsMode.gainCapacitor(1,true);
    physicsMode.injectionResistor(true);
       
    const unsigned char v(_runType.version());

    switch(_runType.type()) {

    case DaqRunType::ahcTest: {
      for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,calibMode);
      break;
    }
      
    case DaqRunType::ahcAnalogOut: {
      if((v&0x80)==0) { 
	for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,calibMode);
      } else {
	for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,physicsMode);
      }
      break;
    }

    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcCmAsic:
    case DaqRunType::ahcCmAsicVcalibScan:
    case DaqRunType::ahcCmAsicHoldScan:
    case DaqRunType::ahcCmLed:
    case DaqRunType::ahcCmLedVcalibScan:
    case DaqRunType::ahcCmLedHoldScan: {
      for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,calibMode);
      break;
    }
      
    case DaqRunType::ahcDacScan:
    case DaqRunType::ahcPmNoise:
    case DaqRunType::ahcPmAsic:
    case DaqRunType::ahcPmAsicVcalibScan:
    case DaqRunType::ahcPmAsicHoldScan:
    case DaqRunType::ahcPmLed:
    case DaqRunType::ahcPmLedVcalibScan:
    case DaqRunType::ahcPmLedHoldScan:
    case DaqRunType::ahcBeam:
    case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:

    case DaqRunType::beamTest:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:
    case DaqRunType::beamStage:
    case DaqRunType::beamStageScan:

    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {
      for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,physicsMode);
      break;
    }

    case DaqRunType::ahcExpert: {      
    }


    default: {
      break;
    }
    };

    if(doPrint(r.recordType(),2)) d->print(std::cout," ") << std::endl;


    if(_runType.type()!=DaqRunType::ahcExpert)
      {
	// Now do the PIN diodes; the SR is the same for all runs
	for(unsigned i(0);i<nPin;i++) {
	  if(_pinLocation[i].crateNumber()==0xac) {

	    d=inserter.insert< CrcLocationData<AhcVfeConfigurationData> >(true);
	    d->location(_pinLocation[i]);
	    d->label(1);

	
	    AhcVfeShiftRegister pinMode;
	    pinMode.gainCapacitor(0,false); 
	    pinMode.gainCapacitor(1,true);
	    pinMode.gainCapacitor(2,false); 
	    pinMode.gainCapacitor(3,true); // SR default setting = calib mode
	    pinMode.shapingCapacitor(0,true); 
	    pinMode.shapingCapacitor(1,true);
	    pinMode.shapingCapacitor(2,true); 
	    pinMode.shapingCapacitor(3,true); // SR default setting = calib mode
	    
	    pinMode.injectionResistor(true);
 
    
	    for(unsigned h(0);h<12;h++) d->data()->shiftRegister(h,pinMode);
    
	    if(doPrint(r.recordType(),2)) d->print(std::cout," ") << std::endl;
	  }
	}
       }
    return true;
  }


  virtual bool ahcSlowConfiguration(RcdRecord &r) {


    SubInserter inserter(r);
    AhcSlowConfigurationData *b(inserter.insert<AhcSlowConfigurationData>(true));
 
     
    //const unsigned char v(_runType.version());

    switch(_runType.type()) {

    case DaqRunType::ahcTest: 
    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcPmNoise: 
    case DaqRunType::ahcAnalogOut:
    case DaqRunType::ahcDacScan:
    case DaqRunType::ahcCmAsic:
    case DaqRunType::ahcCmAsicVcalibScan:
    case DaqRunType::ahcCmAsicHoldScan:
    case DaqRunType::ahcPmAsic:
    case DaqRunType::ahcPmAsicVcalibScan:
    case DaqRunType::ahcPmAsicHoldScan:
    case DaqRunType::ahcCmLed:
    case DaqRunType::ahcCmLedVcalibScan:
    case DaqRunType::ahcCmLedHoldScan:
    case DaqRunType::ahcPmLed:
    case DaqRunType::ahcPmLedVcalibScan:
    case DaqRunType::ahcPmLedHoldScan: 
    case DaqRunType::ahcBeam:
    case DaqRunType::ahcBeamHoldScan:{
      break;
    }

    case DaqRunType::ahcBeamStage:{
      //single position
      b->mmXPosition(10);
      b->mmYPosition(10);
      b->print(std::cout);
      
      break;
    }

    case DaqRunType::ahcBeamStageScan:{
      //use read in file information
      std::cout << "BeamStageScan chosen" << std::endl;
      
      // Set stage position
      if (_configurationNumber%3==0)
	{
	  std::cout << (int)(_configurationNumber/3) << " " << getSTAGExArray((int)_configurationNumber/3) << std::endl;
      
	  b->mmXPosition( getSTAGExArray((int)(_configurationNumber/3)) );
	  b->mmYPosition( getSTAGEyArray((int)(_configurationNumber/3)) );
	  b->print(std::cout);
	}
      break;

    }
    case DaqRunType::ahcCosmics:
    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:
    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: 
    case DaqRunType::ahcCosmicsHoldScan: 
    case DaqRunType::ahcExpert: {
      break;
    }
    
    default: {
      break;
    }
    };

    if(doPrint(r.recordType(),1)) b->print(std::cout," ") << std::endl;

    return true;
  }



protected:
  unsigned const static maxPin=2;
  CrcLocation _pinLocation[maxPin];
  unsigned nPin;
  CrcReadoutConfigurationData _readoutConfiguration;
  std::vector< CrcLocationData<AhcVfeStartUpData> > _vDac;

  AhcConfigReader* config[8];
  unsigned _holdEndEdge;
  unsigned _holdCmAsic;
  unsigned _holdPmAsic;
  unsigned _holdCmSipm;
  unsigned _holdPmSipm;
  unsigned _holdBeamSipm;
  unsigned _holdPin;

//stage positions
  vector<int> _STAGEchipArray;
  vector<int> _STAGEchanArray;
  vector<int> _STAGExArray;
  vector<int> _STAGEyArray;

  int AhcConfiguration::getSTAGExArray(int i)
  {
    return _STAGExArray[i];
  }

  int AhcConfiguration::getSTAGEyArray(int i)
  {
    return _STAGEyArray[i];
  }

  unsigned AhcConfiguration::getSTAGEArraySize()
  {
    return _STAGExArray.size();
  }

  unsigned AhcConfiguration::getROchip(int i)
  {
    return _STAGEchipArray[i];
  }

  unsigned AhcConfiguration::getROchan(int i)
  {
    return _STAGEchanArray[i];
  }


};

#endif
