#ifndef DaqConfiguration_HH
#define DaqConfiguration_HH

#include <vector>
#include <fstream>
#include <iostream>

#include "RcdUserRW.hh"

#include "SubInserter.hh"
#include "SubAccessor.hh"

//#include "DaqStateData.hh"
//#include "DaqStartUp.hh"
#include "DaqRunStart.hh"
#include "DaqRunEnd.hh"
#include "DaqConfigurationStart.hh"
#include "DaqConfigurationEnd.hh"
#include "DaqSlowReadout.hh"
#include "DaqAcquisitionStart.hh"
#include "DaqAcquisitionEnd.hh"
#include "DaqSpillStart.hh"
#include "DaqSpillEnd.hh"
#include "DaqTransferStart.hh"
#include "DaqTransferEnd.hh"

#include "DaqRunNumber.hh"


class DaqConfiguration : public RcdUserRW {

public:
  enum Counter {
    rInJ,
    cInR,
    srInR,srInC,
    aInR,aInC,
    sInR,sInC,sInA,
    tInR,tInC,tInA,
    preInR,preInC,preInA,preInS,preInT,
    eInR,eInC,eInA,eInS,eInT,
    pteInR,pteInC,pteInA,pteInS,pteInT,
    endOfCounterEnum
  };

  DaqConfiguration() :
    _shmRunControl(RunControl::shmKey), _pRc(_shmRunControl.payload()) {
    assert(_pRc!=0);
  }

  virtual ~DaqConfiguration() {
  }

  DaqRunStart runStart() const {
    return _runStartFromRunControl;
  }

  void runStart(DaqRunStart r) {
    _runStartFromRunControl=r;
    assert(_runStartFromRunControl.runType().knownType());
  }

  bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "DaqConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
 
    // Check record type
    switch (r.recordType()) {

    case RcdHeader::startUp: {
      _count[rInJ]=0;
      
      /*
      SubAccessor accessor(r);
      std::vector<const DaqStartUp*> v(accessor.extract<DaqStartUp>());
      assert(v.size()<=1);

      _runType=DaqRunType(DaqRunType::daq,DaqRunType::daqTest,0);

      if(v.size()==1) {
	if(_printLevel>0) v[0]->print(std::cout," ") << std::endl;
	_runType=v[0]->runType();
      }

      assert(_runType.knownType());
      */
     
      break;
    }
      
    case RcdHeader::runStart: {
      _count[cInR]=0; //c
      _count[srInR]=0; //sr
      _count[aInR]=0; //a
      _count[sInR]=0; //s
      _count[tInR]=0; //t
      _count[preInR]=0; //e
      _count[pteInR]=0; //e
      _count[eInR]=0; //e

      SubInserter inserter(r);
      DaqRunStart *d(inserter.insert<DaqRunStart>(true));

      _runStartFromRunControl=_pRc->runStart();
      _pRc->reset();

      if(_runStartFromRunControl.runType().writeRun()) {
	_runNumber=daqReadRunNumber();
	daqWriteRunNumber(_runNumber+1);
      } else {
	_runNumber=r.recordTime().seconds();
      }

      d->runNumber(_runNumber);
      d->runType(_runStartFromRunControl.runType());
      setRun(*d);
      
      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }
      
    case RcdHeader::configurationStart: {
      _count[srInC]=0; //a
      _count[aInC]=0; //a
      _count[sInC]=0; //s
      _count[tInC]=0; //t
      _count[preInC]=0; //e
      _count[pteInC]=0; //e
      _count[eInC]=0; //e
          
      _daqConfigurationStart.reset();
      _daqConfigurationStart.configurationNumberInRun(_count[cInR]);
      setConfiguration(_daqConfigurationStart);
     
      SubInserter inserter(r);
      inserter.insert<DaqConfigurationStart>(_daqConfigurationStart);
    
      _count[cInR]++;
     
      if(doPrint(r.recordType(),1)) _daqConfigurationStart.print(std::cout," ") << std::endl;
      
      break;
    }

    case RcdHeader::slowReadout: {
      SubInserter inserter(r);
      DaqSlowReadout *d(inserter.insert<DaqSlowReadout>(true));

      d->slowReadoutNumberInRun(_count[srInR]);
      d->slowReadoutNumberInConfiguration(_count[srInC]);

      _count[srInR]++;
      _count[srInC]++;
      
      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }

    case RcdHeader::acquisitionStart: {
      _count[sInA]=0;//s
      _count[tInA]=0;//t
      _count[preInA]=0;//e
      _count[pteInA]=0;//e
      _count[eInA]=0;//e
    
      SubInserter inserter(r);
      DaqAcquisitionStart *d(inserter.insert<DaqAcquisitionStart>(true));

      d->acquisitionNumberInRun(_count[aInR]);
      d->acquisitionNumberInConfiguration(_count[aInC]);
      d->maximumNumberOfEventsInAcquisition(_daqConfigurationStart.maximumNumberOfEventsInAcquisition());
      d->maximumTimeOfAcquisition(_daqConfigurationStart.maximumTimeOfAcquisition());

      _count[aInR]++;
      _count[aInC]++;
      
      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }
      
    case RcdHeader::spillStart:
    case RcdHeader::spill: {
      _count[preInS]=0;//e
      _count[pteInS]=0;//e
      _count[eInS]=0;//e
    
      SubInserter inserter(r);
      DaqSpillStart *d(inserter.insert<DaqSpillStart>(true));

      d->spillNumberInRun(_count[sInR]);
      d->spillNumberInConfiguration(_count[sInC]);
      d->maximumNumberOfEventsInSpill(_daqConfigurationStart.maximumNumberOfEventsInSpill());
      d->maximumTimeOfSpill(_daqConfigurationStart.maximumTimeOfSpill());
    
      _count[sInR]++;
      _count[sInC]++;
      _count[sInA]++;

      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }
      
    case RcdHeader::transferStart:
    case RcdHeader::transfer: {
      _count[preInT]=0;//e
      _count[pteInT]=0;//e
      _count[eInT]=0;//e

      SubInserter inserter(r);
      DaqTransferStart *d(inserter.insert<DaqTransferStart>(true));

      d->transferNumberInRun(_count[tInR]);
      d->transferNumberInConfiguration(_count[tInC]);
      
      _count[tInR]++;
      _count[tInC]++;
      _count[tInA]++;
      
      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      break;
   }

    case RcdHeader::preEvent: {
      SubInserter inserter(r);
      DaqEvent *d(inserter.insert<DaqEvent>(true));
      
      d->eventNumberInRun(_count[preInR]);
      d->eventNumberInConfiguration(_count[preInC]);
      d->eventNumberInAcquisition(_count[preInA]);
      
      _count[preInR]++;
      _count[preInC]++;
      _count[preInA]++;
      _count[preInS]++;

      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }

    case RcdHeader::postEvent: {
      SubInserter inserter(r);
      DaqEvent *d(inserter.insert<DaqEvent>(true));
      
      d->eventNumberInRun(_count[pteInR]);
      d->eventNumberInConfiguration(_count[pteInC]);
      d->eventNumberInAcquisition(_count[pteInA]);
      
      _count[pteInR]++;
      _count[pteInC]++;
      _count[pteInA]++;
      _count[pteInT]++;

      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }

    case RcdHeader::event: {
      SubInserter inserter(r);
      DaqEvent *d(inserter.insert<DaqEvent>(true));
      
      d->eventNumberInRun(_count[eInR]);
      d->eventNumberInConfiguration(_count[eInC]);
      d->eventNumberInAcquisition(_count[eInA]);
      
      _count[eInR]++;
      _count[eInC]++;
      _count[eInA]++;

      if(doPrint(r.recordType(),1)) d->print(std::cout," ") << std::endl;
      
      break;
    }
      
    default: {
      break;
    }
    };

    return true;
  }


  virtual void setRun(DaqRunStart &d) const {
    const unsigned char v(_runStartFromRunControl.runType().version());

    switch(_runStartFromRunControl.runType().type()) {

    case DaqRunType::daqTest: {
      break;
    }

    case DaqRunType::crcTest: {
      break;
    }
    case DaqRunType::crcNoise: {
      break;
    }
    case DaqRunType::crcBeParameters: {
      if(v== 0) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 1) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 2) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 3) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 4) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 5) d.maximumNumberOfConfigurationsInRun(4096);
      if(v== 6) d.maximumNumberOfConfigurationsInRun(4);
      if(v== 7) d.maximumNumberOfConfigurationsInRun(256);
      if(v== 8) d.maximumNumberOfConfigurationsInRun(4);
      if(v== 9) d.maximumNumberOfConfigurationsInRun(65536);
      if(v==10) d.maximumNumberOfConfigurationsInRun(65536);
      break;
    }
    case DaqRunType::crcFeParameters: {
      if(v== 0) d.maximumNumberOfConfigurationsInRun(4);
      if(v== 1) d.maximumNumberOfConfigurationsInRun(4);
      if(v== 2) d.maximumNumberOfConfigurationsInRun(18);
      if(v== 3) d.maximumNumberOfConfigurationsInRun(32);
      if(v== 4) d.maximumNumberOfConfigurationsInRun(32);
      if(v== 5) d.maximumNumberOfConfigurationsInRun(16);
      if(v== 6) d.maximumNumberOfConfigurationsInRun(128);
      break;
    }
    case DaqRunType::crcIntDac: {
      break;
    }
    case DaqRunType::crcIntDacScan: {
      d.maximumNumberOfConfigurationsInRun(2*v+4);
      break;
    }
    case DaqRunType::crcExtDac: {
      break;
    }
    case DaqRunType::crcExtDacScan: {
      d.maximumNumberOfConfigurationsInRun(2*v+2);
      break;
    }
    case DaqRunType::crcFakeEvent: {
      break;
    }
    case DaqRunType::crcModeTest: {
      d.maximumNumberOfConfigurationsInRun(21);
      break;
    }

    case DaqRunType::trgTest: {
      break;
    }
    case DaqRunType::trgParameters: {
      break;
    }
    case DaqRunType::trgSpill: {
      break;
    }
    case DaqRunType::trgNoise: {
      break;
    }

    case DaqRunType::emcTest: {
      break;
    }
    case DaqRunType::emcNoise: {
      break;
    }
    case DaqRunType::emcFeParameters: {
      break;
    }
    case DaqRunType::emcVfeDac: {
      break;
    }
    case DaqRunType::emcVfeDacScan: {
      d.maximumNumberOfConfigurationsInRun(6*(v+1)+4);
      break;
    }
    case DaqRunType::emcVfeHoldScan: {
      d.maximumNumberOfConfigurationsInRun(6*(v+1)+4);
      break;
    }
    case DaqRunType::emcTrgTiming: {
      break;
    }
    case DaqRunType::emcTrgTimingScan: {
      d.maximumNumberOfConfigurationsInRun(56);
      break;
    }
    case DaqRunType::emcBeam: {
      break;
    }
    case DaqRunType::emcBeamHoldScan: {
      break;
    }
    case DaqRunType::emcCosmics: {
      break;
    }
    case DaqRunType::emcCosmicsHoldScan: {
      break;
    }

    case DaqRunType::ahcTest: {
      break;
    }
    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcPmNoise: {
      break;
    }
    case DaqRunType::ahcDacScan: {
      d.maximumNumberOfConfigurationsInRun(2*v+1);
      break;
    }
    case DaqRunType::ahcAnalogOut: {
      break;
    }
    case DaqRunType::ahcCmAsic: {
      break;
    }
    case DaqRunType::ahcCmAsicVcalibScan: {
      d.maximumNumberOfConfigurationsInRun(2*(v+1));
      break;
    }
    case DaqRunType::ahcCmAsicHoldScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcPmAsic: {
      break;
    }
    case DaqRunType::ahcPmAsicVcalibScan: {
      d.maximumNumberOfConfigurationsInRun(2*(v+1));
      break;
    }
    case DaqRunType::ahcPmAsicHoldScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcCmLed: {
      break;
    }
    case DaqRunType::ahcCmLedVcalibScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcCmLedHoldScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcPmLed: {
      break;
    }
    case DaqRunType::ahcPmLedVcalibScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcPmLedHoldScan: {
      d.maximumNumberOfConfigurationsInRun(v+1);
      break;
    }
    case DaqRunType::ahcScintillatorHoldScan: {
      d.maximumNumberOfConfigurationsInRun(32);
      break;
    }
    case DaqRunType::ahcBeam: {
      break;
    }
    case DaqRunType::ahcBeamHoldScan: {
      break;
    }
    case DaqRunType::ahcBeamStage: {
      break;
    }
    case DaqRunType::ahcBeamStageScan: {
      break;
    }
    case DaqRunType::ahcCosmics: {
      break;
    }
    case DaqRunType::ahcCosmicsHoldScan: {
      break;
    }
    case DaqRunType::ahcExpert: {
      if (v == 1 || v == 2) d.maximumNumberOfConfigurationsInRun(1000);
      break;
    }

    case DaqRunType::tcmCalPedestal: //KF added 022406
    case DaqRunType::tcmPhysPedestal: //KF added 022406
    case DaqRunType::tcmCalLed: //KF added 022406
    case DaqRunType::tcmPhysLed: //KF added 022406
    case DaqRunType::tcmCosmics: {//KF added 022406 //same as tcmBeam for now, functionality to be enhanced later
      break;
    }
    case DaqRunType::tcmBeam: { //KF added
      break;
    }
    case DaqRunType::tcmBeamHoldScan: {
      d.maximumNumberOfConfigurationsInRun(1000);
      break;
    }

    case DaqRunType::bmlTest: {
      break;
    }
    case DaqRunType::bmlNoise: {
      break;
    }
    case DaqRunType::bmlInternalTest: {
      break;
    }
    case DaqRunType::bmlBeam: {
      break;
    }

    case DaqRunType::slowTest: {	
      d.maximumNumberOfAcquisitionsInRun(0);
      d.maximumNumberOfEventsInRun(0);
      break;
    }
    case DaqRunType::slowMonitor: {
      d.maximumNumberOfAcquisitionsInRun(0);
      d.maximumNumberOfEventsInRun(0);
      break;
    }

    case DaqRunType::beamTest: {
      break;
    }
    case DaqRunType::beamNoise: {
      break;
    }
    case DaqRunType::beamData: {
      break;
    }
    case DaqRunType::beamHoldScan: {
      break;
    }

    case DaqRunType::cosmicsTest: {
      break;
    }
    case DaqRunType::cosmicsNoise: {
      break;
    }
    case DaqRunType::cosmicsData: {
      break;
    }
    case DaqRunType::cosmicsHoldScan: {
      break;
    }

    default: {
      break;
    }
    };

    // Reset limits if smaller from run control
    if(d.maximumNumberOfConfigurationsInRun()>
       _runStartFromRunControl.maximumNumberOfConfigurationsInRun())
      d.maximumNumberOfConfigurationsInRun(_runStartFromRunControl.maximumNumberOfConfigurationsInRun());

    if(d.maximumNumberOfAcquisitionsInRun()>
       _runStartFromRunControl.maximumNumberOfAcquisitionsInRun())
      d.maximumNumberOfAcquisitionsInRun(_runStartFromRunControl.maximumNumberOfAcquisitionsInRun());

    if(d.maximumNumberOfEventsInRun()>
       _runStartFromRunControl.maximumNumberOfEventsInRun())
      d.maximumNumberOfEventsInRun(_runStartFromRunControl.maximumNumberOfEventsInRun());

    if(d.maximumTimeOfRun()>
       _runStartFromRunControl.maximumTimeOfRun())
      d.maximumTimeOfRun(_runStartFromRunControl.maximumTimeOfRun());
  }

  virtual void setConfiguration(DaqConfigurationStart &d) const {
    const unsigned iCfg(d.configurationNumberInRun());

    const unsigned char v(_runStartFromRunControl.runType().version());

    switch(_runStartFromRunControl.runType().type()) {

    case DaqRunType::daqTest: {
      break;
    }

    case DaqRunType::trgTest: {
      break;
    }
    case DaqRunType::trgParameters: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::trgNoise: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::trgSpill: {
      d.maximumNumberOfEventsInAcquisition(500);
      d.maximumTimeOfSpill(UtlTimeDifference(1,0));
      break;
    }

    case DaqRunType::crcTest: {
      break;
    }
    case DaqRunType::crcNoise: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcBeParameters: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcFeParameters: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcIntDac: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcIntDacScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcExtDac: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcExtDacScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcFakeEvent: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::crcModeTest: {
      d.maximumNumberOfAcquisitionsInConfiguration(8);
      unsigned i(iCfg%7);
      unsigned j((iCfg/7)%3);
      if(j==1) d.maximumNumberOfEventsInAcquisition(500);
      else     d.maximumNumberOfEventsInAcquisition(500+100*i);
      break;
    }

    case DaqRunType::emcTest: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcNoise: {
      if((v&0x80)==0) d.maximumNumberOfEventsInAcquisition(500);
      else            d.maximumNumberOfEventsInAcquisition(100);
      break;
    }
    case DaqRunType::emcVfeDac: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcVfeDacScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcVfeHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcTrgTiming: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcTrgTimingScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::emcBeam:
    case DaqRunType::emcBeamHoldScan: {
      if((iCfg%3)<2) { // Pedestals or LED
	d.maximumNumberOfAcquisitionsInConfiguration(1);
	d.maximumNumberOfEventsInAcquisition(500);
      } else { // Data
	d.maximumNumberOfAcquisitionsInConfiguration(2);
	d.maximumNumberOfEventsInAcquisition(500);

	// TEMP FOR TESTING MISSING TRIGGERS
	d.maximumTimeOfEvent(UtlTimeDifference(20));

      }
      break;
    }
    case DaqRunType::emcCosmics:
    case DaqRunType::emcCosmicsHoldScan: {
      if((iCfg%3)<2) { // Pedestals or LED
	d.maximumNumberOfAcquisitionsInConfiguration(1);
	d.maximumNumberOfEventsInAcquisition(100);
      } else { // Data
	//d.maximumNumberOfAcquisitionsInConfiguration(2);
	//d.maximumNumberOfEventsInAcquisition(500);
	d.maximumNumberOfAcquisitionsInConfiguration(100); // TEMP FOR F/W BUG
	d.maximumNumberOfEventsInAcquisition(10);          // TEMP FOR F/W BUG
	d.maximumTimeOfEvent(UtlTimeDifference(20,0));
      }
      break;
    }

    case DaqRunType::ahcTest: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmNoise:
    case DaqRunType::ahcPmNoise: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcDacScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcAnalogOut: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmAsic: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmAsicVcalibScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmAsicHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmAsic: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmAsicVcalibScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmAsicHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmLed: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmLedVcalibScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(4);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcCmLedHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmLed: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmLedVcalibScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(4);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcPmLedHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcScintillatorHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::ahcBeam:
    //case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:
    case DaqRunType::ahcCosmicsHoldScan: {
      if((iCfg%3)<2) { // Pedestals or LED
	d.maximumNumberOfAcquisitionsInConfiguration(1);
	d.maximumNumberOfEventsInAcquisition(500);
      } else { // Data
	d.maximumNumberOfAcquisitionsInConfiguration(20);
	d.maximumNumberOfEventsInAcquisition(500);
	//d.maximumTimeOfConfiguration(UtlTimeDifference(5*60,0));

	// TEMP FOR TESTING MISSING TRIGGERS
	if(_runStartFromRunControl.runType().type()==DaqRunType::ahcBeam) d.maximumTimeOfEvent(UtlTimeDifference(20));

      }
      break;
    }
    case DaqRunType::ahcBeamHoldScan: {
      d.maximumNumberOfAcquisitionsInConfiguration(2);
      d.maximumNumberOfEventsInAcquisition(500);
      //d.maximumTimeOfConfiguration(UtlTimeDifference(5*60,0));
      break;
    }
    case DaqRunType::ahcExpert: {
      d.maximumNumberOfEventsInAcquisition(500);
      if (v == 1 || v == 2) d.maximumNumberOfAcquisitionsInConfiguration(1);
      break;
    }

    case DaqRunType::tcmCalPedestal: //KF added 022406
    case DaqRunType::tcmPhysPedestal: //KF added 022406
    case DaqRunType::tcmCalLed: //KF added 022406
    case DaqRunType::tcmPhysLed: {//KF added 022406
      d.maximumNumberOfEventsInAcquisition(500);
      d.maximumNumberOfAcquisitionsInConfiguration(20); //beni changed to 20 from 1
      break;
    }
    case DaqRunType::tcmCosmics: //KF added 022406 same as tcmBeam for now funtionality to be added later
    case DaqRunType::tcmBeam: { //KF added
      d.maximumTimeOfConfiguration(UtlTimeDifference(1*60,0));
      break;
    }
    case DaqRunType::tcmBeamHoldScan: {
      d.maximumNumberOfEventsInAcquisition(500);
      d.maximumNumberOfAcquisitionsInConfiguration(1);
      break;
    }

    case DaqRunType::bmlTest: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::bmlNoise: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::bmlInternalTest: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }
    case DaqRunType::bmlBeam: {
      d.maximumNumberOfEventsInAcquisition(500);
      break;
    }

    case DaqRunType::slowTest: {
      break;
    }
    case DaqRunType::slowMonitor: {
      d.maximumNumberOfAcquisitionsInConfiguration(0);
      d.maximumNumberOfEventsInConfiguration(0);
      d.maximumNumberOfEventsInAcquisition(0);
      break;
    }

    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:
    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {
      if((iCfg%3)<2) { // Pedestals or LED
	//d.maximumNumberOfAcquisitionsInConfiguration(1);
	//d.maximumNumberOfEventsInAcquisition(500);
	d.maximumNumberOfEventsInConfiguration(500);

      } else { // Data
	d.maximumNumberOfEventsInAcquisition(500);

	d.maximumTimeOfConfiguration(UtlTimeDifference(15*60,0));

#ifdef DESY_SETTINGS
	//d.maximumNumberOfEventsInConfiguration(10000);
	d.maximumNumberOfEventsInConfiguration(5000); // Goetz requested reduction from 10k 22/5/06

	//d.maximumNumberOfEventsInAcquisition(32);          // TDC limit
	//d.maximumNumberOfEventsInAcquisition(12);

	d.maximumNumberOfEventsInSpill(32);          // TDC limit
	//d.maximumTimeOfSpill(UtlTimeDifference(0,60000)); // 60ms
	//d.maximumTimeOfSpill(UtlTimeDifference(0,40000)); // 40ms
	//d.maximumTimeOfSpill(UtlTimeDifference(0,100000)); // 100ms
	d.maximumTimeOfSpill(UtlTimeDifference(1));

	d.maximumTimeOfEvent(UtlTimeDifference(0,900000)); // 0.9s
	//d.maximumTimeOfEvent(UtlTimeDifference(1));
#endif

#ifdef CERN_SETTINGS
	d.maximumNumberOfEventsInConfiguration(10000);
	d.maximumNumberOfEventsInSpill(2000);
	d.maximumTimeOfSpill(UtlTimeDifference(4,800000)); // 4.8s
	d.maximumTimeOfEvent(UtlTimeDifference(12)); // 12s
#endif

      }
      break;
    }

    default: {
      break;
    }
    };
  }


protected:
  DaqRunStart _runStartFromRunControl;

private:
  unsigned _runNumber;
  DaqConfigurationStart _daqConfigurationStart;
  unsigned _count[endOfCounterEnum];


  ShmObject<RunControl> _shmRunControl;
  RunControl *_pRc;

};

#endif
