#ifndef TrgConfiguration_HH
#define TrgConfiguration_HH

#include <iostream>
#include <fstream>

#include "RcdUserRW.hh"
#include "SubRecordType.hh"
#include "SubInserter.hh"
#include "SubAccessor.hh"


class TrgConfiguration : public RcdUserRW {

public:
  TrgConfiguration(unsigned char c, unsigned char s) :
    _location(c,s,CrcLocation::beTrg,1) {
  }

  virtual ~TrgConfiguration() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << std::endl << "TrgConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    // Check record type
    switch (r.recordType()) {


    // Run start 
    case RcdHeader::runStart: {

      // Access the DaqRunStart
      SubAccessor accessor(r);
      std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
      assert(v.size()==1);
      if(doPrint(r.recordType(),1)) v[0]->print(std::cout," ") << std::endl;
      _runType=v[0]->runType();
      break;
    }


    // Configuration start is used to set up system
    case RcdHeader::configurationStart: {

      // Access the DaqConfigurationStart
      SubAccessor accessor(r);
      std::vector<const DaqConfigurationStart*>
	v(accessor.extract<DaqConfigurationStart>());
      assert(v.size()==1);
      if(doPrint(r.recordType(),1)) v[0]->print(std::cout," ") << std::endl;

      assert(trgReadoutConfiguration(r,*(v[0])));
      assert(trgConfiguration(r,*(v[0])));

      break;
    }

    default: {

      break;
    }
    }; // switch(recordtype)

    return true;
  }

  bool trgReadoutConfiguration(RcdRecord &r, const DaqConfigurationStart &d) {

    // Always put a readout configuration into the record

    // Turn off trigger for several major run types
    if(_runType.majorType()==DaqRunType::daq ||
       _runType.majorType()==DaqRunType::crc ||
       _runType.majorType()==DaqRunType::slow) {
      _trgReadoutConfiguration.enable(false);

    // Otherwise need to handle by type
    } else {

      // Assume most types need a trigger
      _trgReadoutConfiguration.enable(true);
      
      // Get the configuration number and version
      const unsigned iCfg(d.configurationNumberInRun());
      const unsigned char v(_runType.version());
      const UtlPack u(_runType.version());
      

      switch(_runType.type()) {
	
      case DaqRunType::trgTest: {
	_trgReadoutConfiguration.enable(true);
	break;
      }
      case DaqRunType::trgReadout: {
	if(iCfg==0) {
	  _trgReadoutConfiguration.enable(true);
	  _trgReadoutConfiguration.clearBeTrgTrigger(true);
	  _trgReadoutConfiguration.beTrgSoftTrigger((u.word()%2)==1);
	  _trgReadoutConfiguration.readPeriod(1);
	  _trgReadoutConfiguration.beTrgSquirt(((u.word()/2)%2)==1);
	}
	break;
      }
      case DaqRunType::trgParameters: {
	if(iCfg==0) {
	  _trgReadoutConfiguration.enable(true);
	  _trgReadoutConfiguration.clearBeTrgTrigger(true);
	  _trgReadoutConfiguration.beTrgSoftTrigger((u.word()%2)==0);
	  _trgReadoutConfiguration.readPeriod(1);
	  _trgReadoutConfiguration.beTrgSquirt(false);
	}
	break;
      }
      case DaqRunType::trgNoise: {
	if(iCfg==0) {
	  _trgReadoutConfiguration.enable(true);
	}
	break;
      }
      case DaqRunType::trgSpill: {
	if(iCfg==0) {
	  _trgReadoutConfiguration.beTrgSquirt(true);
	  _trgReadoutConfiguration.beTrgPollNumber(10000);
	  _trgReadoutConfiguration.beTrgSpillNumber(500);
	  _trgReadoutConfiguration.beTrgSpillTime(UtlTimeDifference(1,0));
	  _trgReadoutConfiguration.enable(true);
	  _trgReadoutConfiguration.clearBeTrgTrigger(false);
	  _trgReadoutConfiguration.readPeriod(1);
	}
	break;
      }
	
      case DaqRunType::emcTest:
      case DaqRunType::emcNoise:
      case DaqRunType::emcFeParameters:
      case DaqRunType::emcVfeDac:
      case DaqRunType::emcVfeDacScan:
      case DaqRunType::emcVfeHoldScan: {
	_trgReadoutConfiguration.enable(false);
	break;
      }
      case DaqRunType::emcTrgTiming:
      case DaqRunType::emcTrgTimingScan: {
	break;
      }
	
      case DaqRunType::ahcTest:
      case DaqRunType::ahcDacScan:
      case DaqRunType::ahcCmNoise:
      case DaqRunType::ahcPmNoise:
      case DaqRunType::ahcAnalogOut:
      case DaqRunType::ahcCmAsic:
      case DaqRunType::ahcCmAsicVcalibScan:
      case DaqRunType::ahcCmAsicHoldScan:
      case DaqRunType::ahcPmAsic:
      case DaqRunType::ahcPmAsicVcalibScan:
      case DaqRunType::ahcPmAsicHoldScan:
      case DaqRunType::ahcCmLed:
      case DaqRunType::ahcCmLedVcalibScan:
      case DaqRunType::ahcCmLedHoldScan:
      case DaqRunType::ahcPmLed:
      case DaqRunType::ahcPmLedVcalibScan:
      case DaqRunType::ahcPmLedHoldScan:
      case DaqRunType::ahcScintillatorHoldScan: {
	_trgReadoutConfiguration.enable(false);
	break;
      }
	
      case DaqRunType::dhcTest:
      case DaqRunType::dhcNoise: {
	_trgReadoutConfiguration.enable(false);
	break;
      }

      case DaqRunType::tcmTest:
      case DaqRunType::tcmNoise:
      case DaqRunType::tcmCalLed:
      case DaqRunType::tcmPhysLed:
      case DaqRunType::tcmCalPedestal:
      case DaqRunType::tcmPhysPedestal: {
	_trgReadoutConfiguration.enable(false);
	break;
      }
	
      case DaqRunType::bmlInternalTest: {
	_trgReadoutConfiguration.enable(false);
	break;
      }
      case DaqRunType::bmlNoise: {
	break;
      }

      case DaqRunType::trgBeam:
      case DaqRunType::trgCosmics:
	
      case DaqRunType::emcBeam:
      case DaqRunType::emcBeamHoldScan:
      case DaqRunType::emcCosmics:
      case DaqRunType::emcCosmicsHoldScan:
	
      case DaqRunType::ahcBeam:
      case DaqRunType::ahcBeamHoldScan:
      case DaqRunType::ahcBeamStage:
      case DaqRunType::ahcBeamStageScan:
      case DaqRunType::ahcCosmics:
      case DaqRunType::ahcCosmicsHoldScan:
	
      case DaqRunType::dhcBeam:
      case DaqRunType::dhcCosmics:
	
      case DaqRunType::tcmBeam:
      case DaqRunType::tcmBeamHoldScan:
      case DaqRunType::tcmCosmics:
      case DaqRunType::tcmCosmicsHoldScan:
	
      case DaqRunType::bmlBeam:
	
      case DaqRunType::beamTest:
      case DaqRunType::beamNoise:
      case DaqRunType::beamData:
      case DaqRunType::beamHoldScan:
      case DaqRunType::beamStage:
      case DaqRunType::beamStageScan:
	
      case DaqRunType::cosmicsTest:
      case DaqRunType::cosmicsNoise:
      case DaqRunType::cosmicsData:
      case DaqRunType::cosmicsHoldScan: {
	
	_trgReadoutConfiguration.clearBeTrgTrigger(true);
	_trgReadoutConfiguration.beTrgSquirt(true);

	// Configurations 0 and 1 are pedestals and calibration
	if((iCfg%3)!=2) {
	  _trgReadoutConfiguration.readPeriod(1);

	// Configuration 2 is data
	} else {
	  _trgReadoutConfiguration.readPeriod(1);

	  // Bit 5 indicates faked trigger
	  if(u.bit(5)) {
	    
	    // Software trigger
	    if(u.bit(6)) {
	      _trgReadoutConfiguration.beTrgSoftTrigger(true);

            // Oscillator trigger
	    } else {
	    }
	  }
	}

	break;
      }
	
      default: {

	// We've missed something!
	assert(false);
	break;
      }
      }; // switch(type)
    }
      
    // Load configuration into record
    SubInserter inserter(r);
    inserter.insert<TrgReadoutConfigurationData>(_trgReadoutConfiguration);
    
    if(doPrint(r.recordType(),1)) {
      std::cout << " Number of TrgReadoutConfigurationData"
		<< "subrecords inserted = 1" << std::endl;
      _trgReadoutConfiguration.print(std::cout,"  ") << std::endl;
    }

    return true;
  }
    
    
  bool trgConfiguration(RcdRecord &r, const DaqConfigurationStart &d) {

    // Only put a configuration object into the record when enabled
    if(!_trgReadoutConfiguration.enable()) return true;

    CrcLocationData<CrcBeTrgConfigurationData> tcd(_location);

    // Set some generally useful parameters
    tcd.data()->generalEnable(1);
    tcd.data()->fifoIdleDepth(240);

    // Get the configuration number and version
    const unsigned iCfg(d.configurationNumberInRun());
    const unsigned char v(_runType.version());
    const UtlPack u(_runType.version());
      

    switch(_runType.type()) {

    case DaqRunType::trgTest: {
      break;
    }
    case DaqRunType::trgReadout: {
      break;
    }
    case DaqRunType::trgParameters: {
      break;
    }
    case DaqRunType::trgNoise: {
	tcd.data()->inputEnable(1<<24); // Trg osc
	tcd.data()->oscillationPeriod(4000); // 0.0001 sec
      break;
    }
    case DaqRunType::trgSpill: {
      tcd.data()->busyTimeout(8000); // 0.0002 sec
      tcd.data()->inputEnable(1<<24); // Trg osc
      tcd.data()->oscillationPeriod(4000); // 0.0001 sec
      break;
    }

    case DaqRunType::emcTrgTiming:
    case DaqRunType::emcTrgTimingScan: {
      break;
    }
      
    case DaqRunType::bmlNoise: {
      break;
    }

    // All the beam and cosmics runs

    case DaqRunType::trgBeam:
    case DaqRunType::trgCosmics:

    case DaqRunType::emcBeam:
    case DaqRunType::emcBeamHoldScan:
    case DaqRunType::emcCosmics:
    case DaqRunType::emcCosmicsHoldScan:

    case DaqRunType::ahcBeam:
    case DaqRunType::ahcBeamHoldScan:
    case DaqRunType::ahcBeamStage:
    case DaqRunType::ahcBeamStageScan:
    case DaqRunType::ahcCosmics:
    case DaqRunType::ahcCosmicsHoldScan:
      
    case DaqRunType::dhcBeam:
      
    case DaqRunType::tcmBeam:
    case DaqRunType::tcmBeamHoldScan:
    case DaqRunType::tcmCosmics:
    case DaqRunType::tcmCosmicsHoldScan:

    case DaqRunType::bmlBeam:
      
    case DaqRunType::beamTest:
    case DaqRunType::beamNoise:
    case DaqRunType::beamData:
    case DaqRunType::beamHoldScan:
    case DaqRunType::beamStage:
    case DaqRunType::beamStageScan:
      
    case DaqRunType::cosmicsTest:
    case DaqRunType::cosmicsNoise:
    case DaqRunType::cosmicsData:
    case DaqRunType::cosmicsHoldScan: {

      // Configurations 0 and 1 are pedestals and calibration
      if((iCfg%3)!=2) {
	if(u.bit(7)) {
	}

      // Configuration 2 is data
      } else {

	// Bit 5 indicates faked trigger
	if(u.bit(5)) {
	  tcd.data()->inputEnable(0);
	  
	  // Software trigger
	  if(u.bit(6)) {
	    
	  // Oscillator trigger
	  } else {
	    tcd.data()->oscillatorEnable(true);
	    tcd.data()->oscillationPeriod(40000); // 1kHz
	  }

	// Real triggers
	} else {

	  // Single input trigger
	  if(!u.bit(4)) {
	    tcd.data()->inputEnable(1<<u.halfByte(0));

	  // Internal coincidence of neighbouring inputs
	  } else {
	    tcd.data()->inputEnable(1<<25);
	    tcd.data()->generalEnable(1+(1<<8));
	    tcd.data()->andEnable(0,3<<u.halfByte(0));
	  }
	}
      }
      break;
    }
 
    default: {
      
      // We've missed something!
      assert(false);
      break;
    }
    }; // switch(type)
    
    
    // Load configuration into record
    SubInserter inserter(r);
    inserter.insert< CrcLocationData<CrcBeTrgConfigurationData> >(tcd);
    
    if(doPrint(r.recordType(),1)) {
      std::cout << " Number of CrcBeTrgConfigurationData"
		<< " subrecords inserted = 1" << std::endl;
      tcd.print(std::cout,"  ") << std::endl;
    }

    return true;
  }

/*
bool pedestalTrigger(unsigned pedType) {
  if(pedType==0) {
    vTcd[t].data()->inputEnable(1<<v);

  } else if(pedType==254) {
    vTrd[0].beTrgSoftTrigger(true);
    vTcd[t].data()->inputEnable(0);

  } else if(pedType==255) {
    vTcd[t].data()->oscillatorEnable(true);
    vTcd[t].data()->oscillationPeriod(40000); // 1kHz
  }
}
*/
    


protected:
  const CrcLocation _location;
  DaqRunType _runType;
  TrgReadoutConfigurationData _trgReadoutConfiguration;
//CrcBeTrgConfigurationData _crcBeTrgConfiguration;
};

#endif
