#ifndef AhcSlowReadoutDataV1_HH
#define AhcSlowReadoutDataV1_HH

#include <string>
#include <iostream>

class AhcSlowReadoutDataV1 {

public:
  enum {
    versionNumber=1
  };

  AhcSlowReadoutDataV1();

  bool parse(std::string r);
  
  time_t timeStamp() const;
  void   timeStamp(time_t t);

  unsigned moduleNumber() const;

  std::ostream& print(std::ostream &o, std::string s="") const;


private:
  time_t _timeStamp;
  unsigned _moduleNumber;
  double _cmbTemperatures[7];
  double _cmbVoltages[9];
  unsigned _ledSetting;
  double _cmbValues[4];
  double _hbabTemperatures[4];
  double _hbabVoltages[12];
};


#ifdef CALICE_DAQ_ICC

#include <time.h>
#include <cstring>


AhcSlowReadoutDataV1::AhcSlowReadoutDataV1() {
  memset(this,0,sizeof(AhcSlowReadoutDataV1));
}

bool AhcSlowReadoutDataV1::parse(std::string r) {
  _timeStamp=0xffffffff;
  _moduleNumber=255;
  memset(_cmbTemperatures,0,36*sizeof(double)+sizeof(unsigned));

  std::istringstream sin(r);
  if(!sin) return false;

  sin >> _timeStamp;
  if(!sin) return false;

  sin >> _moduleNumber;
  if(!sin) return false;

  for(unsigned i(0);i<7;i++) {
    sin >> _cmbTemperatures[i];
    if(!sin) return false;
  }

  for(unsigned i(0);i<9;i++) {
    sin >> _cmbVoltages[i];
    if(!sin) return false;
  }

  sin >> std::hex >> _ledSetting >> std::dec;
  if(!sin) return false;

  for(unsigned i(0);i<4;i++) {
    sin >> _cmbValues[i];
    if(!sin) return false;
  }

  for(unsigned i(0);i<4;i++) {
    sin >> _hbabTemperatures[i];
    if(!sin) return false;
  }

  for(unsigned i(0);i<12;i++) {
    sin >> _hbabVoltages[i];
    if(!sin) return false;
  }

  return true;
}

time_t AhcSlowReadoutDataV1::timeStamp() const {
  return _timeStamp;
}

void AhcSlowReadoutDataV1::timeStamp(time_t t) {
  _timeStamp=t;
}

unsigned AhcSlowReadoutDataV1::moduleNumber() const {
  return _moduleNumber;
}

std::ostream& AhcSlowReadoutDataV1::print(std::ostream &o, std::string s) const {
  o << s << "AhcSlowReadoutDataV1::print()" << std::endl;
  o << s << " Timestamp = " << _timeStamp << " = " << ctime(&_timeStamp);
  o << s << " Module number = " << _moduleNumber << std::endl;

  for(unsigned i(0);i<5;i++) {
    o << s << " CMB temperature " << std::setw(2) << i+1
      << "    = " << std::setw(5) << _cmbTemperatures[i] << " V" << std::endl;
  }
  o << s << " CMB temperature lower = " << std::setw(5)
    << _cmbTemperatures[5] << " V" << std::endl;
  o << s << " CMB temperature upper = " << std::setw(5)
    << _cmbTemperatures[6] << " V" << std::endl;

  o << s << " CMB calib U041  voltage = " << _cmbVoltages[0] << " V" << std::endl;
  o << s << " CMB 12V power   voltage = " << _cmbVoltages[1] << " V" << std::endl;
  o << s << " CMB 1.235V      voltage = " << _cmbVoltages[2] << " V" << std::endl;
  o << s << " CMB VLD upper   voltage = " << _cmbVoltages[3] << " V" << std::endl;
  o << s << " CMB VLB upper   voltage = " << _cmbVoltages[4] << " V" << std::endl;
  o << s << " CMB VLB upper?  voltage = " << _cmbVoltages[5] << " V" << std::endl;
  o << s << " CMB VLD for LED voltage = " << _cmbVoltages[6] << " V" << std::endl;
  o << s << " CMB 10V bias    voltage = " << _cmbVoltages[7] << " V" << std::endl;
  o << s << " CMB calib U051  voltage = " << _cmbVoltages[8] << " V" << std::endl;

  o << s << " LED setting = " << _ledSetting << std::endl;

  o << s << " CMB width  = " << _cmbValues[0] << " cm?" << std::endl;
  o << s << " CMB height = " << _cmbValues[1] << " cm?" << std::endl;
  o << s << " CMB 12V external = " << _cmbValues[2] << " V" << std::endl;
  o << s << " CMB 12V external = " << _cmbValues[3] << " V" << std::endl;

  o << s << " HBAB temperature top 1 = " << _hbabTemperatures[0] << " C" << std::endl;
  o << s << " HBAB temperature top 2 = " << _hbabTemperatures[1] << " C" << std::endl;
  o << s << " HBAB temperature bot 1 = " << _hbabTemperatures[2] << " C" << std::endl;
  o << s << " HBAB temperature bot 2 = " << _hbabTemperatures[3] << " C" << std::endl;

  o << s << " HBAB HV top voltage = " << _hbabVoltages[0] << " V" << std::endl;
  o << s << " HBAB HV bot voltage = " << _hbabVoltages[1] << " V" << std::endl;
  o << s << " HBAB HV top current = " << _hbabVoltages[2] << " A" << std::endl;
  o << s << " HBAB HV bot current = " << _hbabVoltages[3] << " A" << std::endl;
  o << s << " HBAB LV top voltage = " << _hbabVoltages[4] << " V" << std::endl;
  o << s << " HBAB LV bot voltage = " << _hbabVoltages[5] << " V" << std::endl;
  o << s << " HBAB LV top current = " << _hbabVoltages[6] << " A" << std::endl;
  o << s << " HBAB LV bot current = " << _hbabVoltages[7] << " A" << std::endl;

  for(unsigned i(0);i<4;i++) {
    o << s << " HBAB unknown " << std::setw(2) << i << "     = " << _hbabVoltages[i+8] << std::endl;
  }

  return o;
}

#endif
#endif
