#ifndef RunWriterDataMultiFile_HH
#define RunWriterDataMultiFile_HH

#include <cassert>

#include "DaqRunStart.hh"

#include "RcdUserRO.hh"
#include "RcdWriterDmy.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"

#include "SubAccessor.hh"


class RunWriterDataMultiFile : public RcdUserRO {

public:
  RunWriterDataMultiFile() : _writer(0) {
  }

  virtual ~RunWriterDataMultiFile() {
  }

  bool record(const RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << std::endl << "RunWriterDataMultiFile::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    if(r.recordType()==RcdHeader::startUp) return true;
    if(r.recordType()==RcdHeader::shutDown) return true;

    if(r.recordType()==RcdHeader::runStart) {

      // Access the DaqRunStart
      SubAccessor accessor(r);
      std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
      assert(v.size()==1);

      DaqRunType rt(v[0]->runType());
      _write=rt.writeRun();
      _ascii=rt.ascWriteRun();
      _runNumber=v[0]->runNumber();
      _fileNumber=0;

      assert(open());
    }

    // Write out records
    assert(_writer->write(r));

    if(r.recordType()==RcdHeader::runEnd) {
      assert(close());

    // Check for file size
    } else {

      unsigned limit(2000000000);
      /*
      if(r.recordType()==RcdHeader::configurationEnd) limit=1700000000;
      if(r.recordType()==RcdHeader::acquisitionEnd  ) limit=1800000000;
      if(r.recordType()==RcdHeader::spillEnd        ) limit=1900000000;
      if(r.recordType()==RcdHeader::spill           ) limit=1900000000;
      if(r.recordType()==RcdHeader::transferEnd     ) limit=1900000000;
      if(r.recordType()==RcdHeader::transfer        ) limit=1900000000;
      */
      
      if(_writer->numberOfBytes()>limit) {
	if(doPrint(r.recordType()),1) std::cout
	  << " Number of bytes written = " << _writer->numberOfBytes()
	  << " > " << limit << std::endl;
	
	RcdRecord fileContinuation;
	fileContinuation.initialise(RcdHeader::fileContinuation);
	fileContinuation.recordTime(r.recordTime()+UtlTimeDifference(0,1));
	
	assert(_writer->write(fileContinuation));
	assert(close());
	assert(open());
	assert(_writer->write(fileContinuation));
      }
    }
    
    return true;
  }

  bool open() {
    // Check for no writer
    assert(_writer==0);

    if(_write) {
      if(_ascii) _writer=new RcdWriterAsc;
      else       _writer=new RcdWriterBin;
    } else {
      _writer=new RcdWriterDmy;
    }
    
    std::ostringstream dout;
    dout << "data/run/Run";

    if(_runNumber<100000) dout << "0";
    if(_runNumber<10000)  dout << "0";
    if(_runNumber<1000)   dout << "0";
    if(_runNumber<100)    dout << "0";
    if(_runNumber<10)     dout << "0";
    dout << _runNumber << ".";

    if(_fileNumber<100)   dout << "0";
    if(_fileNumber<10)    dout << "0";
    dout << _fileNumber;

    // Open data output file
    assert(_writer->open(dout.str()));

    // Increment file number for next file
    _fileNumber++;

    return true;
  }

  bool close() {

    // Check for writer
    assert(_writer!=0);
    
    // Close file and delete writer
    assert(_writer->close());
    delete _writer;
    _writer=0;
    
    return true;
  }


private:
  RcdWriter *_writer;
  bool _write;
  bool _ascii;
  unsigned _runNumber;
  unsigned _fileNumber;
};

#endif
