#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <signal.h>
#include <fcntl.h>
#include <unistd.h>

#include <iostream>
#include <sstream>
#include <vector>
#include <cstdio>

// records/inc/utl
#include "UtlTime.hh"
#include "UtlArguments.hh"

// records/inc/rcd
#include "RcdArena.hh"
#include "RcdWriterAsc.hh"
#include "RcdWriterBin.hh"
#include "RcdWriterDmy.hh"
#include "RcdMultiUserRW.hh"
#include "RcdMultiUserRO.hh"

// records/inc/sub
#include "SubAccessor.hh"

// records/inc/daq
#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqAcquisitionStart.hh"
#include "DaqSoftware.hh"

#include "RunWriter.hh"

// records/inc/crc
#include "CrcLocationData.hh"
#include "CrcReadoutConfigurationData.hh"
#include "CrcReadoutStartupData.hh"
#include "TrgReadoutStartupData.hh"

// daquser/inc/hst
#include "HstNoise.hh"
#include "HstChipNoise.hh"
#include "HstChanNoise.hh"

// online/inc/daq
#include "DaqConfiguration.hh"
#include "DaqReadout.hh"

// online/inc/emc
#include "CrcReadout.hh"
#include "EmcReadout.hh"
#include "AhcReadout.hh"
#include "TrgReadout.hh"
//#include "AhcSlowReadout.hh"
#include "TrgConfiguration.hh"
#include "AhcConfiguration.hh"
#include "DaqConfiguration.hh"

// online/inc/dvr
//#include "DvrEmcTest.hh"
//#include "DvrAhcNoise.hh"

#include "DaqSlwControl.hh"
#include "ShmObject.hh"

using namespace std;

unsigned continueFlag(4);

void signalHandler(int signal) {
  std::cerr << "Process " << getpid() << " received signal "
	    << signal << std::endl;

  continueFlag=0;
  if(signal==SIGUSR1) continueFlag=1;
  if(signal==SIGUSR2) continueFlag=2;
}

/*
bool startup(RcdRecord &r) {
  std::cout << std::endl << "StartUp data being defined" << std::endl;

  CrcReadoutStartupData *crsd(inserter.insert<CrcReadoutStartupData>(true));
  crsd->crateNumber(0);
  crsd->pciNumber(0);
  crsd->print(std::cout);
    
  TrgReadoutStartupData *trsd(inserter.insert<TrgReadoutStartupData>(true));
  trsd->crateNumber(0);
  trsd->pciNumber(0);
  trsd->slotNumber(12);
  trsd->print(std::cout);

  return true;
}
*/

int main(int argc, const char **argv) {
  unsigned eTime(CALICE_DAQ_TIME);
  cout << argv[0] << " compiled at " << ctime((const time_t*)&eTime);

  UtlArguments argh(argc,argv);

  const bool useWriteDmy(argh.option('w',"Dummy output file"));
  const bool useWriteAsc(argh.option('a',"Ascii output file"));
  const bool doHistograms(argh.option('s',"Display histograms"));
  
  const unsigned nRun(argh.optionArgument('r',1,"Number of runs"));
  const unsigned printLevel(argh.optionArgument('p',0,"Print level"));
  const unsigned version(argh.optionArgument('v',0,"Run type version"));
  
  if(argh.help()) return 0;

  cout << "Command = " << argh.command() << endl << endl;
  
  if(doHistograms)  cout << "Histograms display selected" << endl;
  else              cout << "Histograms display not selected" << endl;
  if(useWriteDmy)   cout << "Dummy output selected" << endl;
  else {
    if(useWriteAsc) cout << "Ascii output selected" << endl;
    else            cout << "Binary output selected" << endl;
  }
  
  cout << "Print level set to " << printLevel << endl;
  cout << "Number of runs set to " << nRun << endl;
  cout << "Run type version set to " << version << endl;
  cout << endl;

  // Define record memory
  RcdArena arena;
  SubAccessor accessor(arena);
  RcdArena *aArray(new RcdArena[1024*4]);

  // Fill startup record
  arena.initialise(RcdHeader::startUp);
  arena.print(std::cout);

  SubInserter inserter(arena);

  DaqRunType rt(DaqRunType::crc,DaqRunType::crcIntDac,0,version);
  rt.dataRun(true);
  rt.writeRun(!useWriteDmy);
  rt.ascWriteRun(useWriteAsc);
  rt.histogramRun(doHistograms);

  DaqStartUp *dsd(inserter.insert<DaqStartUp>(true));
  dsd->runType(rt);
  dsd->print(std::cout);

  // Put in software information
  DaqSoftware *dsw(inserter.insert<DaqSoftware>(true));
  dsw->message(argh.command());
  dsw->print(std::cout);

  // Stop slow data run
  ShmObject<DaqSlwControl> shmSlwControl(DaqSlwControl::shmKey);
  DaqSlwControl *pDsc(shmSlwControl.payload());
  assert(pDsc!=0);
  pDsc->commandRegister();
  pDsc->command(0,DaqSlwControl::waiting);

  // Define lists of user modules
  RcdMultiUserRW vRrw;
  vRrw.printLevel(printLevel);
  RcdMultiUserRO vrub;
  vrub.printLevel(printLevel);
  
  // Add DAQ counter module
  DaqConfiguration dc;
  vRrw.addUser(dc);
  
  TrgConfiguration trgc(0xec,12);
  vRrw.addUser(trgc);
  TrgReadout trgr(1,0xec,12);
  vRrw.addUser(trgr);

#ifndef NEVER

  // Add driver module
#ifdef CALICE_DAQ_AHC
  AhcConfiguration ac;
  vRrw.addUser(ac);
  EmcConfiguration ec;
  vRrw.addUser(ec);

  AhcReadout ar(0);
  vRrw.addUser(ar);
  EmcReadout er(1);
  vRrw.addUser(er);
#else
  CrcConfiguration cc0(0xec);
  vRrw.addUser(cc0);
  CrcConfiguration cc1(0xac);
  vRrw.addUser(cc1);

  CrcReadout cr0(0,0xec);
  vRrw.addUser(cr0);
  CrcReadout cr1(1,0xac);
  vRrw.addUser(cr1);
#endif
#endif
  
  DaqReadout dr;
  vRrw.addUser(dr);

  // Add histogram module
  HstChipNoise *hcn(new HstChipNoise(true));
  //HstChanNoise *hcn(new HstChanNoise(true));
  if(doHistograms) vrub.addUser(*hcn);

  // Add writer module
  RunWriter dw(useWriteDmy,!useWriteAsc);
  vrub.addUser(dw);
  
  // Send StartUp
  vRrw.record(arena);
  vrub.record(arena);

  // Ignor Ctrl^C for a while
  signal(SIGINT,SIG_IGN);

  unsigned lastSlowRecord(0);
  
  // Loop over runs
  for(unsigned iRun(0);iRun<nRun && continueFlag>0;iRun++) {
    /*
    bool first(true);
    while(pDsc->status()==DaqSlwControl::initialising ||
	  pDsc->status()==DaqSlwControl::running) {
      if(first) {
	std::cout << "Waiting for slwData";
	first=false;
      }
      sleep(1);
      std::cout << "." << std::flush;
    }
    std::cout << endl;
    */

    // Now catch Ctrl^C
    signal(SIGINT,signalHandler);
    signal(SIGTERM,signalHandler);
    signal(SIGUSR1,signalHandler);
    signal(SIGUSR2,signalHandler);
    
    arena.initialise(RcdHeader::runStart);
    vRrw.record(arena);
    vrub.record(arena);
    
    // Access the DaqRunStart
    std::vector<const DaqRunStart*> v(accessor.extract<DaqRunStart>());
    assert(v.size()==1);
    
    // Loop over configurations
    const unsigned nCfg(v[0]->maximumNumberOfConfigurationsInRun());
    if(continueFlag==1) continueFlag=2;
    
    for(unsigned iCfg(0);iCfg<nCfg && continueFlag>1;iCfg++) {
      
      arena.initialise(RcdHeader::configurationStart);
      vRrw.record(arena);
      vrub.record(arena);
      
      // Access the DaqConfigurationStart
      std::vector<const DaqConfigurationStart*> v(accessor.extract<DaqConfigurationStart>());
      assert(v.size()==1);
      
      const UtlPack aMode(v[0]->acquisitionMode());
      const double slwDt(v[0]->minimumTimeBeforeSlowReadout().deltaTime());

      // Loop over acquisitions
      const unsigned nAcq(v[0]->maximumNumberOfAcquisitionsInConfiguration());
      if(continueFlag==2) continueFlag=3;
      
      for(unsigned iAcq(0);iAcq<nAcq && continueFlag>2;iAcq++) {
	
	UtlTime start;
	start.update();
	if(printLevel>4) cout << "About to do acquisition "
			      << std::setw(6) << iAcq
			      << " with mode = " << printHex(aMode) << endl;
	  
	arena.initialise(RcdHeader::acquisitionStart);
	vRrw.record(arena);
	vrub.record(arena);
	
	// Access the DaqAcquisitionStart
	std::vector<const DaqAcquisitionStart*> v(accessor.extract<DaqAcquisitionStart>());
	assert(v.size()==1);

	unsigned iEvt(0);
	const unsigned nEvt(v[0]->maximumNumberOfEventsInAcquisition());
	if(continueFlag==3) continueFlag=4;
	
	// Switch depending on spill state machine mode
	if(aMode.bit(0)) {
	  if(aMode.bit(1)) {
	    arena.initialise(RcdHeader::spill);
	    vRrw.record(arena);
	    vrub.record(arena);

	    // Access the DaqSpillEnd
	    std::vector<const DaqSpillEnd*> v(accessor.extract<DaqSpillEnd>());
	    assert(v.size()==1);
	    iEvt=v[0]->actualNumberOfEventsInSpill();

	  } else {
	    arena.initialise(RcdHeader::spillStart);
	    vRrw.record(arena);
	    vrub.record(arena);
	    
	    for(iEvt=0;iEvt<nEvt && continueFlag>3;iEvt++) {
	      arena.initialise(RcdHeader::preEvent);
	      vRrw.record(arena);
	      vrub.record(arena);
	    }

	    arena.initialise(RcdHeader::spillEnd);
	    vRrw.record(arena);
	    vrub.record(arena);
	  }
	  
	  if(aMode.bit(2)) {
	    aArray[0].initialise(RcdHeader::transfer);
	    vRrw.record(aArray[0]);
	    vrub.record(aArray[0]);

	  } else {
	    arena.initialise(RcdHeader::transferStart);
	    vRrw.record(arena);
	    vrub.record(arena);
	    
	    for(unsigned jEvt(0);jEvt<iEvt;jEvt++) {
	      arena.initialise(RcdHeader::postEvent);
	      vRrw.record(arena);
	      vrub.record(arena);
	    }
	    
	    arena.initialise(RcdHeader::transferEnd);
	    vRrw.record(arena);
	    vrub.record(arena);
	  }

	// Simple structure	  
	} else {

	  // Loop over events
	  for(iEvt=0;iEvt<nEvt && continueFlag>3;iEvt++) {
	    arena.initialise(RcdHeader::event);
	    vRrw.record(arena);
	    vrub.record(arena);
	  }      
	}

	// Do acquisition end
	arena.initialise(RcdHeader::acquisitionEnd);
	vRrw.record(arena);
	vrub.record(arena);
	hcn->update();

	UtlTime finish;
	finish.update();
	if(printLevel>4) cout << " Done " << iEvt << " events in "
			      << (finish-start).deltaTime() << " sec, "
			      << iEvt/(finish-start).deltaTime()
			      << " events/sec, "
			      << (finish-start).deltaTime()/iEvt 
			      << " secs/event" << std::endl;
	
	// Do a slow readout
	if((time(0)-lastSlowRecord)>=slwDt) {
	  arena.initialise(RcdHeader::slowReadout);
	  vRrw.record(arena);
	  vrub.record(arena);

	  lastSlowRecord=arena.recordTime().seconds();
	}
      }

      arena.initialise(RcdHeader::configurationEnd);
      vRrw.record(arena);
      vrub.record(arena);
    }

    arena.initialise(RcdHeader::runEnd);
    vRrw.record(arena);
    vrub.record(arena);

    // Ignor Ctrl-C again
    signal(SIGINT,SIG_IGN);
  }

  pDsc->command(0,DaqSlwControl::running);
  pDsc->unregister();
  return 0;
}
