#ifndef MpsConfiguration_HH
#define MpsConfiguration_HH

#include <iostream>
#include <fstream>

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubInserter.hh"
#include "SubAccessor.hh"



class MpsConfiguration  : public RcdUserRW {

public:
  MpsConfiguration(unsigned s) {
    _location.siteNumber(s);
    _location.label(1);
  }

  virtual ~MpsConfiguration() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "MpsConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
    
    
    // Check record type
    switch (r.recordType()) {
      
      // Run start 
    case RcdHeader::runStart: {

      // Access the IlcRunStart
      SubAccessor accessor(r);
      std::vector<const IlcRunStart*>
        v(accessor.extract<IlcRunStart>());
      assert(v.size()==1);

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsConfiguration::record()  runStart"
		  << ", found 1 IlcRunStart object" << std::endl;
	v[0]->print(std::cout," ") << std::endl;
      }

      _runType=v[0]->runType();

      break;
    }
      
      // Configuration start 
    case RcdHeader::configurationStart: {
 
      // Access the IlcConfigurationStart
      SubAccessor accessor(r);
      std::vector<const IlcConfigurationStart*>
        v(accessor.extract<IlcConfigurationStart>());
      assert(v.size()==1);

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsConfiguration::record()  configurationStart"
		  << ", found 1 IlcConfigurationStart object" << std::endl;
	v[0]->print(std::cout," ") << std::endl;
      }

      _configurationNumber=v[0]->configurationNumberInRun();

      // Get version number
      const UtlPack ver(_runType.version());

      SubInserter inserter(r);
      
      if(_runType.majorType()!=IlcRunType::mps) {

	// Put the overall configuration into the record
	MpsReadoutConfigurationData c;
	c.usbDaqEnable(false);
	inserter.insert<MpsReadoutConfigurationData>(c);

      } else {
      
	// Put the overall configuration into the record
	MpsReadoutConfigurationData c;
	c.usbDaqEnable(true);
	inserter.insert<MpsReadoutConfigurationData>(c);

	// Set up the USB_DAQ configuration vector
	std::vector< MpsLocationData<MpsUsbDaqConfigurationData> > u;
	u.push_back(MpsLocationData<MpsUsbDaqConfigurationData>());
	u[0].location(_location);
	
	// Set up the sensor PCB configuration
	std::vector< MpsLocationData<MpsPcb1ConfigurationData> > p;
	p.push_back(MpsLocationData<MpsPcb1ConfigurationData>());
	p[0].location(_location);

	// Set up the sensor configuration
	std::vector< MpsLocationData<MpsSensor1ConfigurationData> > s;
	s.push_back(MpsLocationData<MpsSensor1ConfigurationData>());
	s[0].location(_location);


	switch(_runType.type()) {
	  
	case IlcRunType::mpsTest: {
	  break;
	}
	case IlcRunType::mpsExpert: {
	  break;
	}
	case IlcRunType::mpsNoise: {
	  break;
	}
	case IlcRunType::mpsConfigurationTest: {
	  break;
	}
	case IlcRunType::mpsThreshold: {
	  p[0].data()->dac(0,16*ver.word());
	  break;
	}
	case IlcRunType::mpsThresholdScan: {
	  const unsigned steps(ver.word()+1);	  
	  p[0].data()->dac(0,(_configurationNumber%steps)*(4096/steps));
	  break;
	}
	case IlcRunType::mpsTrim: {
	  s[0].data()->trimQuadrant(ver.bits(4,5),ver.halfByte(0));
	  break;
	}
	case IlcRunType::mpsTrimScan: {
	  s[0].data()->trimQuadrant((_configurationNumber/16)%4,
				    _configurationNumber%16);
	  break;
	}
	case IlcRunType::mpsBeam: {
	  break;
	}
	case IlcRunType::mpsCosmics: {
	  break;
	}
	case IlcRunType::mpsSource: {
	  break;
	}
	case IlcRunType::mpsLaser: {
	  u[0].data()->numberOfBunchCrossings(2000);

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserPosition: {
	  u[0].data()->numberOfBunchCrossings(2000);

	  MpsLaserConfigurationData l;
	  l.stageX((int)(20.0*(ver.bits(0,3)-4)));
	  l.stageY((int)(20.0*(ver.bits(4,7)-4)));
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserPositionScan: {
	  u[0].data()->numberOfBunchCrossings(2000);

	  MpsLaserConfigurationData l;
	  l.numberOfPulses(16);

	  const unsigned steps(ver.word()+1);

	  l.stageX((int)(-4000.0+(_configurationNumber%steps)*(8000.0/steps)));
	  l.stageY((int)(-4000.0+((_configurationNumber/steps)%steps)*(8000.0/steps)));
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserThreshold: {
	  u[0].data()->numberOfBunchCrossings(2000);

	  p[0].data()->dac(0,16*ver.word());

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserThresholdScan: {
	  u[0].data()->numberOfBunchCrossings(2000);

	  const unsigned steps(ver.word()+1);
	  p[0].data()->dac(0,(_configurationNumber%steps)*(4096/steps));

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	  
	default: {
	  // We missed a run type
	  assert(false);
	  break;
	}
	};

	// Put the configurations into the record
	for(unsigned i(0);i<u.size();i++) {
	  u[i].usbDaqNumber(i);
	  //u[i].master(i==0);
	  inserter.insert< MpsLocationData<MpsUsbDaqConfigurationData> >(u[i]);
	}
	
	for(unsigned i(0);i<p.size();i++) {
	  p[i].usbDaqNumber(i);
	  inserter.insert< MpsLocationData<MpsPcb1ConfigurationData> >(p[i]);
	}

	for(unsigned i(0);i<s.size();i++) {
	  s[i].usbDaqNumber(i);
	  inserter.insert< MpsLocationData<MpsSensor1ConfigurationData> >(s[i]);
	}
      }

      break;
    }
      
    default: {
      break;
    }
    };
    
    return true;
  }

private:
  MpsLocation _location;

  IlcRunType _runType;
  unsigned _configurationNumber;
};

#endif
