#ifndef MpsSensor1ConfigurationData_HH
#define MpsSensor1ConfigurationData_HH

#include <iostream>
#include <fstream>

#include "UtlPack.hh"


class MpsSensor1ConfigurationData {

public:
  enum {
    versionNumber=0
  };

  MpsSensor1ConfigurationData();

  bool mask(unsigned x, unsigned y) const;
  void mask(unsigned x, unsigned y, bool b);
  void maskQuadrant(unsigned q, bool b);

  unsigned trim(unsigned x, unsigned y) const;
  void     trim(unsigned x, unsigned y, unsigned t);
  void     trimQuadrant(unsigned q, unsigned t);

  unsigned rowCheckBits(unsigned r) const;
  void     rowCheckBits(unsigned r, unsigned b);

  unsigned columnCheckBits(unsigned c) const;
  void     columnCheckBits(unsigned c, unsigned b);

  unsigned char* rowWritePtr(unsigned r);

  std::ostream& print(std::ostream &o=std::cout, std::string s="") const;


private:
  // 168bits/8 = 21bytes + 3 bytes check = 24 bytes per row
  // 5 bits/row * 168 rows + 3 bytes check/row = 864 rows

  // In terms of x=0-167 and y=0-167 in the sensor
  // [6]x32bits is x+24, [864] is 5*y+24
  // The 5 bits per pixel are: 0-3 threshold trim, 4 mask

  // A quadrant is 84x84 pixels
  // Quadrant 0 is x= 0- 83,y= 0- 83
  // Quadrant 1 is x=84-167,y= 0- 83
  // Quadrant 2 is x= 0- 83,y=84-167
  // Quadrant 3 is x=84-167,y=84-167
  
  UtlPack _data[864][6];
};


#ifdef CALICE_DAQ_ICC

#include <cstring>

#include "UtlPrintHex.hh"


MpsSensor1ConfigurationData::MpsSensor1ConfigurationData() {
  memset(this,0,sizeof(MpsSensor1ConfigurationData));

  for(unsigned i(0);i<168;i++) {
    for(unsigned j(0);j<168;j++) {
      trim(i,j,8);
    }
  }

  maskQuadrant(0,false);
  maskQuadrant(1,false);
  maskQuadrant(2,false);
  maskQuadrant(3,false);
}

bool MpsSensor1ConfigurationData::mask(unsigned x, unsigned y) const {
  
  // Check x and y are in the right range for the sensor
  assert(x<168 && y<168);
  
  // Return correct bit = 4
  return _data[5*y+24+4][(x+24)/32].bit((x+24)%32);
}

void MpsSensor1ConfigurationData::mask(unsigned x, unsigned y, bool b) {
  
  // Check x and y are in the right range for the sensor
  assert(x<168 && y<168);
  
  // Set correct bit = 4
  _data[5*y+24+4][(x+24)/32].bit((x+24)%32,b);
}

void MpsSensor1ConfigurationData::maskQuadrant(unsigned q, bool b) {

  // Check for only four quadrants
  assert(q<4);

  // Loop over correct x and y ranges
  for(unsigned x(84*(q%2));x<84*((q%2)+1);x++) {
    for(unsigned y(84*(q/2));y<84*((q/2)+1);y++) {
      mask(x,y,b);
    }
  }
}

unsigned MpsSensor1ConfigurationData::trim(unsigned x, unsigned y) const {
  
  // Check x and y are in the right range for the sensor
  assert(x<168 && y<168);
  
  // Get each of the 0-3 bits in turn and pack into the result
  UtlPack result;
  for(unsigned i(0);i<4;i++) {
    result.bit(i,_data[5*y+24+i][(x+24)/32].bit((x+24)%32));
  }

  // Send back the completed value
  return result.word();
}

void MpsSensor1ConfigurationData::trim(unsigned x, unsigned y, unsigned t) {

  // Check x, y and t are in the right range for the sensor
  assert(x<168 && y<168 && t<16);
  
  // Get each of the 4 bits in turn and pack into the correct location
  UtlPack u(t);
  for(unsigned i(0);i<4;i++) {
    _data[5*y+24+i][(x+24)/32].bit((x+24)%32,u.bit(i));
  }
}

void MpsSensor1ConfigurationData::trimQuadrant(unsigned q, unsigned t) {


  // Check for only four quadrants
  assert(q<4);

  // Loop over correct x and y ranges
  for(unsigned x(84*(q%2));x<84*((q%2)+1);x++) {
    for(unsigned y(84*(q/2));y<84*((q/2)+1);y++) {
      trim(x,y,t);
    }
  }
}

unsigned MpsSensor1ConfigurationData::rowCheckBits(unsigned r) const {
  
  // Check r is in the right range for the sensor
  assert(r<864);

  // Get each of the 0-2 bytes in turn and pack into the result
  UtlPack result;
  for(unsigned i(0);i<3;i++) {
    result.byte(i,_data[r][0].byte(i));
  }

  // Send back the completed value
  return result.word();
}

void MpsSensor1ConfigurationData::rowCheckBits(unsigned r, unsigned b) {
  
  // Check r and b are in the right range for the sensor
  assert(r<864 && b<0x01000000);
  
  // Get each of the 0-2 bytes in turn and pack into the result
  UtlPack bits(b);
  for(unsigned i(0);i<3;i++) {
    _data[r][0].byte(i,bits.byte(i));
  }
}

unsigned MpsSensor1ConfigurationData::columnCheckBits(unsigned c) const {
  
  // Check r is in the right range for the sensor
  assert(c<192);

  // Get each of the 0-23 bits in turn and pack into the result
  UtlPack result;
  for(unsigned i(0);i<24;i++) {
    result.bit(i,_data[i][c/32].bit(c%32));
  }

  // Send back the completed value
  return result.word();
}

void MpsSensor1ConfigurationData::columnCheckBits(unsigned c, unsigned b) {
  
  // Check r and b are in the right range for the sensor
  assert(c<192 && b<0x01000000);
  
  // Get each of the 0-23 bits in turn and pack into the result
  UtlPack bits(b);
  for(unsigned i(0);i<24;i++) {
    _data[i][c/32].bit(c%32,bits.bit(i));
  }
}

unsigned char* MpsSensor1ConfigurationData::rowWritePtr(unsigned r) {
  
  // Check r is in the right range for the sensor
  assert(r<864);

  // Return pointer to beginning of row information
  return (unsigned char*)_data[r];
}

std::ostream& MpsSensor1ConfigurationData::print(std::ostream &o, std::string s) const {
  o << s << "MpsSensor1ConfigurationData::print()" << std::endl;

  for(unsigned q(0);q<4;q++) {
    o << s << " Quadrant = " << q 
      << ", pixels x = "
      << std::setw(2) << 84*(q%2) << "-"
      << std::setw(3) << 84*((q%2)+1)
      << ", y = "
      << std::setw(2) << 84*(q/2) << "-"
      << std::setw(3) << 84*((q/2)+1)
      << ", threshold trim/mask settings"  << std::endl << std::hex;

    for(unsigned y(84*(q/2));y<84*((q/2)+1);y++) {
      o << s << "  ";
      for(unsigned x(84*(q%2));x<84*((q%2)+1);x++) {
	if(mask(x,y)) o << ".";
	else          o << trim(x,y);
      }
      o << std::endl;
    }
    o << std::dec;
  }

  return o;
}

#endif
#endif
