#ifndef HstFlags_HH
#define HstFlags_HH

#include <iostream>
#include <vector>

#include "DaqRunStart.hh"
#include "DaqConfigurationStart.hh"
#include "DaqAcquisitionStart.hh"
#include "DaqEvent.hh"
#include "RcdRecord.hh"
#include "SubAccessor.hh"
#include "HstArray.hh"


class HstFlags {

public:
  enum Level {
    job,
    run,
    configuration,
    acquisition,
    event
  };

  HstFlags() {
    _validRun=false;
    _validConfiguration=false;
    _validAcquisition=false;
    _validEvent=false;

    _inJob=true;
    _inRun=false;
    _inConfiguration=false;
    _inAcquisition=false;

    _resetLevel=configuration;
  }

  std::string label(Level l=event) const {
    if(l==event)         return "Evt";
    if(l==acquisition)   return "Acq";
    if(l==configuration) return "Cfg";
    if(l==run)           return "Run";
    if(l==job)           return "Job";
    return "???";
  }

  std::string title() const {
    std::ostringstream sout;
    if(!_validRun) return sout.str();

    sout << "Run " << _drs.runNumber() << ", ";
    if(!_validConfiguration) return sout.str();

    sout << "Cfg " << _dcs.configurationNumberInRun() << ", ";
    if(!_validAcquisition) return sout.str();

    sout << "Acq " << _das.acquisitionNumberInRun()
	 << "/"    << _das.acquisitionNumberInConfiguration() << ", ";
    if(!_validEvent) return sout.str();

    sout << "Evt " << _dev.eventNumberInRun()
	 << "/"    << _dev.eventNumberInConfiguration()
	 << "/"    << _dev.eventNumberInAcquisition() << ", ";

    return sout.str(); 
  }

  bool ready(Level l) {
    if(l==event)         return                                             _validEvent;
    if(l==acquisition)   return !_inAcquisition   && _validAcquisition   && _validEvent;
    if(l==configuration) return !_inConfiguration && _validConfiguration && _validEvent;
    if(l==run)           return !_inRun           && _validRun           && _validEvent;
    return false;
  }

  void fill(const RcdRecord &r) {

    switch (r.recordType()) {

    case RcdHeader::shutDown: {
      _inJob=false;
      break;
    }

    case RcdHeader::runStart: {
      SubAccessor accessor(r);
      std::vector<const DaqRunStart*> 
	v(accessor.access<DaqRunStart>());
      if(v.size()>0) _drs=*(v[0]);
      //_drs.print(std::cout);

      _validRun=true;
      _validConfiguration=false;
      _validAcquisition=false;
      _validEvent=false;
      _inRun=true;
      break;
    }

    case RcdHeader::runEnd: {
      _inRun=false;
      break;
    }

    case RcdHeader::configurationStart: {
      SubAccessor accessor(r);
      std::vector<const DaqConfigurationStart*> 
	v(accessor.access<DaqConfigurationStart>());
      if(v.size()>0) _dcs=*(v[0]);
      //_dcs.print(std::cout);
      _validConfiguration=true;
      _validAcquisition=false;
      _validEvent=false;
      _inConfiguration=true;
      break;
    }

    case RcdHeader::configurationEnd: {
      _inConfiguration=false;
      break;
    }

    case RcdHeader::acquisitionStart: {
      SubAccessor accessor(r);
      std::vector<const DaqAcquisitionStart*> 
	v(accessor.access<DaqAcquisitionStart>());
      if(v.size()>0) _das=*(v[0]);
      //_das.print(std::cout);
      _validAcquisition=true;
      _validEvent=false;
      _inAcquisition=true;
      break;
    }

    case RcdHeader::acquisitionEnd: {
      _inAcquisition=false;
      break;
    }

    case RcdHeader::event: {
      SubAccessor accessor(r);
      std::vector<const DaqEvent*> 
	v(accessor.access<DaqEvent>());
      if(v.size()>0) _dev=*(v[0]);
      //_dev.print(std::cout);

      // This must be reset when event has been processed completely
      _validEvent=false;
      break;
    }

    default: {
      break;
    }
    };
  }

  std::ostream& print(std::ostream &o, std::string s) const {
    o << s << "HstFlags::print()" << std::endl;
    _drs.print(o,s+" ") << std::endl;
    _dcs.print(o,s+" ") << std::endl;
    _das.print(o,s+" ") << std::endl;
    _dev.print(o,s+" ") << std::endl;
    return o;
  }

  // Keep all data public for access speed
  bool _inJob,_inRun,_inConfiguration,_inAcquisition;
  bool _validRun,_validConfiguration,_validAcquisition,_validEvent;

  Level _resetLevel;
  bool _resetNow;

  DaqRunStart           _drs;
  DaqConfigurationStart _dcs;
  DaqAcquisitionStart   _das;
  DaqEvent              _dev;

  HstArray<10000> _timers[RcdHeader::endOfRecordTypeEnum];
};

#endif
