#ifndef MpsAnalysisBase_HH
#define MpsAnalysisBase_HH

#include <cassert>

#include "TFile.h"
#include "TF1.h"
#include "TH1F.h"
#include "TH2F.h"
#include "TProfile.h"

#include "RcdUserRO.hh"
#include "SubAccessor.hh"


class MpsAnalysisBase : public RcdUserRO {

public:
  MpsAnalysisBase(const std::string &name) :
    _mpsAnalysisName(name), _validRunType(true), _rootFile(0) {
  }

  virtual ~MpsAnalysisBase() {
    endRoot();
  }

  const std::string& mpsAnalysisName() const {
    return _mpsAnalysisName;
  }

  virtual bool mpsAnalysisValidRun(IlcRunType::Type t) const {
    return false;
  }

  virtual bool runStart(const RcdRecord &r) {
    return true;
  }

  virtual bool runEnd(const RcdRecord &r) {
    return true;
  }

  virtual bool configurationStart(const RcdRecord &r) {
    return true;
  }

  virtual bool configurationEnd(const RcdRecord &r) {
    return true;
  }

  virtual bool slowReadout(const RcdRecord &r) {
    return true;
  }

  virtual bool bunchTrain(const RcdRecord &r) {
    return true;
  }

  bool record(const RcdRecord &r) {
    if(doPrint(r.recordType()) && _validRunType) {
      std::cout << _mpsAnalysisName << "::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }

    SubAccessor accessor(r);

    switch (r.recordType()) {
  
    case RcdHeader::runStart: {
      std::vector<const IlcRunStart*>
        vs(accessor.access<IlcRunStart>());
      assert(vs.size()==1);
      _runStart=*(vs[0]);

      _validRunType=mpsAnalysisValidRun(_runStart.runType().type());

      if(_validRunType) {
	if(doPrint(r.recordType(),1)) _runStart.print(std::cout," ") << std::endl;

        std::ostringstream sFile;
        sFile << _mpsAnalysisName << std::setfill('0') << std::setw(6)
              << vs[0]->runNumber() << ".root";

        if(doPrint(r.recordType())) {
	  std::cout << _mpsAnalysisName << "::record()  Creating ROOT file "
		    << sFile.str() << std::endl << std::endl;
	}
        _rootFile = new TFile(sFile.str().c_str(),"RECREATE");

	std::ostringstream rLabel;
	rLabel << "Run" << std::setfill('0') << std::setw(6) << vs[0]-> runNumber();
	_runLabel=rLabel.str();

	std::ostringstream rTitle;
	rTitle << "Run " << vs[0]-> runNumber();
	_runTitle=rTitle.str();

	std::vector<const MpsLocationData<MpsUsbDaqRunData>* >
	  v(accessor.access< MpsLocationData<MpsUsbDaqRunData> >());

	_vLocation.clear();
	_vUsbDaqConfigurationData.clear();
	_vPcbConfigurationData.clear();
	_vSensorConfigurationData.clear();

	for(unsigned i(0);i<v.size();i++) {
	  if(doPrint(r.recordType(),2)) v[i]->print(std::cout) << std::endl;

	  _vLocation.push_back(v[i]->location());

	  std::ostringstream sLabel;
	  sLabel << "Sensor" << std::setfill('0') << std::setw(2) << (unsigned)_vLocation[i].sensorId();
	  _sensorLabel.push_back(sLabel.str());

	  std::ostringstream sTitle;
	  sTitle << "Sensor " << (unsigned)_vLocation[i].sensorId();
	  _sensorTitle.push_back(sTitle.str());
	}

	if(doPrint(r.recordType(),2)) {
	  std::cout << " Run titles set for "
		    << _vLocation.size() << " USB_DAQs" << std::endl
		    << "  Run title = " << _runTitle << std::endl;
	  
	  for(unsigned i(0);i<_vLocation.size();i++) {
	    std::cout << "  USB_DAQ " << i << " sensor title = "
		      << _sensorTitle[i] << std::endl;
	  }
	  std::cout << std::endl;
	}

	_rootFile->cd();
	runStart(r);

      } else {
        if(doPrint(r.recordType(),1)) {
	  std::cout << _mpsAnalysisName << "::record()  Run ignored"
		    << std::endl << std::endl;
	}
      }

      break;
    }
      
    case RcdHeader::runEnd: {
      if(_validRunType) {
	std::vector<const IlcRunEnd*>
	  ve(accessor.access<IlcRunEnd>());
	assert(ve.size()==1);
	if(doPrint(r.recordType(),1)) ve[0]->print(std::cout) << std::endl;

	_rootFile->cd();
	runEnd(r);
	endRoot();
      }

      // Reset switch
      _validRunType=true;
      break;
    }
  
    case RcdHeader::configurationStart: {
      if(_validRunType) {
	std::vector<const IlcConfigurationStart*>
	  vs(accessor.access<IlcConfigurationStart>());
	assert(vs.size()==1);
	_configurationStart=*(vs[0]);
	if(doPrint(r.recordType(),1)) vs[0]->print(std::cout) << std::endl;

	std::ostringstream sLabel;
	sLabel << "Cfg" << std::setfill('0') << std::setw(6)
	       << vs[0]->configurationNumberInRun();
	_cfgLabel=sLabel.str();
	
	std::ostringstream sTitle;
	sTitle << _runTitle << ", Cfg " << vs[0]->configurationNumberInRun();
	_cfgTitle=sTitle.str();
	
	for(unsigned i(0);i<_vLocation.size();i++) {
	  std::ostringstream sLabel;
	  sLabel << "Sensor"  << std::setfill('0') << std::setw(2) 
		 << (unsigned)_vLocation[i].sensorId() << _cfgLabel;
	  _cfgSensorLabel.push_back(sLabel.str());
	  
	  std::ostringstream sTitle;
	  sTitle << _runTitle << ", " << _sensorTitle[i] << ", " << _cfgTitle;
	  _cfgSensorTitle.push_back(sTitle.str());
	}

	if(doPrint(r.recordType(),2)) {
	  std::cout << " Configuration labels and titles set for "
		    << _vLocation.size() << " USB_DAQs" << std::endl
		    << "  Overall label = " << _cfgLabel << ", title = " 
		    << _cfgTitle << std::endl;

	  for(unsigned i(0);i<_vLocation.size();i++) {
	    std::cout << "  USB_DAQ " << i << " label = " << _cfgSensorLabel[i]
		      << ", title = " << _cfgSensorTitle[i] << std::endl;
	  }
	  std::cout << std::endl;
	}

	std::vector<const MpsLocationData<MpsUsbDaqConfigurationData>* >
	  w(accessor.access< MpsLocationData<MpsUsbDaqConfigurationData> >());
	
	unsigned read(0);
	for(unsigned i(0);i<w.size();i++) {
	  if(doPrint(r.recordType(),2)) w[i]->print() << std::endl;

	  if(!w[i]->write()) {
	    _usbDaqConfigurationData=*(w[i]->data());

	    if(_configurationStart.configurationNumberInRun()==0) {
	      _vUsbDaqConfigurationData.push_back(*(w[i]->data()));
	    } else {
	      _vUsbDaqConfigurationData[read]=*(w[i]->data());	      
	    }
	    read++;
	  }
	}
	assert(read==_vLocation.size());
	

	std::vector<const MpsLocationData<MpsPcb1ConfigurationData>* >
	  v(accessor.access< MpsLocationData<MpsPcb1ConfigurationData> >());

	read=0;
	for(unsigned i(0);i<v.size();i++) {
	  if(doPrint(r.recordType(),2)) v[i]->print() << std::endl;
	  
	  if(!v[i]->write()) {
	    _pcbConfigurationData=*(v[i]->data());

	    if(_configurationStart.configurationNumberInRun()==0) {
	      _vPcbConfigurationData.push_back(*(v[i]->data()));
	    } else {
	      _vPcbConfigurationData[read]=*(v[i]->data());
	    }
	    read++;
	  }
	}
	assert(read==_vLocation.size());


	std::vector<const MpsLocationData<MpsSensor1ConfigurationData>* >
	  z(accessor.access< MpsLocationData<MpsSensor1ConfigurationData> >());

	for(unsigned i(0);i<z.size();i++) {
	  if(doPrint(r.recordType(),2)) z[i]->print() << std::endl;
	  //assert(z[i]->write());
	  if(z[i]->write()) {

	    _sensorConfigurationData=*(z[i]->data());
	    
	    for(unsigned j(0);j<_vLocation.size();j++) {
	      MpsLocation l(z[i]->location());
	      l.write(false);
	      if(_vLocation[j]==l) {
		if(_configurationStart.configurationNumberInRun()==0) {
		  _vSensorConfigurationData.push_back(*(z[i]->data()));
		  
		} else {
		_vSensorConfigurationData[j]=*(z[i]->data());
		}
	      }
	    }
	  }
	}

	_rootFile->cd();
	configurationStart(r);
      }

      break;
    }
      
    case RcdHeader::configurationEnd: {
      if(_validRunType) {
	std::vector<const IlcConfigurationEnd*>
	  ve(accessor.access<IlcConfigurationEnd>());
	assert(ve.size()==1);
	if(doPrint(r.recordType(),1)) ve[0]->print(std::cout) << std::endl;

	_rootFile->cd();
	configurationEnd(r);
      }
      break;
    }
  
    case RcdHeader::slowReadout: {
      if(_validRunType) {
	std::vector<const IlcSlowReadout*>
	  vs(accessor.access<IlcSlowReadout>());
	assert(vs.size()==1);
	_slowReadout=*(vs[0]);
	if(doPrint(r.recordType())) vs[0]->print() << std::endl;

	_rootFile->cd();
	slowReadout(r);
      }
      break;
    }
      
    case RcdHeader::bunchTrain: {
      if(_validRunType) {
	std::vector<const IlcBunchTrain*>
	  vb(accessor.access<IlcBunchTrain>());
	assert(vb.size()==1);
	_bunchTrain=*(vb[0]);
	if(doPrint(r.recordType())) vb[0]->print() << std::endl;

	_rootFile->cd();
	bunchTrain(r);
      }
      break;
    }
      
      
    default: {
      break;
    }
    };
    
    return true;
  }

  virtual void endRoot() {
    if(_rootFile!=0) {
      _rootFile->Write();
      _rootFile->Close();
      delete _rootFile;
      _rootFile=0;
    }
  }


private:
  std::string _mpsAnalysisName;
  bool _validRunType;

protected:
  TFile* _rootFile;

  IlcRunStart _runStart;
  std::string _runLabel;
  std::string _runTitle;
  std::vector<std::string> _sensorLabel;
  std::vector<std::string> _sensorTitle;

  IlcConfigurationStart _configurationStart;
  std::string _cfgLabel;
  std::string _cfgTitle;
  std::vector<std::string> _cfgSensorLabel;
  std::vector<std::string> _cfgSensorTitle;

  IlcSlowReadout _slowReadout;
  IlcBunchTrain _bunchTrain;

  MpsUsbDaqConfigurationData _usbDaqConfigurationData;
  MpsPcb1ConfigurationData _pcbConfigurationData;
  MpsSensor1ConfigurationData _sensorConfigurationData;

  std::vector<MpsLocation> _vLocation;
  std::vector<MpsUsbDaqConfigurationData>  _vUsbDaqConfigurationData;
  std::vector<MpsPcb1ConfigurationData>    _vPcbConfigurationData;
  std::vector<MpsSensor1ConfigurationData> _vSensorConfigurationData;
};

#endif
