#ifndef MpsConfiguration_HH
#define MpsConfiguration_HH

#include <iostream>
#include <fstream>

#include "RcdHeader.hh"
#include "RcdUserRW.hh"

#include "SubInserter.hh"
#include "SubAccessor.hh"



class MpsConfiguration  : public RcdUserRW {

public:
  MpsConfiguration(unsigned s) {
    _location.siteNumber(s);
    _location.usbDaqBroadcast(true);
    _location.sensorBroadcast(true);
    _location.label(1);
  }

  virtual ~MpsConfiguration() {
  }

  virtual bool record(RcdRecord &r) {
    if(doPrint(r.recordType())) {
      std::cout << "MpsConfiguration::record()" << std::endl;
      r.RcdHeader::print(std::cout," ") << std::endl;
    }
    
    
    // Check record type
    switch (r.recordType()) {
      
      // Run start 
    case RcdHeader::runStart: {

      // Access the IlcRunStart
      SubAccessor accessor(r);
      std::vector<const IlcRunStart*>
        v(accessor.extract<IlcRunStart>());
      assert(v.size()==1);

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsConfiguration::record()  runStart"
		  << ", found 1 IlcRunStart object" << std::endl;
	v[0]->print(std::cout," ") << std::endl;
      }

      _runType=v[0]->runType();

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsConfiguration::record()  runStart"
		  << ", has " << _vLocation.size()
		  << " MpsLocation objects stored" << std::endl;

	if(doPrint(r.recordType(),2)) {
	  for(unsigned i(0);i<_vLocation.size();i++) {
	    _vLocation[i].print(std::cout," ") << std::endl;
	  }
	}
      }

      break;
    }
      
      // Configuration start 
    case RcdHeader::configurationStart: {
 
      // Access the IlcConfigurationStart
      SubAccessor accessor(r);
      std::vector<const IlcConfigurationStart*>
        v(accessor.extract<IlcConfigurationStart>());
      assert(v.size()==1);

      if(doPrint(r.recordType(),1)) {
	std::cout << "MpsConfiguration::record()  configurationStart"
		  << ", found 1 IlcConfigurationStart object" << std::endl;
	v[0]->print(std::cout," ") << std::endl;
      }

      _configurationNumber=v[0]->configurationNumberInRun();

      // Get version number
      const UtlPack ver(_runType.version());

      SubInserter inserter(r);

      // Put the overall configuration control into the record
      MpsReadoutConfigurationData *c(inserter.insert<MpsReadoutConfigurationData>(true));

      // Set up the USB_DAQ configuration vector
      std::vector< MpsLocationData<MpsUsbDaqConfigurationData> > u;

      // Set up the sensor PCB configuration
      std::vector< MpsLocationData<MpsPcb1ConfigurationData> > p;
      
      // Set up the sensor configuration
      std::vector< MpsLocationData<MpsSensor1ConfigurationData> > s;


      // Now do run type specific settings
      if(_runType.majorType()==IlcRunType::daq ||
	 _runType.majorType()==IlcRunType::slw) {

	c->usbDaqEnableAll(false);

	// Add a PCB object for safety
	if(_configurationNumber==0) {
	  p.push_back(MpsLocationData<MpsPcb1ConfigurationData>());
	  p[0].location(_location);
	}
      }
      
      if(_runType.majorType()==IlcRunType::usb) {

	c->usbDaqEnableAll(true);

	// Add a USB_DAQ configuration object
	u.push_back(MpsLocationData<MpsUsbDaqConfigurationData>());
	u[0].location(_location);

        // Set location specific PMT thresholds
        if(MPS_LOCATION==MpsLocation::birmingham) {
          u[0].data()->discriminatorThreshold(0,152);
          u[0].data()->discriminatorThreshold(1,152);
        }
	
	switch(_runType.type()) {
	  
	case IlcRunType::usbTest: {
	  break;
	}
	case IlcRunType::usbPmtThreshold: {
	  u[0].data()->spillModeHistoryEnable(true);

	  // Turn off sensor data on first configuration
	  if(_configurationNumber==0) {
	    for(unsigned i(0);i<s.size();i++) {
	      s[i].data()->maskSensor(true);
	    }
	  } else {
	    s.clear();
	  }

	  u[0].data()->discriminatorThreshold(0,ver.word());
	  u[0].data()->discriminatorThreshold(1,ver.word());
	  break;
	}
	case IlcRunType::usbPmtThresholdScan: {
	  u[0].data()->spillModeHistoryEnable(true);

	  // Turn off sensor data on first configuration
	  if(_configurationNumber==0) {
	    for(unsigned i(0);i<s.size();i++) {
	      s[i].data()->maskSensor(true);
	    }
	  } else {
	    s.clear();
	  }

	  const unsigned steps((ver.word())%100+1);
	  const unsigned step(_configurationNumber%steps);
	  u[0].data()->discriminatorThreshold(0,100+step*(100/steps));
	  u[0].data()->discriminatorThreshold(1,100+step*(100/steps));
	  break;
	}
	  
	default: {
	  // We missed a run type
	  assert(false);
	  break;
	}
	};

	if(_configurationNumber==0) {

	  // Add a PCB object for safety
	  p.push_back(MpsLocationData<MpsPcb1ConfigurationData>());
	  p[0].location(_location);

	  // Add a fully masked sensor object
	  s.push_back(MpsLocationData<MpsSensor1ConfigurationData>());
	  s[0].location(_location);
	  s[0].data()->maskSensor(true);
	}
      }

      if(_runType.majorType()==IlcRunType::mps) {
	c->usbDaqEnableAll(true);

	// Add one of every object
	u.push_back(MpsLocationData<MpsUsbDaqConfigurationData>());
	u[0].location(_location);

        // Set location specific PMT thresholds
        if(MPS_LOCATION==MpsLocation::birmingham) {
          u[0].data()->discriminatorThreshold(0,152);
          u[0].data()->discriminatorThreshold(1,152);
        }
	
	p.push_back(MpsLocationData<MpsPcb1ConfigurationData>());
	p[0].location(_location);

	// Add one per sensor
	for(unsigned i(0);i<_vLocation.size();i++) {
	  s.push_back(MpsLocationData<MpsSensor1ConfigurationData>());
	  s[i].location(_vLocation[i]);

	  std::ostringstream sout;
	  sout << "cfg/MpsSensor1ConfigurationDataSensor" << std::setfill('0')
	       << std::setw(2) << (unsigned)_vLocation[i].sensorId() << ".cfg";

	  std::cout << sout.str() << std::endl;
	  s[i].data()->readFile(sout.str());
	}

	switch(_runType.type()) {
	  
	case IlcRunType::mpsTest: {
	  break;
	}
	case IlcRunType::mpsExpert: {
	  break;
	}
	case IlcRunType::mpsNoise: {
	  if(_configurationNumber==0) {
	    for(unsigned i(0);i<s.size();i++) {
	      for(unsigned j(0);j<4;j++) {
		if(ver.bit(j)) s[i].data()->maskRegion(j,true);
	      }
	      if(ver.bit(4)) s[i].data()->maskInvert();
	    }
	  } else {
	    s.clear();
	  }
	  
	  if(ver.bit(5)) u[0].data()->pixelEnable12(false);
	  if(ver.bit(6)) u[0].data()->pixelEnable34(false);

	  u[0].data()->hitOverride(ver.bit(7));
	  break;
	}
	case IlcRunType::mpsPcbConfigurationScan: {
	  if(ver.bits(0,4)<32) {
	    unsigned dac(ver.bits(0,4));

	    if(dac==21 || dac==22 || dac==23 ||
	       dac==25 || dac==27 || dac==30) {

	      if(dac==21) {
		p[0].data()->samplerThreshold(p[0].data()->samplerThresholdValue(),
					      p[0].data()->samplerThresholdCommonMode()
					      -100+2*(_configurationNumber%100));
	      }	      
	      if(dac==22) {
		p[0].data()->debugThreshold(p[0].data()->debugThresholdValue(),
					    p[0].data()->debugThresholdCommonMode()
					    -100+2*(_configurationNumber%100));
	      }	      
	      if(dac==23) {
		p[0].data()->shaperThreshold(p[0].data()->shaperThresholdValue(),
					     p[0].data()->shaperThresholdCommonMode()
					     -100+2*(_configurationNumber%100));
	      }	      
	      if(dac==25) {
		p[0].data()->samplerThreshold(p[0].data()->samplerThresholdValue()
					      -100+2*(_configurationNumber%100),
					      p[0].data()->samplerThresholdCommonMode());
	      }	      
	      if(dac==27) {
		p[0].data()->debugThreshold(p[0].data()->debugThresholdValue()
					    -100+2*(_configurationNumber%100),
					    p[0].data()->debugThresholdCommonMode());
	      }
	      if(dac==30) {
		p[0].data()->shaperThreshold(p[0].data()->shaperThresholdValue()
					     -100+2*(_configurationNumber%100),
					     p[0].data()->shaperThresholdCommonMode());
	      }	      

	    } else {


	      if(dac!=24) {
		p[0].data()->dac(dac,p[0].data()->dac(dac)-100+2*(_configurationNumber%100));
	      } else {
		p[0].data()->dac(dac,p[0].data()->dac(dac)-1000+20*(_configurationNumber%100));
	      }
	    }



	  } else {
	    const unsigned steps(2*ver.halfByte(0)+1);	  
	    
	    if(_configurationNumber==0) {
	      _nominalPcbConfiguration=*(p[0].data());
	    } else if(_configurationNumber==(32*steps+1)) {
	      *(p[0].data())=_nominalPcbConfiguration;
	    } else {
	      const unsigned nDac((_configurationNumber-1)/steps);
	      const unsigned nStep((_configurationNumber-1)%steps);
	      unsigned value(_nominalPcbConfiguration.dac(nDac));
	      value+=(nStep-(ver.halfByte(0)+1))*(2048/steps);
	      p[0].data()->dac(nDac,value);
	    }
	  }
	  break;
	}
	case IlcRunType::mpsThreshold: {
	  /*
	  p[0].data()->vth12Pos(2048+ver.word());
	  p[0].data()->vth12Neg(2048-ver.word());
	  p[0].data()->vth34Pos(2048+ver.word());
	  p[0].data()->vth34Neg(2048-ver.word());
	  */
	  //u[0].data()->pixelEnable12(false);
	  //u[0].data()->pixelEnable34(false);
	  /*
          for(unsigned i(0);i<s.size();i++) {
	      s[i].data()->maskRegion(2,true);
	      s[i].data()->maskRegion(3,true);
	  }
	  */	  
	  p[0].data()->shaperThreshold( 4*ver.word());
	  p[0].data()->samplerThreshold(8*ver.word());
          /*
          u[0].data()->pixelEnable12(false);
          p[0].data()->shaperThreshold(ver.word());
          p[0].data()->samplerThreshold(ver.word());
	  */
	  break;
	}
	case IlcRunType::mpsThresholdScan: {
	  //u[0].data()->pixelEnable12(false);
	  //u[0].data()->pixelEnable34(false);

	  // Only set sensor configuration at first time
	  // as it doesn't change
	  if(_configurationNumber==0) {
	    for(unsigned i(0);i<s.size();i++) {
	      //s[i].data()->trimRegion(0,8);
	      //s[i].data()->trimRegion(1,8);
	      //s[i].data()->maskRegion(0,true);
	      //s[i].data()->maskRegion(1,true);

	      //s[i].data()->maskRegion(2,true);
	      //s[i].data()->maskRegion(3,true);
	    }
	  } else {
	    s.clear();
	  }

	  const unsigned steps(ver.word()+1);	  
	  const unsigned nStepV(steps-(_configurationNumber%steps)-1);
	  //const unsigned nStepC(_configurationNumber/steps);

	  int offset=0;
	  p[0].data()->shaperThreshold( offset+(int)nStepV*(1000/steps));
	  p[0].data()->samplerThreshold(offset+(int)nStepV*(2000/steps));

	  //by Jamie - 29/11/2007 10.25 am
	  /*
	  int offset=ver.word();
          p[0].data()->shaperThreshold( offset+_configurationNumber);
	  p[0].data()->samplerThreshold(offset+_configurationNumber);
	  */	  

	  /*
	  p[0].data()->shaperThreshold(-128+(int)nStepV*(512/steps),2048-256+nStepC*(512/steps));
	  p[0].data()->samplerThreshold( 384-(int)nStepV*(512/steps),2048-256+nStepC*(512/steps));
	  */

	  //Jamie - sanity check! 16:51
	  // p[0].data()->shaperThreshold(100+_configurationNumber);

	  //if((_configurationNumber%2)==0) {
	  /*
	    p[0].data()->vth12Pos(2048+nStep*(256/steps));
	    p[0].data()->vth12Neg(2048-nStep*(256/steps));
	    p[0].data()->vth34Pos(2048+nStep*(256/steps));
	    p[0].data()->vth34Neg(2048-nStep*(256/steps));
	  */
	    //p[0].data()->vth12Pos(2048    +nStep*(512/steps));
	    //p[0].data()->vth12Neg(2048);
	    //p[0].data()->vth34Pos(2048);
	    //p[0].data()->vth34Neg(2048-512+nStep*(512/steps));
	    //} else {
	    //p[0].data()->vth12Pos(2048);
	    //p[0].data()->vth12Neg(2048-512+nStep*(512/steps));
	    //p[0].data()->vth34Pos(2048    +nStep*(512/steps));
	    //p[0].data()->vth34Neg(2048);
	    //}
	  break;
	}
	case IlcRunType::mpsTrim: {
	  for(unsigned i(0);i<s.size();i++) {
	    s[i].data()->trimQuadrant(ver.bits(4,5),ver.halfByte(0));
	  }
	  break;
	}
	case IlcRunType::mpsTrimScan: {
	  for(unsigned i(0);i<s.size();i++) {
	    s[i].data()->trimSensor(_configurationNumber%16);
	    s[i].data()->maskSensor(true);

	    for(unsigned y(0);y<168;y++) {
	      for(unsigned region(0);region<4;region++) {
		s[i].data()->mask(42*region+((_configurationNumber/16)%42),y,false);
	      }
	    }
	  }

	  p[0].data()->shaperThreshold(100);
	  p[0].data()->samplerThreshold(140);

	  // 2D trim and general threshold scan
	  /*
	  for(unsigned i(0);i<s.size();i++) {
	    s[i].data()->maskSensor(false);
	  }

	  if(_configurationNumber==0 || _configurationNumber==257) {
	    for(unsigned i(0);i<s.size();i++) {
	      s[i].data()->trimRegion(0,7);
	      s[i].data()->trimRegion(1,7);
	      s[i].data()->trimRegion(2,7);
	      s[i].data()->trimRegion(3,7);
	    }
	    p[0].data()->shaperThreshold(10*(ver.halfByte(0)+1));
	    p[0].data()->samplerThreshold(10*(ver.halfByte(1)+1));
	      
	  } else {
	    for(unsigned i(0);i<s.size();i++) {
	      s[i].data()->trimRegion(0,(_configurationNumber-1)%16);
	      s[i].data()->trimRegion(1,(_configurationNumber-1)%16);
	      s[i].data()->trimRegion(2,(_configurationNumber-1)%16);
	      s[i].data()->trimRegion(3,(_configurationNumber-1)%16);
	    }  
	    p[0].data()->shaperThreshold(10*(ver.halfByte(0)+1)+10*((_configurationNumber-1)/16-8));
	    p[0].data()->samplerThreshold(10*(ver.halfByte(1)+1)+10*((_configurationNumber-1)/16-8));
	  }
	  */

	  break;
	}
	case IlcRunType::mpsHitOverride: {
	  u[0].data()->spillCycleCount(10);
	  u[0].data()->hitOverride(true);

	  for(unsigned i(0);i<s.size();i++) {
	    s[i].data()->maskSensor(true);
	  }

	  //u[0].data()->pixelEnable12(false);
	  //u[0].data()->pixelEnable34(false);
	  //u[0].data()->monoPOR(true);


	  //p[0].data()->readFile("Pcb1ConfigurationData.txt");
	  /*
	  const unsigned steps(ver.word()+1);
	  const unsigned nStep(_configurationNumber%steps);

	  p[0].data()->shaperThreshold(-256+nStep*(512/steps));
	  p[0].data()->samplerThreshold(-256+nStep*(512/steps));

	  u[0].data()->spillModeInhibitSpill((_configurationNumber%2)==0);
	  u[0].data()->spillModeFillSrams((_configurationNumber%2)==0);
	  */
	  /*
	  Offset stack pointer true
	    Offset stack pointer value=18 (or 19)
	  */


	  //u[0].data()->clockPhase(0,nStep*(256/steps));
	  //u[0].data()->clockPhase(1,nStep*(256/steps));
	  //u[0].data()->senseDelay(24+nStep*(256/steps));

	  /*
	  p[0].data()->shaperThreshold(-4000+nStep*(8000/steps));
	  p[0].data()->samplerThreshold(-4000+nStep*(8000/steps));
	  */

	    /*
	  const unsigned steps(ver.word()+1);
	  const unsigned nStep(_configurationNumber);
	  //p[0].data()->iSenseBias(2048-256+nStep*(512/steps));
	  p[0].data()->iSenseColRef(2048-256+nStep*(512/steps));
	  */

	  break;
	}
	case IlcRunType::mpsBeamThresholdScan:
	case IlcRunType::mpsBeam: {

	  // Only read out PMTs of masters
	  u[0].usbDaqBroadcast(false);
	  u[0].usbDaqMasterBroadcast(true);
	  u[0].data()->spillModeHistoryEnable(true);

	  u.push_back(MpsLocationData<MpsUsbDaqConfigurationData>());
	  u[1].location(_location);
	  u[1].usbDaqBroadcast(false);
	  u[1].usbDaqSlaveBroadcast(true);

	  break;
	}
	case IlcRunType::mpsCosmics: {
	  break;
	}
	case IlcRunType::mpsSourceThresholdScan: {
	  u[0].data()->spillModeHistoryEnable(true);

	  const unsigned steps(ver.word()+1);	  
	  const unsigned nStepV(_configurationNumber%steps);
	  //const unsigned nStepC(_configurationNumber/steps);
	  p[0].data()->shaperThreshold((int)nStepV*20);
	  p[0].data()->samplerThreshold((int)nStepV*20);

	  break;
	}
	case IlcRunType::mpsSource: {
	  u[0].data()->spillModeHistoryEnable(true);

	  p[0].data()->shaperThreshold(100);
	  p[0].data()->samplerThreshold(50);
	  break;
	}
	case IlcRunType::mpsLaser: {
	  u[0].data()->spillCycleCount(2000);

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserPosition: {
	  u[0].data()->spillCycleCount(2000);

	  MpsLaserConfigurationData l;
	  l.stageX((int)(20.0*(ver.bits(0,3)-4)));
	  l.stageY((int)(20.0*(ver.bits(4,7)-4)));
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserPositionScan: {
	  u[0].data()->spillCycleCount(2000);

	  MpsLaserConfigurationData l;
	  l.numberOfPulses(16);

	  const unsigned steps(ver.word()+1);

	  l.stageX((int)(-4000.0+(_configurationNumber%steps)*(8000.0/steps)));
	  l.stageY((int)(-4000.0+((_configurationNumber/steps)%steps)*(8000.0/steps)));
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserThreshold: {
	  u[0].data()->spillCycleCount(2000);

	  p[0].data()->dac(0,16*ver.word());

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsLaserThresholdScan: {
	  u[0].data()->spillCycleCount(2000);

	  const unsigned steps(ver.word()+1);
	  p[0].data()->dac(0,(_configurationNumber%steps)*(4096/steps));

	  MpsLaserConfigurationData l;
	  inserter.insert<MpsLaserConfigurationData>(l);
	  break;
	}
	case IlcRunType::mpsMonostableLengthScan: {
	  p[0].data()->shaperThreshold(100);
	  p[0].data()->samplerThreshold(100);

	  const unsigned steps(ver.word()+1);
	  const unsigned nStep(_configurationNumber%steps);
	  p[0].data()->i12MSOBias1(nStep*(4096/steps));
	  p[0].data()->i34MSOBias1(nStep*(4096/steps));
	  p[0].data()->i34MSOBias2(nStep*(4096/steps));
	  break;
	}
	case IlcRunType::mpsConfigurationTest: {
	  unsigned short *ptr((unsigned short*)u[0].data());
	  for(unsigned j(0);j<sizeof(MpsUsbDaqConfigurationData)/2;j++) {
	    ptr[j]=(rand()&0xffff);
	  }
	  u[0].data()->spillCycleCount(rand()%8192);
	  u[0].data()->clockPhase(0,rand()%256);
	  u[0].data()->clockPhase(1,rand()%256);
	  u[0].data()->clockPhase(2,rand()%256);
	  u[0].data()->senseDelay(rand()%256);
	  u[0].data()->timeStampPrescale(rand()%256);
	  u[0].data()->spillMode(rand()%256);
	  u[0].data()->masterClockResetDuration(rand()%256);
	  u[0].data()->sramFillClockSleep(rand()%256);
	  u[0].data()->readoutColumnOrder(rand()%256);
	  u[0].data()->staticTimeStampGray(rand()%8192);
	  u[0].data()->readoutStartIndex(rand()%4);
	  u[0].data()->stackPointerOffset(rand()%19);
	  u[0].data()->testTriggerMode(rand()%16);
	  u[0].data()->discriminatorThreshold(0,rand()%256);
	  u[0].data()->discriminatorThreshold(1,rand()%256);

	  u[0].data()->zeroBools();
	  UtlPack rndm(rand());
	  u[0].data()->debugReset200Hold(rndm.bit( 0));
	  u[0].data()->debugReset600Hold(rndm.bit( 1));
	  u[0].data()->debugDiodeResetHold(rndm.bit( 2));
	  u[0].data()->hitOverride(rndm.bit( 3));
	  u[0].data()->monoPOR(rndm.bit( 4));
	  u[0].data()->pixelEnable12(rndm.bit( 5));
	  u[0].data()->pixelEnable34(rndm.bit( 6));
	  u[0].data()->senseEnable(rndm.bit( 7));
	  u[0].data()->debugHitInEnable(rndm.bit( 8));
	  u[0].data()->fastPhiOverride(rndm.bit( 9));
	  u[0].data()->readEnableOverride(rndm.bit(10));
	  u[0].data()->initBPolarity(rndm.bit(11));
	  u[0].data()->rstBPolarity(rndm.bit(12));
	  u[0].data()->fwdBPolarity(rndm.bit(13));
	  u[0].data()->slowSpillPhi(rndm.bit(14));
	  u[0].data()->timeStampOverride(rndm.bit(15));

	  for(unsigned i(0);i<32;i++) {
	    p[0].data()->dac(i,(rand()%4096));
	  }

	  if(ver.bit(0)) {
	    for(unsigned i(0);i<s.size();i++) {
	      unsigned short *ptr((unsigned short*)s[i].data());
	      for(unsigned j(0);j<sizeof(MpsSensor1ConfigurationData)/2;j++) {
		ptr[j]=(rand()&0xffff);
	      }
	    }
	  }

	  break;
	}
	case IlcRunType::mpsUsbDaqConfigurationScan: {

	  if(ver.halfByte(0)==0) {
	    double multiplier(0.1*(_configurationNumber%100));
	    u[0].data()->diodeResetDuration( (unsigned)(u[0].data()->diodeResetDuration() *multiplier));
	    u[0].data()->preAmpResetDuration((unsigned)(u[0].data()->preAmpResetDuration()*multiplier));
	    u[0].data()->shaperResetDuration((unsigned)(u[0].data()->shaperResetDuration()*multiplier));
	  }
	  break;
	}
	case IlcRunType::mpsLaserCoordinates: {
	  unsigned majorStep(_configurationNumber/9);
	  unsigned minorStep(_configurationNumber%9);
	  int dMajor(3000),dMinor(25);

	  int x(0),y(0);
	  if(majorStep==0) {x+= dMajor;y+= dMajor;}
	  if(majorStep==1) {x+=-dMajor;y+= dMajor;}
	  if(majorStep==2) {x+=-dMajor;y+=-dMajor;}
	  if(majorStep==3) {x+= dMajor;y+=-dMajor;}

	  if(minorStep==0) {x+= dMinor;y+= dMinor;}
	  if(minorStep==1) {x+=      0;y+= dMinor;}
	  if(minorStep==2) {x+=-dMinor;y+= dMinor;}
	  if(minorStep==3) {x+=-dMinor;y+=      0;}
	  if(minorStep==4) {x+=      0;y+=      0;}
	  if(minorStep==5) {x+= dMinor;y+=      0;}
	  if(minorStep==6) {x+= dMinor;y+=-dMinor;}
	  if(minorStep==7) {x+=      0;y+=-dMinor;}
	  if(minorStep==8) {x+=-dMinor;y+=-dMinor;}

	  MpsLaserConfigurationData l;
	  l.stageX(x);
	  l.stageY(y);
	  inserter.insert<MpsLaserConfigurationData>(l);

	  break;
	}
	  
	default: {
	  // We missed a run type
	  assert(false);
	  break;
	}
	};
      }

      // Put the configurations into the record
      for(unsigned i(0);i<u.size();i++) {
	inserter.insert< MpsLocationData<MpsUsbDaqConfigurationData> >(u[i]);
      }
      
      for(unsigned i(0);i<p.size();i++) {
	inserter.insert< MpsLocationData<MpsPcb1ConfigurationData> >(p[i]);
      }
      
      for(unsigned i(0);i<s.size();i++) {
	inserter.insert< MpsLocationData<MpsSensor1ConfigurationData> >(s[i]);
      }
      break;
    }

    // Do nothing for other record types      
    default: {
      break;
    }
    };
    
    return true;
  }

  static void addLocation(MpsLocation l) {
    l.write(true);
    _vLocation.push_back(l);
  }


private:
  MpsLocation _location;

  IlcRunType _runType;
  unsigned _configurationNumber;

  MpsPcb1ConfigurationData _nominalPcbConfiguration;

  static std::vector<MpsLocation> _vLocation;
};

std::vector<MpsLocation> MpsConfiguration::_vLocation;

#endif
