#ifndef BmlFnalSlowReadoutDataV0_HH
#define BmlFnalSlowReadoutDataV0_HH

#include <vector>
#include <string>
#include <iostream>


class BmlFnalSlowReadoutDataV0 {

public:
  enum {
    versionNumber=0
  };

  BmlFnalSlowReadoutDataV0();

  bool parse(std::string r);
  
  time_t slowTimeStamp() const;
  void   slowTimeStamp(time_t t);

  time_t fnalTimeStamp() const;
  void   fnalTimeStamp(time_t t);

  std::vector<double> magnetCurrents() const;
  std::vector<double> targetAndCollimators() const;
  std::vector<double> pinholeCollimators() const;
  std::vector<double> cherenkovPressures() const;

  std::ostream& print(std::ostream &o=std::cout, std::string s="") const;


private:
  time_t _timeStamp[2];

  double _magnetCurrent[12];
  double _targetAndCollimator[9];
  double _pinholeCollimator[4];
  double _cherenkovPressure[2];
};


#ifdef CALICE_DAQ_ICC

#include <time.h>
#include <cstring>


BmlFnalSlowReadoutDataV0::BmlFnalSlowReadoutDataV0() {
  memset(this,0,sizeof(BmlFnalSlowReadoutDataV0));
}

bool BmlFnalSlowReadoutDataV0::parse(std::string r) {
  std::istringstream sin(r);

  // Store timestamps
  sin >> _timeStamp[0] >> _timeStamp[1];
  if(!sin) return false;
  
  // Store data; first the doubles
  for(unsigned i(0);i<(12+9+4+2) && sin;i++) {
    sin >> _magnetCurrent[i];
    if(!sin) return false;
  }

  return sin;
}

time_t BmlFnalSlowReadoutDataV0::slowTimeStamp() const {
  return _timeStamp[0];
}

void BmlFnalSlowReadoutDataV0::slowTimeStamp(time_t t) {
  _timeStamp[0]=t;
}

time_t BmlFnalSlowReadoutDataV0::fnalTimeStamp() const {
  return _timeStamp[1];
}

void BmlFnalSlowReadoutDataV0::fnalTimeStamp(time_t t) {
  _timeStamp[1]=t;
}

std::vector<double> BmlFnalSlowReadoutDataV0::magnetCurrents() const {
  std::vector<double> v(12);
  for(unsigned i(0);i<12;i++) v[i]=_magnetCurrent[i];
  return v;
}

std::vector<double> BmlFnalSlowReadoutDataV0::targetAndCollimators() const {
  std::vector<double> v(9);
  for(unsigned i(0);i<9;i++) v[i]=_targetAndCollimator[i];
  return v;
}

std::vector<double> BmlFnalSlowReadoutDataV0::pinholeCollimators() const {
  std::vector<double> v(4);
  for(unsigned i(0);i<4;i++) v[i]=_pinholeCollimator[i];
  return v;
}

std::vector<double> BmlFnalSlowReadoutDataV0::cherenkovPressures() const {
  std::vector<double> v(2);
  for(unsigned i(0);i<2;i++) v[i]=_cherenkovPressure[i];
  return v;
}

std::ostream& BmlFnalSlowReadoutDataV0::print(std::ostream &o, std::string s) const {
  o << s << "BmlFnalSlowReadoutDataV0::print()" << std::endl;

  o << s << " Slow controls timestamp = " << _timeStamp[0] << " = " << ctime(_timeStamp  );
  o << s << " FNAL database timestamp = " << _timeStamp[1] << " = " << ctime(_timeStamp+1);

  o << s << " Magnet main west bend         = " << _magnetCurrent[ 0] << " A" << std::endl;
  o << s << " Magnet target bend            = " << _magnetCurrent[ 1] << " A" << std::endl;
  o << s << " Magnet target low limit       = " << _magnetCurrent[ 2] << " A" << std::endl;
  o << s << " Magnet last east bend         = " << _magnetCurrent[ 3] << " A" << std::endl;
  o << s << " Magnet last east low limit    = " << _magnetCurrent[ 4] << " A" << std::endl;
  o << s << " Magnet quadrupole 1           = " << _magnetCurrent[ 5] << " A" << std::endl;
  o << s << " Magnet quadrupole 1 low limit = " << _magnetCurrent[ 6] << " A" << std::endl;
  o << s << " Magnet quadrupole 2           = " << _magnetCurrent[ 7] << " A" << std::endl;
  o << s << " Magnet quadrupole 2 low limit = " << _magnetCurrent[ 8] << " A" << std::endl;
  o << s << " Magnet vertical trim 1        = " << _magnetCurrent[ 9] << " A" << std::endl;
  o << s << " Magnet vertical trim 2        = " << _magnetCurrent[10] << " A" << std::endl;
  o << s << " Magnet horizontal trim        = " << _magnetCurrent[11] << " A" << std::endl;

  o << s << " Target upstream              = " << _targetAndCollimator[0] << " mm" << std::endl;
  o << s << " Target downstream            = " << _targetAndCollimator[1] << " mm" << std::endl;
  o << s << " Collimator first vertical    = " << _targetAndCollimator[2] << " mm" << std::endl;
  o << s << " Collimator first horizontal  = " << _targetAndCollimator[3] << " mm" << std::endl;
  o << s << " Lead sheet                   = " << _targetAndCollimator[4] << " mm" << std::endl;
  o << s << " Collimator second horizontal = " << _targetAndCollimator[5] << " mm" << std::endl;
  o << s << " Collimator second vertical   = " << _targetAndCollimator[6] << " mm" << std::endl;
  o << s << " Muon absorber half 1         = " << _targetAndCollimator[7] << " mm" << std::endl;
  o << s << " Muon absorber half 2         = " << _targetAndCollimator[8] << " mm" << std::endl;

  o << s << " Pinhole collimator upstream horizontal   = " << _pinholeCollimator[0] << " mm" << std::endl;
  o << s << " Pinhole collimator downstream horizontal = " << _pinholeCollimator[1] << " mm" << std::endl;
  o << s << " Pinhole collimator upstream vertical     = " << _pinholeCollimator[2] << " mm" << std::endl;
  o << s << " Pinhole collimator downstream vertical   = " << _pinholeCollimator[3] << " mm" << std::endl;

  o << s << " Cherenkov pressure inner? = " << _cherenkovPressure[0] << std::endl;
  o << s << " Cherenkov pressure outer? = " << _cherenkovPressure[1] << std::endl;

  return o;
}

#endif
#endif
