//
// $Id: DhcFeConfigurationData.hh,v 1.6 2008/02/18 15:03:31 jls Exp $
//

#ifndef DhcFeConfigurationData_HH
#define DhcFeConfigurationData_HH

#include <string>
#include <iostream>

#include "UtlPack.hh"


class DhcFeConfigurationData {

public:
  enum {
    versionNumber=0
  };

  enum ShapeTime {
    t65=0,
    t85=1,
    t100=2,
    t125=3,
    endOfDcrShapingTineEnum
  };

public:
  DhcFeConfigurationData();

  const unsigned char* chipid() const;
  void          chipid(unsigned char n);

  const unsigned char* plsr() const;
  void          plsr(unsigned char n);

  const unsigned char* intd() const;
  void          intd(unsigned char n);

  const unsigned char* shp2() const;
  void          shp2(unsigned char n);

  const unsigned char* shp1() const;
  void          shp1(unsigned char n);

  const unsigned char* blrd() const;
  void          blrd(unsigned char n);

  const unsigned char* vtnd() const;
  void          vtnd(unsigned char n);

  const unsigned char* vtpd() const;
  void          vtpd(unsigned char n);

  const unsigned char* dcr() const;
  void          dcr(unsigned char n);

  const unsigned char dcrCapSelect() const;
  void                dcrCapSelect(unsigned char n);

  const bool dcrLowGain() const;
  void       dcrLowGain(bool b);

  const bool dcrDisableBaselineRestore() const;
  void       dcrDisableBaselineRestore(bool b);
  
  const bool dcrAccept() const;
  void       dcrAccept(bool b);

  const bool dcrActivatePipeline() const;
  void       dcrActivatePipeline(bool b);

  const bool dcrExternalTrigger() const;
  void       dcrExternalTrigger(bool b);

  const unsigned char* inj() const;
  void inj(const unsigned char* n);
  void inj(long long n);

  const unsigned char* kill() const;
  void kill(const unsigned char* n);
  void kill(long long n);

  std::ostream& print(std::ostream &o, std::string s="") const;

private:
  static const std::string _dcrShapingTime[endOfDcrShapingTineEnum];

private:
  unsigned char _chipid;
  unsigned char _plsr;
  unsigned char _intd;
  unsigned char _shp2;
  unsigned char _shp1;
  unsigned char _blrd;
  unsigned char _vtnd;
  unsigned char _vtpd;
  unsigned char _dcr;
  unsigned char _inj[8];
  unsigned char _kill[8];
};

DhcFeConfigurationData::DhcFeConfigurationData() {
  _chipid=21;
  _plsr=0;
  _intd=139;
  _shp2=121;
  _shp1=116;
  _blrd=81;
  _vtnd=0;
  _vtpd=255;
  _dcr=0;
  memset(_inj,0,sizeof(_inj));
  memset(_kill,0,sizeof(_kill));
}

const unsigned char* DhcFeConfigurationData::chipid() const {
  return &_chipid;
}

void DhcFeConfigurationData::chipid(unsigned char n) {
  _chipid=n;
}

const unsigned char* DhcFeConfigurationData::plsr() const {
  return &_plsr;
}

void DhcFeConfigurationData::plsr(unsigned char n) {
  _plsr=n;
}

const unsigned char* DhcFeConfigurationData::intd() const {
  return &_intd;
}

void DhcFeConfigurationData::intd(unsigned char n) {
  _intd=n;
}

const unsigned char* DhcFeConfigurationData::shp2() const {
  return &_shp2;
}

void DhcFeConfigurationData::shp2(unsigned char n) {
  _shp2=n;
}

const unsigned char* DhcFeConfigurationData::shp1() const {
  return &_shp1;
}

void DhcFeConfigurationData::shp1(unsigned char n) {
  _shp1=n;
}

const unsigned char* DhcFeConfigurationData::blrd() const {
  return &_blrd;
}

void DhcFeConfigurationData::blrd(unsigned char n) {
  _blrd=n;
}

const unsigned char* DhcFeConfigurationData::vtnd() const {
  return &_vtnd;
}

void DhcFeConfigurationData::vtnd(unsigned char n) {
  _vtnd=n;
}

const unsigned char* DhcFeConfigurationData::vtpd() const {
  return &_vtpd;
}

void DhcFeConfigurationData::vtpd(unsigned char n) {
  _vtpd=n;
}

const unsigned char* DhcFeConfigurationData::dcr() const {
  return &_dcr;
}

void DhcFeConfigurationData::dcr(unsigned char n) {
  _dcr=n;
}

const unsigned char DhcFeConfigurationData::dcrCapSelect() const {
  return (_dcr & 0xc0) >> 6;
}

void DhcFeConfigurationData::dcrCapSelect(unsigned char n) {
  assert(n<4);
  _dcr &= 0x3f;
  _dcr |= (n<<6);
}

const bool DhcFeConfigurationData::dcrLowGain() const {
  return _dcr & 0x20;
}

void DhcFeConfigurationData::dcrLowGain(bool b) {
  ( b ? _dcr |= 0x20 : _dcr &= ~0x20 );
}

const bool DhcFeConfigurationData::dcrDisableBaselineRestore() const {
  return _dcr & 0x10;
}

void DhcFeConfigurationData::dcrDisableBaselineRestore(bool b) {
  ( b ? _dcr |= 0x10 : _dcr &= ~0x10 );
}
  
const bool DhcFeConfigurationData::dcrAccept() const {
  return _dcr & 0x08;
}

void DhcFeConfigurationData::dcrAccept(bool b) {
  ( b ? _dcr |= 0x08 : _dcr &= ~0x08 );
}

const bool DhcFeConfigurationData::dcrActivatePipeline() const {
  return _dcr & 0x04;
}

void DhcFeConfigurationData::dcrActivatePipeline(bool b) {
  ( b ? _dcr |= 0x04 : _dcr &= ~0x04 );
}

const bool DhcFeConfigurationData::dcrExternalTrigger() const {
  return _dcr & 0x02;
}

void DhcFeConfigurationData::dcrExternalTrigger(bool b) {
  ( b ? _dcr |= 0x02 : _dcr &= ~0x02 );
}

const unsigned char* DhcFeConfigurationData::inj() const {
  return _inj;
}

void  DhcFeConfigurationData::inj(const unsigned char* n) {
  memcpy(_inj,n,sizeof(_inj));
}

void  DhcFeConfigurationData::inj(long long n) {
  memcpy(_inj,&n,sizeof(_inj));
}

const unsigned char* DhcFeConfigurationData::kill() const {
  return _kill;
}

void  DhcFeConfigurationData::kill(const unsigned char* n) {
  memcpy(_kill,n,sizeof(_kill));
}

void  DhcFeConfigurationData::kill(long long n) {
  memcpy(_kill,&n,sizeof(_kill));
}

std::ostream& DhcFeConfigurationData::print(std::ostream &o, std::string s) const {
  o << s << "DhcFeConfigurationData::print()" << std::endl;

  o << s << " ChipID  = " << printHex(_chipid) << std::endl;
  o << s << " PLSR    = " << printHex(_plsr) << std::endl;
  o << s << " IntD    = " << printHex(_intd) << std::endl;
  o << s << " Shp2D   = " << printHex(_shp2) << std::endl;
  o << s << " Shp1D   = " << printHex(_shp1) << std::endl;
  o << s << " BlrD    = " << printHex(_blrd) << std::endl;
  o << s << " VtnD    = " << printHex(_vtnd) << std::endl;
  o << s << " VtpD    = " << printHex(_vtpd) << std::endl;
  o << s << " DCR     = " << printHex(_dcr) << std::endl;

  o << s << "   Shaping Time is " << _dcrShapingTime[dcrCapSelect()];
  o << s << std::endl;
  o << s << "   Gain is " << (dcrLowGain() ? "Low,RPC" : "High,GEM");
  o << s << std::endl;
  o << s << "   Baseline Restore is " << (dcrDisableBaselineRestore() ? "Disabled" : "Enabled");
  o << s << std::endl;
  o << s << "   Chip wide mask is " << (dcrAccept() ? "Accept" : "Mask");
  o << s << std::endl;
  o << s << "   Pipeline is " << (dcrActivatePipeline() ? "Enabled" : "Disabled");
  o << s << std::endl;
  o << s << "   Trigger is " << (dcrExternalTrigger() ? "External" : "Internal");
  o << s << std::endl;

  o << s << " Inj     = ";
  for(unsigned i(0); i<sizeof(_inj); i++)
    o << s << std::setfill('0') << std::setw(2)
      << std::hex << static_cast<unsigned int>(_inj[i]) << std::dec;
  o << s << std::endl;

  o << s << " Kill    = ";
  for(unsigned i(0); i<sizeof(_kill); i++)
    o << s << std::setfill('0') << std::setw(2)
      << std::hex << static_cast<unsigned int>(_kill[i]) << std::dec;
  o << s << std::setfill(' ');
  o << s << std::endl;

  return o;
}

const std::string DhcFeConfigurationData::_dcrShapingTime[]={
  "65ns","85ns","100ns","125ns"
};


#endif // DhcFeConfigurationData_HH
