#ifndef CaliceMaps1TestSystemInterface_HH_
#define CaliceMaps1TestSystemInterface_HH_

#include "TypeDefs.hh"

#include "USBDAQHardwareInterface.hh"

#include "AddressMapper.hh"



namespace CALICEMAPS1
{
  using namespace ::USBDAQ;
  
  class CaliceMaps1TestSystemInterface
  {

    

    enum IDS{MASTER=0x7, SLAVE=1};
    // this is the specific address space:
    enum ADDRESSES{
      SPI_DATA=1, 
      SPI_CTRL, 
      SPI_STATUS=SPI_CTRL, 
      DATA_COUNT_0,
      DATA_COUNT_1,
      DATA_COUNT_2,
      SPILL_CTRL_0, 
      SPILL_CTRL_1, 
      SPILL_CTRL_2, 
      SPILL_CTRL_3, 
      SPILL_CTRL_4, 
      SPILL_CTRL_5, 
      TEST_TRIG_0,
      TEST_TRIG_1,
      TEST_TRIG_2,
      WC_REG0, 
      G_STATUS, 
      LASER_INT_0,
      LASER_INT_1,
      LASER_INT_2,
      TRIG_UNIT_CTRL_0,
      GEN_SETTINGS_2,
      DEBUG_SETTINGS, 
      ID_REG=DEBUG_SETTINGS,
      PHI_CTRL_0, 
      PHI_CTRL_1,
      CLK_0, 
      TRIGGER, 
      STATUS_1=TRIGGER, 
      G_CONFIG,
      FW_TIME,      
      FW_REV,
      LASTREG
    };
    
    enum SPI_CONTROL{
      READ16=0,
      READ32=1, 
      WRITE=0x8, 
      R_EDGE=0x4
    };
    
    enum SPI_MUX_ADDR{
      DAC_0=0, 
      DAC_1, 
      DAC_2, 
      DAC_3,
      TEMPERATURE 
    };
    
    enum DAC_CHANNEL_DEFS{
      I34_OUT_BIAS=0,   //
      I34_MSO_BIAS2,    //
      I34_MSO_BIAS1,    //
      I34_COMP_BIAS2,   //
      I34_PRE_BIAS,     //
      I34_OUTSF_BIAS,   //
      I34_COMP_BIAS1,   //
      I34_SF_BIAS,      //
      I34_COMP_BIAS_TRIM,// 8 
      I34_DEBUGSF_BIAS, //
      ISENSE_COLREF,    //
      I12_COMP_BIAS_TRIM,//
      I12_SHAPER_BIAS,  //
      I12_PRE_BIAS,     //
      I12_COMP2_BIAS,   //
      I12_MSO_BIAS1,    //
      I12_IOUT_BIAS,    //16
      ISENSE_COMP_BIAS, // 
      ISENSE_IOUT_BIAS, // 
      I12_COMP1_BIAS,   //
      ISENSE_BIAS,      //
      VTH_34_NEG,       //
      DEBUG_VTH_NEG,    //
      VTH_12_NEG,       //
      VRST,             // 24
      VTH_34_POS,       //
      PREAMP_VCASC_34,  //
      DEBUG_VTH_POS,    //
      SHAPE_VCASC_12,   //
      PREAMP_VCASC_12,  //
      VTH_12_POS        //
    };

    enum SPILL_CTRL_MODE{
      DEFAULT=0,
      LAZY_INIT_B=1,
      INHIBIT_READOUT=2,
      REQUIRE_READOUT_TRIG=4,
      INHIBIT_SPILL=8,
      FILLSRAMS=0x10,
      OFFSET_STACK_PTR=0x20,
      ENABLE_PMT_RECORD=0x40,	
      PIXEL_RST_OVRD=0x80
    };

    enum TEST_TRIG{
      NONE=0,
      DIODE_RST_OVERRIDE=1,
      SHAPER_RST=2,
      SAMPLER_RST=4,
      EXT=8
    };

    enum TRIG_SOURCE{
      SW=0,
      HW=1,
      LASER=2,// currently unconnected: redundant with HW
      SPARE=3 // currently unconnected: redundant with HW
    };

    
    

  public:

    
    static int Instance(CaliceMaps1TestSystemInterface ** aCM1TSI, int aLen=4);
    

    // this part of the interface will remain
    // public: 

    CaliceMaps1TestSystemInterface();

    virtual ~CaliceMaps1TestSystemInterface();
    
    
    
    void LogicReset();
    void BufferReset();
    
    // returns the sensor card ID. The least significant 
    // 6 bits are valid, the rest are 0.
    u16 ReadSensorId();
    
    // returns the daq board address. the least significant 
    // 4 bits are valid, the rest are 0.  
    u16 ReadDAQAddress();
    
    void StartSpill();
    void StopSpill();
    
    // 32 bit register. Note that a number 
    // above 8191 will work but doesn't make much sense
    // due to the sensor limitations. The full 32 bits
    // returned are valid. 
    void SetSpillCycleCount(u32 aCount);
    u32 ReadSpillCycleCount();
    
    void SetSpillCycleStart(u16 aStart);
    u16 ReadSpillCycleStart();
    

    void SetSensorVersion(u16 lVersion=1);
    u16 ReadSensorVersion();


    //u32 DAQ(u32 ** aBuf);
    
    u32 DAQ(u32 * aBuf, u32 aBufLen);
    
    
    
    // 8 bit clock phase settings indexex by the 
    // first argument. Most significant 8 bits returned 
    // will be 0.
    void SetClockPhase(u16 aClk, u16 aPhase);
    u16 ReadClockPhase(u16 aClk);
    

    void SetPMTCoincidenceCountEnable(bool aEn);
    bool ReadPMTCoincidenceCountEnable();
    void TriggerPMTCoincidenceCountReset();
    u32 ReadPMTCoincidenceCount();


    // returns true if the board is configured as the 
    // master. 
    bool IsMaster();
    
    //
    // trigger interface: 
    
    void SetTriggerMask(u16 aMask=0x10);
    unsigned short ReadTriggerMask();

    void SetTriggerSource(u16 aSource);
    u16 ReadTriggerSource();
    
    void SetTriggerEnable(bool aEn);
    bool ReadTriggerEnable();

    i32 HasTriggered();

    // triggers the debug reset. Both the 200 and 600
    // start at the same time. 
    void TriggerDebugReset();
    
    // sets the debug reset 200 duration. LSB=25ns.
    void SetDebugReset200Duration(u16 aRstLen=8);
    // sets the debug 600 duration. lsb=25ns. 
    void SetDebugReset600Duration(u16 aRstLen=24);

    void SetDebugDiodeResetDuration(u16 aRstLen=4);
    
    void SetTestTrigStart(u32 aStart);
    u32 ReadTestTrigStart();
    
    void SetTestTrigStop(u32 aStop);
    u32 ReadTestTrigStop();

    void SetTestTrigConfig(u16 aConfig);
    u16  ReadTestTrigConfig();


    void SetDebugReset200Hold(bool aHold);
    bool ReadDebugReset200Hold();
   
    void SetDebugReset600Hold(bool aHold);
    bool ReadDebugReset600Hold();

    void SetDebugDiodeResetHold(bool aHold);
    bool ReadDebugDiodeResetHold();
    
    
    void SetDiodeReset12Hold(bool aHold);
    bool ReadDiodeReset12Hold();

    void SetSampleReset34Hold(bool aHold);
    bool ReadSampleReset34Hold();
    
    void SetShaperReset34Hold(bool aHold);
    bool ReadShaperReset34Hold();
    
    

    void SetPostPixResetSleepDuration(u16 aPRstSleep);
    u16 ReadPostPixResetSleepDuration();

    //
    void SetHitOverride(bool aOvrd);
    bool ReadHitOverride();

    // accessors for the debug trim level. 
    void SetDebugTrim(u16 aTrim);
    u16 ReadDebugTrim();
    
    // reads the temperature from the sensor and converts 
    // it to degrees centigrade. 
    double Temperature();
    
    // converts from the peculiar integer encoding 
    // returned from the sensor to degC. 
    static double Temperature(u16 aTemp);

    // returns the raw integer encoded temperature. 
    u16 ReadSensorCardTemperature();
        
    // accessors to the monopole power on reset
    void SetMonoPOR(bool aPOR);
    bool ReadMonoPOR();
        
    void SetPixelEnable12(bool aEb12);
    bool ReadPixelEnable12();
 
    void SetPixelEnable34(bool aEb12);
    bool ReadPixelEnable34();

    void SetSenseEnable(bool aSenseEn);
    bool ReadSenseEnable();
    
    void SetDebugHitInEnable(bool aDbgEn, u16 aLen=10);
    bool ReadDebugHitInEnable();
    
    //

    void SetFastPhiOverride(bool aOvrd);
    bool ReadFastPhiOverride();

    void SetSlowSpillPhi(bool aSlowSpillPhi);
    bool ReadSlowSpillPhi();
    
    void SetReadEnableOverride(bool aRdEnOvrd);
    bool ReadReadEnableOverride();
    

    void SetInitBPolarity(bool aPol);
    bool ReadInitBPolarity();

    void SetFwdBPolarity(bool aPol);
    bool ReadFwdBPolarity();
    
    void SetRstBPolarity(bool aPol);
    bool ReadRstBPolarity();

    void SetMCResetDuration(u16 aRstLen);
    u16 ReadMCResetDuration();

    
    void SetStackPointerOffset(u16 aStackOffset);
    u16 ReadStackPointerOffset();


    // the least significant 8 bits set the column order mapping
    // for the readout, such that: 
    //  1:0 = first column accessed
    //  3:2 = second 
    //  5:4 = third
    //  7:6 = fourth. 
    // columns may be repeated.
    // top byte returned as 0s.
    
    void SetReadoutColumnOrder(u16 aCols);
    u16 ReadReadoutColumnOrder();
    
    // sets the start index for the 
    // readout column counter. 
    void SetReadoutStartIndex(u16 aStartIndex);
    u16 ReadReadoutStartIndex();
    
    
    
    
    // least siginificant 13 bits are used,
    // returned values are 0 padded.
    void SetStaticTimeStamp(u16 aTS);
    u16 ReadStaticTimeStamp();

    void SetSramFillClockSleep(u16 aSleep);
    u16 ReadSramFillClockSleep();

    void SetTimeStampOverride(bool aOvrd);
    bool ReadTimeStampOverride();

    u32 ReadFirmwareRevision();
    u32 ReadFirmwareRevisionUTC();
    


    // sensor references: 
    // 
    // all the sensor references can be individually loaded and read
    // back. Each of the following functions accepts or returns a u16,
    // the least significant 12 bits of which are valid. 
    // 
    
    void SetDebugVthPos(u16 aPos);
    u16 ReadDebugVthPos();
    
    void SetDebugVthNeg(u16 aNeg);
    u16 ReadDebugVthNeg();
    
    //   
    void SetVth12Pos(u16 aPos);
    u16 ReadVth12Pos();
    
    void SetVth12Neg(u16 aNeg);
    u16 ReadVth12Neg();
    
    void SetVth34Pos(u16 aPos);
    u16 ReadVth34Pos();
    
    void SetVth34Neg(u16 aNeg);
    u16 ReadVth34Neg();
    
    //
    void SetVRst(u16 aRst);
    u16 ReadVRst();
    
    void SetISenseColRef(u16 aIsense);
    u16 ReadISenseColRef();
    
    void SetPreAmp34VCasc(u16 aVCascP);
    u16 ReadPreAmp34VCasc();
    
    void SetISenseBias(u16 aBias);
    u16 ReadISenseBias();
    
    void SetI12Comp1Bias(u16 aBias);
    u16 ReadI12Comp1Bias();
    
    void SetI12Comp2Bias(u16 aBias);
    u16 ReadI12Comp2Bias();
    
    
    void SetISenseIoutBias(u16 aBias);
    u16 ReadISenseIoutBias();
    

    // 
    void SetISenseCompBias(u16 aBias);
    u16 ReadISenseCompBias();

    void SetI12IoutBias(u16 aBias);
    u16 ReadI12IoutBias();
 
    
    void SetI12PreAmpBias(u16 aBias);
    u16 ReadI12PreAmpBias();

    void SetI12ShaperBias(u16 aBias);
    u16 ReadI12ShaperBias();

    void SetI12CompBiasTrim(u16 aBias);
    u16 ReadI12CompBiasTrim();
    
    void SetI34CompBiasTrim(u16 aBias);
    u16 ReadI34CompBiasTrim();

    void SetI34DebugSFBias(u16 aBias);
    u16 ReadI34DebugSFBias();
    
    void SetI34SFBias(u16 aBias);
    u16 ReadI34SFBias();

    void SetI34CompBias1(u16 aBias);
    u16 ReadI34CompBias1();

    void SetI34CompBias2(u16 aBias);
    u16 ReadI34CompBias2();
    
    void SetI34MSOBias1(u16 aBias);
    u16 ReadI34MSOBias1();

    void SetI34MSOBias2(u16 aBias);
    u16 ReadI34MSOBias2();
    
    void SetI34PreAmpBias(u16 aBias);
    u16 ReadI34PreAmpBias();

    void SetI34OutBias(u16 aBias);
    u16 ReadI34OutBias();
    
    void SetI34OutSFBias(u16 aBias);
    u16 ReadI34OutSFBias();

    void SetI12MSOBias1(u16 aBias);
    u16 ReadI12MSOBias1();

    void SetShaper12VCasc(u16 aVCasc);
    u16 ReadShaper12VCasc();

    void SetPreAmp12VCasc(u16 aVCasc);
    u16 ReadPreAmp12VCasc();
    
    //
    // end of the reference accessors.
    //
    
    
    ////////////////////

    // trigger interface control: 

    // the references against which the PMT amplifier
    // output is discriminated are controlled here. The 
    // first u16 is for channel 0 and the second for channel 2. 
    // The least siginificant 8 bits are valid. The low and 
    // high limits of the reference are 0 and 4.2V respectively 
    // which makes the LSB ~16mV. The DC bias at the input 
    // to the amplifier is set at 2.5V and the pulse resulting 
    // from PMT activity willl be negative relative to this. 
    // Thus, a sensible place to start looking at the reference 
    // is around (say) 2.2V. This corresponds to a DAC code 
    // of about 137 (decimal). 
    // 
    void SetDiscriminatorThresholds(u16 aCh0, u16 aCh1);
    u16 ReadDiscriminatorThreshold(u16 aCh);
    

    // this function set the correct mode of the 
    // DACs and is called once by the initialisation
    // function. 
    void SetupDiscriminatorReferenceDAC();
    
    
    // Loads the entire configuration. 
    //  aBufIn: points to a block of mememory 
    //          containing the data to be loaded. 
    //  aBufOut: data passed completely through the 
    //           top shift register are placed here. 
    //  aRowLen: row length in bytes including the extra
    //           pass-through bytes. 
    //  aNumRows: number of rows.  
    void LoadConfig(const u8 *  aBufIn, 
		    u8 * aBufOut, 
		    u16 aRowLen=24,
		    u16 aNumRows=168);
    
    // reads back a loaded configuration.
    // aBufOut: points to a buffer that needs to be 
    //          (aRowLen * aBumRows) bytes long (or longer).
    // aRowLen: the number of bytes per row to be recaptured. 
    // aNumRows: bumber of rows to be read back. 
    void ReadBackConfig(u8 * aBufOut, 
			u16 aRowLen=21,
			u16 aNumRows=168);
    

    // fills the top shift register with the 
    // last 21 bytes pointed at by aBufIn and 
    // clocks the data down one column. 
    // the extra (leading) bytes are put into the
    // buffer pointed at by aBufOut. 
    void LoadTopConfigRow(const u8 * aBufIn, 
			  u8 * aBufOut,
			  u16 aBufLen=24);

    // parallel loads the data at the column bottoms 
    // into the readout shift register and then
    // clocks them out, putting them into the buffer 
    // pointed at by aBuf. It doesn't make sense to 
    // put anything other than 21 as the second argument. 
    void ReadBackBottomRow(u8 * aBuf, 
			   u16 aBufLen=21, 
			   bool aInhibitVShift=false);

    // sets up the bottom row load clock parameters
    // and switches the clock data muxes over. 
    // loads the data into the input fifo, triggers 
    // the shift into the shift register, polls 
    // the busy flag, and then reverts the data mux 
    // back. 21 bytes must be available to be loaded. 
    void LoadBottomRow(u8 * aBuf);
    
    void SetBottomRowLoad(u16 aBottomLoad);
    u16 ReadBottomRowLoad();


    // triggers the load of the input shift 
    // register. 
    void TriggerTopConfigLoad();

    // issues a reset to the input 
    // configuration register. 
    void IssueTopConfigReset();
    
    // triggers the vertical shift. 
    void TriggerSlowConfigShift(u16 aNoClks=1);
    
    // resets the slow shoft register.
    void IssueSlowConfigReset();
   
    // triggers the readback shifting.  
    void TriggerReadbackConfigShift();
    
    // triggers the parallel load of the readout 
    // shift register from the column bottoms. 
    // void TriggerReadbackConfigPLoad();
    
    // resets the readout shift register. 
    void IssueReadbackConfigReset();
    
    // accessors for the configuration space of the 
    // bit shifting firmware. 
    u16 ReadReadBackConfigSRBitCount();    
    void SetReadbackConfigSRBitCount(u16 aBC=168);
    u16 ReadFastConfigSRBitCount();    
    void SetFastConfigSRBitCount(u16 aBC=192);
    u16 ReadSlowConfigSRBitCount();
    void SetSlowConfigSRBitCount(u16 aBC=1);
    
    

    void SetSlowConfigPhi2Override(bool aEnable, bool aValue);
    u16 ReadSlowConfigPhi2Override();
 
    void SetSlowConfigPhi3Override(bool aEnable, bool aValue);
    u16 ReadSlowConfigPhi3Override();
     
    void SetSlowConfigRstOverride(bool aEnable, bool aValue);
    u16 ReadSlowConfigRstOverride();


    
    // returns the total number of 
    // words put into the readout buffer. 
    u32 ReadDAQWordCount();

    // returns the status of the spill controller FSM.
    u16 ReadSpillControllerStatus();
    
    // accessors for the sense delay wait in 
    // LSB = 10ns. 
    u16 ReadSenseDelay();
    void SetSenseDelay(u16 aDly);

    // accessors for the start of spill 
    // pre-amp reset. LSB = 10ns 
    u16 ReadPreAmpResetDuration();
    void SetPreAmpResetDuration(u16 aRstLen);
 
    // accessors for the star of spill shaper reset. 
    // lsb=10ns.
    u16 ReadShaperResetDuration();
    void SetShaperResetDuration(u16 aRstLen);

    // accessors for the start of spill diode 
    // reset. lsb=10ns.
    u16 ReadDiodeResetDuration();
    void SetDiodeResetDuration(u16 aRstLen);
    
    // accessors for the timestamp counter prescale. 
    // the base clock frequency = 100MHz, so a value of 
    // 15 or 16 should be used. 
    u16 ReadTimeStampPrescale();
    void SetTimeStampPrescale(u16 aPS);
    
    // accessors for the spill mode register. 
    u16 ReadSpillMode();
    void SetSpillMode(u16 aMode);
        
    
    // general status register. The mask is applied
    // to the data before being returned. 
    u32 ReadStatus(u32 lMask=0xffffffff);
    

    u16 ReadSensorStatusFlags();

    // enables the LVDS buffers and triggers the 
    // safe exit from the power-up shift reg. condition. 
    void EnableLVDSBuffers();
    void DisableLVDSBuffers();
    
    // internal function. 
    void SetConfigReadoutMode();
    void ClearConfigReadoutMode();

    //void ReadSensorWordCount(u16 * aBuf, u16 aLen=4);
    u16 ReadSensorWordCount(u16 aCol);

    
    u32 ReadBufferInputCount();
    u32 ReadBufferOutputCount();

    u32 ReadTriggerHistoryWordCount();

    u32 ReadTriggerHistory(u8 * aBuf, u32 aBufLen);
    
  public:
    
    void SetupReferences();
    void SetReferenceLevel(u16 aReference, u16 aLevel);
    u16 ReadReferenceLevel(u16 aReference);
    
  public: 
    
    void SpiWrite(u16 aMode, u16 aSlave, u32 aData);
    u32 SpiRead(u16 aMode, u16 aSlave, u32 aCmd=0);
    
    void IssueTrigger(u32 aTrigVector, u32 aDuration=3);
    
  private:
    CaliceMaps1TestSystemInterface(const CaliceMaps1TestSystemInterface &);
    CaliceMaps1TestSystemInterface & operator=(const CaliceMaps1TestSystemInterface &);

  private: 
    
    ::USBDAQ::USBDAQHardwareInterface * mInterface;
    u16 mLastTemp;
    u16 mReadoutStartIndex;
    u16 mMode;
    u32 mSpillCycleCount;
  };

}//~namespace CALICEMAPS1

#endif//~CaliceMaps1TestSystemInterface_HH_

