#include "BunchTrainBuffer.hh" 

#include <iostream>
#include <cstring>
#include <vector>

namespace CALICEMAPS1
{
  


  PixelHit::PixelHit():
    mTimeCode(0),
    mRowAddr(0),
    mColAddr(0),
    mHitPattern(0),
    mMuxAddr(0)
  {
    
  }

  u16 PixelHit::GetTimeCode()const{
    return mTimeCode;
  }
  void PixelHit::SetTimeCode(u16 aTC){
    mTimeCode=aTC;
  }
  
  u16 PixelHit::GetRowAddress()const{
    return mRowAddr;
  }
  void PixelHit::SetRowAddress(u16 aRowAddr){
    mRowAddr=aRowAddr;
  }
  
  u16 PixelHit::GetColumnAddress()const{
    return mColAddr;
  }
  void PixelHit::SetColumnAddress(u16 aColAddr){
    mColAddr=aColAddr;
  }
  
  u8 PixelHit::GetHitPattern()const{
    return mHitPattern;
  }
  
  void PixelHit::SetHitPattern(u8 aHitPat){
    mHitPattern=aHitPat;
  }
  
  void PixelHit::Reset(){
    mTimeCode=0;
    mRowAddr=0;
    mColAddr=0;
    mHitPattern=0;
    mMuxAddr=0;
  }
    
  
  u8 PixelHit::GetMuxAddress()const{
    return mMuxAddr;
  }
  void PixelHit::SetMuxAddress(u8 aMuxAddr){
    mMuxAddr=aMuxAddr;
  }
  
  std::ostream & operator<<(std::ostream & aOs, 
			    const PixelHit & aPH)
  {
    aOs<<aPH.mRowAddr<<'\t'
       <<aPH.mColAddr<<'\t'
       <<aPH.mTimeCode;
    return aOs;
  }
 

  //

  BunchTrainBuffer::BunchTrainBuffer():
    mBuf(0),
    mLen(0), 
    mIsGood(false)
  {
    
  }
  
  BunchTrainBuffer::BunchTrainBuffer(u32 aLen):
    mBuf(0),
    mLen(0),
    mIsGood(false)
  {
    this->ResetBuffer(aLen);
  }

  BunchTrainBuffer::BunchTrainBuffer(const BunchTrainBuffer & aBTBuf){
    
    this->mLen=aBTBuf.mLen;
    this->mBuf=new u32 [this->mLen];
    std::memcpy(reinterpret_cast<void *>(this->mBuf), 
		reinterpret_cast<void *>(aBTBuf.mBuf), 
		(this->mLen)*sizeof(u32)); 
    std::memcpy(reinterpret_cast<void *>(this->mColLengths), 
		reinterpret_cast<const void *>(aBTBuf.mColLengths),
		4*sizeof(u16));
    this->mIsGood=aBTBuf.mIsGood;
  }

  BunchTrainBuffer & BunchTrainBuffer::operator=(const BunchTrainBuffer & aRhs){
    if(this!=(&aRhs)){
      this->AllocateBuffer(aRhs.mLen);
      if(this->mLen!=0){
	std::memcpy(reinterpret_cast<void *>(this->mBuf), 
		    reinterpret_cast<const void *>(aRhs.mBuf), 
		    (this->mLen)*sizeof(u32));
      }
      std::memcpy(reinterpret_cast<void *>(this->mColLengths), 
		  reinterpret_cast<const void *>(aRhs.mColLengths),
		  4*sizeof(u16));
      this->mIsGood=aRhs.mIsGood;
    }
    return (*this);
  }


  BunchTrainBuffer::~BunchTrainBuffer()
  {
    delete [] mBuf;
    mBuf=0;
    mLen=0;
  }
  
  void BunchTrainBuffer::AllocateBuffer(u32 aLen){
    if(aLen!=mLen){
      delete [] mBuf;
      mBuf=new u32 [aLen];
      if(mBuf==0){
	// allocation error;
	std::cerr<<"error: unable to allocate requested memory buffer."<<std::endl;
      }
      mLen=aLen;
    }
  }
  
  bool BunchTrainBuffer::IsGood()const{
    return mIsGood;
  }

  void BunchTrainBuffer::ResetBuffer(u32 aLen){
    this->AllocateBuffer(aLen);
    std::memset(reinterpret_cast<void *>(mBuf), 0, mLen*sizeof(u32));
  }
  
  u32 BunchTrainBuffer::GetBufferLength()
  {
    return mLen;
  }

  const u32 * BunchTrainBuffer::GetBufferReference()const{
    return mBuf;
  }
  
  u32 * BunchTrainBuffer::GetBufferReference(){
    return mBuf;
  }

  u32 * BunchTrainBuffer::ExtractBuffer(){
    u32 * lTemp=mBuf;
    mBuf=0;
    mLen=0;
    return lTemp;
  }


  void BunchTrainBuffer::SetColumnLength(u16 aCol, u16 aLen)
  {
    aCol=aCol&0x3;
    mColLengths[aCol]=aLen;
  }

  std::ostream & operator<<(std::ostream & aOs, 
			    const BunchTrainBuffer & aBTB){
    
    aOs<<"Header:  "<<std::hex<<aBTB.mBuf[0]<<'\n';
    aOs<<"Header:  "<<std::hex<<aBTB.mBuf[1]<<'\n';
    aOs<<"Header:  "<<std::hex<<aBTB.mBuf[2]<<'\n';
    for(u32 i=3; i<aBTB.mLen; i++){
      u16 lTemp=(((~aBTB.mBuf[i])&0x0000e000)>>13);
      u16 lTemp2=lTemp>>2;
      u16 lTemp3=lTemp<<2;
      lTemp=lTemp&0x2 | lTemp2&0x1 | lTemp3&0x4; 

      aOs<<((aBTB.mBuf[i]&0x80000000)==0?' ':'*')<<'\t'
	 <<((aBTB.mBuf[i]&0x7fd00000)>>22)<<'\t'
	 <<((aBTB.mBuf[i]&0x003f0000)>>16)<<'\t'
	 <<lTemp<<'\t'
	 <<((aBTB.mBuf[i]&0x00001fff))<<'\t'
	 <<'\n';
    }
    return aOs;
  }
  
  BunchTrainInterpreter::BunchTrainInterpreter():
    mBuffer(0), 
    mIndex(0),
    mCurQuad(0),
    mHitBuf(0)
  {

  }
  BunchTrainInterpreter::BunchTrainInterpreter(BunchTrainBuffer & aBTB):
    mBuffer(&aBTB),
    mIndex(0),
    mCurQuad(0),
    mHitBuf(0)
  {
  }

  BunchTrainInterpreter::~BunchTrainInterpreter()
  {
    // the interpreter is not meant to own the buffer
    // it points at.
    mBuffer=0;
    mIndex=0;
    mCurQuad=0;
    mHitBuf=0;
  }
  
  void BunchTrainInterpreter::SetBuffer(BunchTrainBuffer & aBTB)
  {
    mBuffer=&aBTB;
  }
    

  u32 BunchTrainInterpreter::IntegrityCheck()
  {
    mBuffer->mIsGood=false;
    if(mBuffer){

      u32 lSum=0;

      for(u16 i=0; i<4; i++){
	lSum+=mBuffer->mColLengths[i];
      }
      if(lSum!=mBuffer->mLen){
	return 1;
      }
      
      u32 lStartBitMask=0x80000000;
      
      std::vector<u32> lStarts;
      lStarts.reserve(10);
      for(u32 i=0; i<mBuffer->mLen; i++){
	if(mBuffer->mBuf[i]&lStartBitMask){
	  lStarts.push_back(i);
	}
      }
      if(lStarts.size()!=4){
	// too many start bits:
	return 2;
      }
      u32 lColBufStart[4]={0, 
			   mBuffer->mColLengths[0], 
			   mBuffer->mColLengths[0]+mBuffer->mColLengths[1], 
			   mBuffer->mColLengths[0]+mBuffer->mColLengths[1]+mBuffer->mColLengths[2]};
      for(u16 i=0; i<4; i++){
	if(lStarts[i]!=lColBufStart[i]){
	  return 3;
	}
      }
    }
    mBuffer->mIsGood=true;
    return 0;
  }
    
//   u32 BunchTrainInterpreter::HitCount()
//   {
//     if(mBuffer)
//       return mBuffer->mLen; 
//     return 0;
//   }

//   u32 BunchTrainInterpreter::HitCount(u16 aCol)
//   {
//     aCol&=0x3;
//     return mBuffer->mColLengths[aCol];
//   }
    
//   PixelHit BunchTrainInterpreter::GetPixelHit(u32 aIdx)
//   {
//     PixelHit lTemp;
//     this->operator()(aIdx, lTemp);
//     return lTemp;
//   }
  

//   void BunchTrainInterpreter::operator()(u32 aIdx, PixelHit & aPH)
//   {

//   }
  
  void BunchTrainInterpreter::SetPixelHitBuffer(PixelHit * aPixHitBuf)const{
    mHitBuf=aPixHitBuf;
  }

  
  void BunchTrainInterpreter::Begin()const{
    mIndex=0;
  }
  
  u16 BunchTrainInterpreter::GetNextPixelHitSet()const{
    static const u16 * const lMuxLUT=BunchTrainInterpreter::GetMuxLUT();
    if(mBuffer==0)
      return 0;
    if(mHitBuf==0)
      return 0;
    if(mIndex==mBuffer->mLen)
      return 0;
    
    
    u16 lHitCount=0;
    if(mBuffer){
      u16 lHitPat=(mBuffer->mBuf[mIndex]>>16)&0x3f;
      u16 lHitMask=0x20;
      for(u16 i=0; i<6; i++){
	if(lHitPat&(lHitMask<<i)){
	  mHitBuf[i].mTimeCode=mBuffer->mBuf[mIndex]&0x1fff;
	  u16 lTemp=((mBuffer->mBuf[mIndex])>>13)&0x7;
	  mHitBuf[i].mMuxAddr=((lTemp>>2)&0x1)|(lTemp&0x2)|((lTemp<<2)&0x4);
	  mHitBuf[i].mRowAddr=((mBuffer->mBuf[mIndex])>>22)&0x1ff;
	  mHitBuf[i].mColAddr=(42*mCurQuad)+(lMuxLUT[mHitBuf[i].mMuxAddr]*6)+i;
	  lHitCount++;
	}
      }
    }
    mIndex++;
    return lHitCount;
  }
  
  const u16 * const BunchTrainInterpreter::GetMuxLUT()
  {
    static u16 lMuxLUT[7]={4,3,5,6,1,2,0};
    return lMuxLUT;
  }


  
  std::ostream & operator<<(std::ostream & aOs, const BunchTrainInterpreter & aBTI)
  {
    aBTI.Begin();
    PixelHit lPH[6];
    
    aBTI.SetPixelHitBuffer(lPH);
    u16 lHitCount=0;
    aBTI.Begin();
    while((lHitCount=aBTI.GetNextPixelHitSet())){
      // user code
      for(u16 i=0; i<lHitCount; i++){
	aOs<<lPH[i]<<std::endl;
      }
    } 
    
    return aOs;
  }

}//~namespace CALICEMAPS1


