#include "CaliceMaps1TestSystemInterface.hh" 

#include "USBDAQException.hh"
#include "PollLimitException.hh"
#include "HardwareAccessException.hh"
#include "LinuxLibUsbInterface.hh"

namespace CALICEMAPS1
{
  
  int CaliceMaps1TestSystemInterface::Instance(CaliceMaps1TestSystemInterface ** aCM1TSI, 
					     int aLen)
  {
    CaliceMaps1TestSystemInterface * lTemp=0;
    int lCount=0;

    for(int i=0; i<aLen; i++){
      lTemp=0;
      try{
	//std::cout<<": "<<i<<std::endl;
	lTemp=new CaliceMaps1TestSystemInterface();
      }catch(USBDAQException & aExc){
	//RETHROW(aExc);
	// we're expecting exceptions:
      }
      if(lTemp){
	aCM1TSI[lCount]=lTemp;
	lCount++;
      }
      
    }
    return lCount;
  }

  
  CaliceMaps1TestSystemInterface::CaliceMaps1TestSystemInterface():
    mInterface(0), 
    mLastTemp(0),
    mReadoutStartIndex(0),
    mMode(0)
  {
    try{
      mInterface=new LinuxLibUsbInterface("",
					  0xbd7,
					  0x1003,
					  0x0000);
      
      // deal with the address space size:
      mInterface->SetAddressSpaceSize((LASTREG));
      
      //std::cout<<"Register space size = "<<std::dec<<LASTREG<<std::endl;
      this->EnableLVDSBuffers();
      this->SetupReferences();
      this->SetupDiscriminatorReferenceDAC();
      
      std::cout<<"Firmware revision = "<<this->ReadFirmwareRevision()<<std::endl;
      time_t lTime=this->ReadFirmwareRevisionUTC();
      char lBuf[128];
      //cout<<"Built = "<<lTime<<endl;
      
      std::cout<<"Built : "<<ctime_r(reinterpret_cast<const time_t *>(&lTime),
 				     lBuf)<<std::endl;
      std::cout<<"DAQ Address = "<<this->ReadDAQAddress()<<std::endl;
      //      this->ClearConfigReadoutMode();
      
      this->SetClockPhase(2, 180);

      this->SetTriggerMask(0x1f);
      this->SetTriggerSource(0);
      this->SetTriggerEnable(1);
      
      // read the sensor ID and set the 
      // sensor version accordingly:
      //  v1.0 = [0,20]
      //  v1.1 > 20
      u16 lID=this->ReadSensorId();
      if(lID>20){
	this->SetSensorVersion(1);
      }else{
	this->SetSensorVersion(0);
      }
      
//       for(u32 i=0; i<10; i++){
//  	mInterface->Write(CLK_0, i);
//  	u32 lTemp=0;
//  	mInterface->Read(CLK_0, lTemp);
//  	std::cout<<i<<" should be "<<lTemp<<std::endl;
//        }
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  CaliceMaps1TestSystemInterface::~CaliceMaps1TestSystemInterface()
  {
    delete mInterface;
    mInterface=0;
  }

  void CaliceMaps1TestSystemInterface::LogicReset()
  {
    try{
      this->IssueTrigger(0x80000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  void CaliceMaps1TestSystemInterface::BufferReset()
  {
    try{
      this->IssueTrigger(0x40000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }



  void CaliceMaps1TestSystemInterface::StartSpill()
  {
    try{ 
      this->IssueTrigger(1<<7, 1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  void CaliceMaps1TestSystemInterface::StopSpill()
  {
    try{ 
      this->IssueTrigger(1<<8);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 

  }
  
  void CaliceMaps1TestSystemInterface::SetSpillCycleCount(u32 aCount){
    try{ 
      mInterface->Write(SPILL_CTRL_3, aCount,0xffff);
      mSpillCycleCount=aCount;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  
  u32 CaliceMaps1TestSystemInterface::ReadSpillCycleCount(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_3, lTemp);
      lTemp&=0xffff;
      if(lTemp!=mSpillCycleCount){
	std::cerr<<__PRETTY_FUNCTION__<<" : Readback check"<<std::endl;
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return 0xffff&lTemp;
  }
   

  void CaliceMaps1TestSystemInterface::SetSpillCycleStart(u16 aStart)
  {
    try{ 
      u32 lTemp=aStart<<16;
      lTemp&=0x1fff0000;
      mInterface->Write(SPILL_CTRL_3, lTemp,0x1fff0000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  u16 CaliceMaps1TestSystemInterface::ReadSpillCycleStart()
  {
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_3, lTemp); 
      //std::cout<<__PRETTY_FUNCTION__<<" : "<<lTemp<<std::endl;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (0x1fff&static_cast<u16>(lTemp>>16));
  }

  
  void CaliceMaps1TestSystemInterface::SetSensorVersion(u16 lVersion)
  { 
    try{ 
      u32 lTemp=lVersion&0x1;
      if(lTemp){
	std::cout<<"INFO: mux will be set for TPAC v1.1."<<std::endl;
      }else{
	std::cout<<"INFO: mux will be set for TPAC v1.0."<<std::endl;
      }

      lTemp<<=19;
      mInterface->Write(GEN_SETTINGS_2, lTemp, 0x80000);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }    
  }
  
  
  u16 CaliceMaps1TestSystemInterface::ReadSensorVersion()
  {
    u16 lVer=0;
    try{ 
      u32 lTemp=0;
      mInterface->Read(GEN_SETTINGS_2,lTemp);
      lVer=(lTemp&0x80000)?(1):(0);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }    
    return lVer;
  }


  void CaliceMaps1TestSystemInterface::SetPMTCoincidenceCountEnable(bool aEn)
  {
    try{ 
      u32 lTemp=aEn?(1<<18):(0);
      mInterface->Write(GEN_SETTINGS_2, lTemp, (1<<18));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  bool CaliceMaps1TestSystemInterface::ReadPMTCoincidenceCountEnable()
  {
    u32 lTemp=0;
    try{ 
      mInterface->Read(GEN_SETTINGS_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&(1<<18));
  }
  void CaliceMaps1TestSystemInterface::TriggerPMTCoincidenceCountReset()
  {
     try{ 
       this->IssueTrigger(0x8000);
     }catch(USBDAQException & aExc){
       RETHROW(aExc);
     } 
  }
  u32 CaliceMaps1TestSystemInterface::ReadPMTCoincidenceCount()
  {
    u32 lCount=0;
    try{
      this->SetPMTCoincidenceCountEnable(true);
      this->TriggerPMTCoincidenceCountReset();
      u32 lPollLimit(0);
      while(((lCount=this->ReadStatus())&0x80000000)==0){
	std::cout<<"Poll: "<<lPollLimit++<<std::endl;
      }
      lCount=lCount&0x7fffffff;
      lCount=lCount>>8;

      this->SetPMTCoincidenceCountEnable(false);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }    
    return lCount;
  }
  





  void CaliceMaps1TestSystemInterface::SetClockPhase(u16 aClk, u16 aPhase)
  {
    if(aClk>2){
      
    }

    try{
      u32 lTemp=(aPhase&0xff);
      u32 lMask=0xff;
      if(aClk==1){
	lTemp<<=8;
	lMask<<=8;
      }
      if(aClk==2){
	lTemp<<=16;
	lMask<<=16;
      }
      mInterface->Write(CLK_0, lTemp, lMask);
      
      if(aClk==1)	
	this->IssueTrigger(1<<10);
      if(aClk==0)
	this->IssueTrigger(1<<9);
      if(aClk==2)
	this->IssueTrigger(1<<14);
      
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }


  u16 CaliceMaps1TestSystemInterface::ReadClockPhase(u16 aClk)
  {
    u32 lTemp=0;
    try{
      mInterface->Read(CLK_0, lTemp);
      if(aClk==1){
	lTemp>>=8;
	
      }
      if(aClk==2){
	lTemp>>=16;
      }

      lTemp&=0xff;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<u16>(lTemp);
  }
  
  
  void CaliceMaps1TestSystemInterface::SetFastPhiOverride(bool aOvrd){
    try{ 
      u32 lTemp=aOvrd?0x80000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x80000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadFastPhiOverride(){
    u32 lTemp=0;
   
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x80000000;
  }


  void CaliceMaps1TestSystemInterface::SetTestTrigStart(u32 aStart){
    try{ 
      mInterface->Write(TEST_TRIG_0, aStart, 0xffffffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  u32 CaliceMaps1TestSystemInterface::ReadTestTrigStart(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(TEST_TRIG_0, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp;
  }
  
  void CaliceMaps1TestSystemInterface::SetTestTrigStop(u32 aStop){
    try{ 
     mInterface->Write(TEST_TRIG_1, aStop, 0xffffffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  u32 CaliceMaps1TestSystemInterface::ReadTestTrigStop(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(TEST_TRIG_1, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }
  
  void CaliceMaps1TestSystemInterface::SetTestTrigConfig(u16 aConfig){
    try{ 
      mInterface->Write(TEST_TRIG_2, aConfig, 0xf);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  

  u16  CaliceMaps1TestSystemInterface::ReadTestTrigConfig(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(TEST_TRIG_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xf);
  }





  void CaliceMaps1TestSystemInterface::SetSlowSpillPhi(bool aSlowSpillPhi){
    try{ 
      u32 lTemp=aSlowSpillPhi?0x40000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x40000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadSlowSpillPhi(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x40000000;
  }

  void CaliceMaps1TestSystemInterface::SetReadEnableOverride(bool aRdEnOvrd){
    try{ 
      u32 lTemp=aRdEnOvrd?0x20000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x20000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadReadEnableOverride(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x20000000;
  }
  void CaliceMaps1TestSystemInterface::SetRstBPolarity(bool aPol){
    try{ 
      u32 lTemp=aPol?0x10000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x10000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadRstBPolarity(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x10000000;
  }


  void CaliceMaps1TestSystemInterface::SetMCResetDuration(u16 aRstLen){
    try{ 
      u32 lTemp=aRstLen&0xff;
      mInterface->Write(SPILL_CTRL_5, lTemp,0xff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  u16 CaliceMaps1TestSystemInterface::ReadMCResetDuration(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_5,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0xff;
  }


  
  void CaliceMaps1TestSystemInterface::SetStackPointerOffset(u16 aStackOffset){
    try{ 
      u32 lTemp=(aStackOffset&0x1f)<<8;
      mInterface->Write(SPILL_CTRL_5, lTemp,0x1f<<8);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  u16 CaliceMaps1TestSystemInterface::ReadStackPointerOffset(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_5,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp>>8)&0x1f;
  }

  void CaliceMaps1TestSystemInterface::SetPostPixResetSleepDuration(u16 aPRstSleep){
    try{ 
      mInterface->Write(SPILL_CTRL_5,
			static_cast<u32>(aPRstSleep)<<16, 
			0xffff0000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadPostPixResetSleepDuration(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_5,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp>>16)&0xffff;
  }


  void CaliceMaps1TestSystemInterface::SetInitBPolarity(bool aPol){
    try{ 
      u32 lTemp=aPol?0x08000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x08000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadInitBPolarity(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x08000000;
  }
  void CaliceMaps1TestSystemInterface::SetFwdBPolarity(bool aPol){
    try{ 
      u32 lTemp=aPol?0x04000000:0;
      mInterface->Write(SPILL_CTRL_2, lTemp,0x04000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadFwdBPolarity(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x04000000;
  }

  
  void CaliceMaps1TestSystemInterface::SetReadoutColumnOrder(u16 aCols){
    try{ 
      u32 lTemp=aCols<<18;
      mInterface->Write(SPILL_CTRL_2, lTemp, 0xff<<18);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  u16 CaliceMaps1TestSystemInterface::ReadReadoutColumnOrder(){
    u32 lTemp=0;
    try{ 

      mInterface->Read(SPILL_CTRL_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp>>18)&0xff;
  }

  
  void CaliceMaps1TestSystemInterface::SetReadoutStartIndex(u16 aStartIndex){
    if(aStartIndex>3){
      //error:
    }
    try{ 
      u32 lTemp=aStartIndex<<16;
      mReadoutStartIndex=aStartIndex&0x3;
      mInterface->Write(SPILL_CTRL_2, lTemp, 0x3<<16);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  u16 CaliceMaps1TestSystemInterface::ReadReadoutStartIndex(){
    u32 lTemp=0;
    try{ 
      
      mInterface->Read(SPILL_CTRL_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<u16>((lTemp>>16)&0x3);
  }
    
  
  void CaliceMaps1TestSystemInterface::SetStaticTimeStamp(u16 aTS){
    try{ 
      u32 lTemp=(aTS<<16);
      mInterface->Write(SPILL_CTRL_4, lTemp, 0x1fff<<16);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
   
  u16 CaliceMaps1TestSystemInterface::ReadStaticTimeStamp(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_4, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp>>16)&0x1fff;
  }
 
  void CaliceMaps1TestSystemInterface::SetSramFillClockSleep(u16 aSleep){
    try{ 
      u32 lTemp=((aSleep&0xff)<<8);
      mInterface->Write(SPILL_CTRL_2, lTemp, (0xff<<8));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  

  u16 CaliceMaps1TestSystemInterface::ReadSramFillClockSleep(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp>>8)&0xff;
  }

  void CaliceMaps1TestSystemInterface::SetTimeStampOverride(bool aOvrd){
    try{ 
      u32 lTemp=aOvrd?0x20000000:0;
      mInterface->Write(SPILL_CTRL_4, lTemp,0x20000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }


  bool CaliceMaps1TestSystemInterface::ReadTimeStampOverride(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(SPILL_CTRL_4, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return lTemp&0x20000000;
  }
  

  u32 CaliceMaps1TestSystemInterface::ReadFirmwareRevision(){
    u32 lTemp=0;
    //return 0;
    try{ 
      mInterface->Read(FW_REV, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp);
  }
  
  u32 CaliceMaps1TestSystemInterface::ReadFirmwareRevisionUTC(){
   u32 lTemp=0;
   //return 0;
    try{ 
      mInterface->Read(FW_TIME, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return (lTemp);
  }
  
  void CaliceMaps1TestSystemInterface::SetMonoPOR(bool aPOR){
    try{ 
      u32 lTemp=aPOR? 0x80000000:0;
      mInterface->Write(DEBUG_SETTINGS, lTemp,0x80000000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  bool CaliceMaps1TestSystemInterface::ReadMonoPOR() {
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&0x80000000);
  }
        
  void CaliceMaps1TestSystemInterface::SetPixelEnable12(bool aEb12){
    try{ 
      u32 lTemp=aEb12?1<<29:0;
      mInterface->Write(DEBUG_SETTINGS, lTemp,1<<29);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  bool CaliceMaps1TestSystemInterface::ReadPixelEnable12(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&(1<<29));
  }
        
 
  void CaliceMaps1TestSystemInterface::SetPixelEnable34(bool aEb34){
    try{ 
      u32 lTemp=aEb34? 1<<28 :0;
      mInterface->Write(DEBUG_SETTINGS,lTemp,1<<28);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    
  }
  bool CaliceMaps1TestSystemInterface::ReadPixelEnable34(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&(1<<28));
  } 
    

  void CaliceMaps1TestSystemInterface::SetSenseEnable(bool aSenseEn){
    try{ 
      u32 lTemp=aSenseEn? 1<<30:0;
      mInterface->Write(DEBUG_SETTINGS,lTemp,1<<30);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  
  bool CaliceMaps1TestSystemInterface::ReadSenseEnable(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&(1<<30));
  }

  void CaliceMaps1TestSystemInterface::SetDebugHitInEnable(bool aDbgEn,
							   u16 aLen){
    try{ 
      u32 lTemp=static_cast<u32>(aLen)<<5;
      //      u32 lMask=;//<<5;
      mInterface->Write(G_CONFIG, lTemp,0x0001fffe0);
      
      lTemp=aDbgEn? 1<<27:0;
      mInterface->Write(DEBUG_SETTINGS,lTemp,1<<27);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }

  bool CaliceMaps1TestSystemInterface::ReadDebugHitInEnable(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&(1<<27));
  }
   

  void CaliceMaps1TestSystemInterface::SetHitOverride(bool aOvrd){
    try{ 
      u32 lTemp=aOvrd? 1<<23:0;
      mInterface->Write(DEBUG_SETTINGS,lTemp,1<<23);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  
  bool CaliceMaps1TestSystemInterface::ReadHitOverride(){
    u32 lTemp=0;
    try{ 
      mInterface->Read(DEBUG_SETTINGS,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<bool>(lTemp&(1<<23));
  }


  
  void CaliceMaps1TestSystemInterface::SetDebugVthPos(u16 aPos){
    try{ 
      this->SetReferenceLevel(DEBUG_VTH_POS ,aPos);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadDebugVthPos(){
    try{
      return this->ReadReferenceLevel(DEBUG_VTH_POS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetDebugVthNeg(u16 aNeg){

    try{ 
      this->SetReferenceLevel(DEBUG_VTH_NEG ,aNeg);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadDebugVthNeg(){
    try{
      return this->ReadReferenceLevel(DEBUG_VTH_NEG);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  //   
  void CaliceMaps1TestSystemInterface::SetVth12Pos(u16 aPos){
    try{ 
      this->SetReferenceLevel(VTH_12_POS,aPos);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadVth12Pos(){
    try{
      return this->ReadReferenceLevel(VTH_12_POS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetVth12Neg(u16 aNeg){
    try{ 
      this->SetReferenceLevel(VTH_12_NEG,aNeg);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadVth12Neg(){
    try{
      return this->ReadReferenceLevel(VTH_12_NEG);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetVth34Pos(u16 aPos){
    try{ 
      this->SetReferenceLevel(VTH_34_POS,aPos);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadVth34Pos(){
    try{
      return this->ReadReferenceLevel(VTH_34_POS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void CaliceMaps1TestSystemInterface::SetVth34Neg(u16 aNeg){
    try{ 
      this->SetReferenceLevel(VTH_34_NEG,aNeg);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadVth34Neg(){
    try{
      return this->ReadReferenceLevel(VTH_34_NEG);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  //
  void CaliceMaps1TestSystemInterface::SetVRst(u16 aRst){
    try{ 
      this->SetReferenceLevel(VRST,aRst);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadVRst(){
    try{
      return this->ReadReferenceLevel(VRST);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetISenseColRef(u16 aISense){
    try{ 
      this->SetReferenceLevel(ISENSE_COLREF,aISense);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadISenseColRef(){
    try{
      return this->ReadReferenceLevel(ISENSE_COLREF);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetPreAmp34VCasc(u16 aVCasc){
    try{ 
      this->SetReferenceLevel(PREAMP_VCASC_34,aVCasc);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadPreAmp34VCasc(){
    try{
      return this->ReadReferenceLevel(PREAMP_VCASC_34);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
    
  void CaliceMaps1TestSystemInterface::SetISenseBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(ISENSE_BIAS, aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadISenseBias(){
    try{
      return this->ReadReferenceLevel(ISENSE_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetI12Comp1Bias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_COMP1_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12Comp1Bias(){
    try{
      return this->ReadReferenceLevel(I12_COMP1_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetI12Comp2Bias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_COMP2_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12Comp2Bias(){
    try{
      return this->ReadReferenceLevel(I12_COMP2_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
    
  void CaliceMaps1TestSystemInterface::SetISenseIoutBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(ISENSE_IOUT_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadISenseIoutBias(){
    try{
      return this->ReadReferenceLevel(ISENSE_IOUT_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }



  // 
  void CaliceMaps1TestSystemInterface::SetISenseCompBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(ISENSE_COMP_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadISenseCompBias(){
    try{
      return this->ReadReferenceLevel(ISENSE_COMP_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI12IoutBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_IOUT_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12IoutBias(){
    try{
      return this->ReadReferenceLevel(I12_IOUT_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  //   void CaliceMaps1TestSystemInterface::SetI12MSOBias1(u16 aBias){
  //     try{ 
  //       this->SetReferenceLevel(ISENSE_IOUT_BIAS,aBias);
  //     }catch(USBDAQException & aExc){
  //       RETHROW(aExc);
  //     }
  //   }
  //   u16 CaliceMaps1TestSystemInterface::ReadI12MSOBias1 try{
  //     return this->ReadReferenceLevel(ISENSE_IOUT_BIAS);
  //   }catch(USBDAQException & aExc){
  //     RETHROW(aExc);
  //   }

  void CaliceMaps1TestSystemInterface::SetI12PreAmpBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_PRE_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadI12PreAmpBias(){
    try{
      return this->ReadReferenceLevel(I12_PRE_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI12ShaperBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_SHAPER_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12ShaperBias(){
    try{
      return this->ReadReferenceLevel(I12_SHAPER_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void CaliceMaps1TestSystemInterface::SetI12CompBiasTrim(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_COMP_BIAS_TRIM,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12CompBiasTrim(){
    try{
      return this->ReadReferenceLevel(I12_COMP_BIAS_TRIM);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void CaliceMaps1TestSystemInterface::SetI34CompBiasTrim(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_COMP_BIAS_TRIM,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34CompBiasTrim(){
    try{
      return this->ReadReferenceLevel(I34_COMP_BIAS_TRIM);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI34DebugSFBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_DEBUGSF_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34DebugSFBias(){
    try{
      return this->ReadReferenceLevel(I34_DEBUGSF_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetI34SFBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_SF_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34SFBias(){
    try{
      return this->ReadReferenceLevel(I34_SF_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI34CompBias1(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_COMP_BIAS1,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  u16 CaliceMaps1TestSystemInterface::ReadI34CompBias1(){
    try{
      return this->ReadReferenceLevel(I34_COMP_BIAS1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  
  void CaliceMaps1TestSystemInterface::SetI34CompBias2(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_COMP_BIAS2,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadI34CompBias2(){
    try{
      return this->ReadReferenceLevel(I34_COMP_BIAS2);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetI34MSOBias1(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_MSO_BIAS1,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34MSOBias1(){
    try{
      return this->ReadReferenceLevel(I34_MSO_BIAS1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  void CaliceMaps1TestSystemInterface::SetI34MSOBias2(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_MSO_BIAS2,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadI34MSOBias2(){
    try{
      return this->ReadReferenceLevel(I34_MSO_BIAS2);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::SetI34PreAmpBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_PRE_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34PreAmpBias(){
    try{
      return this->ReadReferenceLevel(I34_PRE_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI34OutBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_OUT_BIAS,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadI34OutBias(){
    try{
      return this->ReadReferenceLevel(I34_OUT_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }    
  }

  void CaliceMaps1TestSystemInterface::SetI34OutSFBias(u16 aBias){
    try{ 
      this->SetReferenceLevel(I34_OUTSF_BIAS, aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI34OutSFBias(){
    try{
      return this->ReadReferenceLevel(I34_OUTSF_BIAS);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetI12MSOBias1(u16 aBias){
    try{ 
      this->SetReferenceLevel(I12_MSO_BIAS1,aBias);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadI12MSOBias1(){
    try{
      return this->ReadReferenceLevel(I12_MSO_BIAS1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  //

  void CaliceMaps1TestSystemInterface::SetShaper12VCasc(u16 aVCasc){
    try{ 
      this->SetReferenceLevel(SHAPE_VCASC_12,aVCasc);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadShaper12VCasc(){
    try{
      return this->ReadReferenceLevel(SHAPE_VCASC_12);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetPreAmp12VCasc(u16 aVCasc){
    try{ 
      this->SetReferenceLevel(PREAMP_VCASC_12,aVCasc);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadPreAmp12VCasc(){
    try{
      return this->ReadReferenceLevel(PREAMP_VCASC_12);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

    
  

  //////////////////////////////////////////////////////////////




  
  void CaliceMaps1TestSystemInterface::SetSpillMode(u16 aMode)
  {
    try{
      u32 lTemp=static_cast<u32>(aMode&0x1ff);
      mInterface->Write(SPILL_CTRL_4, lTemp, (0x1ff));
      mMode=aMode&0x1ff;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadSpillMode()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_4,lTemp);
      if(mMode!=lTemp&0xff){
	// consistency check error:
	
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xff); 
  }
  
  void CaliceMaps1TestSystemInterface::SetConfigReadoutMode()
  {
    try{
      mInterface->Write(G_CONFIG, 0, 0x1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::ClearConfigReadoutMode()
  {
    try{
      mInterface->Write(G_CONFIG, 1, 0x1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u32 CaliceMaps1TestSystemInterface::ReadStatus(u32 lMask){
    u32 lTemp=0;
    try{ 
      mInterface->Read(STATUS_1, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&lMask);
  }

  u16 CaliceMaps1TestSystemInterface::ReadSensorStatusFlags()
  {
    u32 lTemp=0;
    try{       
      mInterface->Read(DEBUG_SETTINGS, lTemp);
      lTemp=((lTemp>>14)& 0xf0)|(lTemp&0xf);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp);
  }


  void CaliceMaps1TestSystemInterface::LoadConfig(const u8 *  aBufIn, 
						  u8 * aBufOut, 
						  u16 aRowLen,
						  u16 aNumRows){
    this->BufferReset();
    
    try{
      this->SetSlowConfigSRBitCount(1);
      this->SetReadbackConfigSRBitCount(1);
            
      const u8 * lRowStart=aBufIn;
      u8 * lOutput=aBufOut;
      for(u16 lRow=0;lRow<aNumRows;lRow++, lRowStart+=aRowLen, lOutput+=(aRowLen-21)){
	//std::cout<<"Loading row "<<lRow<<std::endl;
	this->LoadTopConfigRow(lRowStart, lOutput, aRowLen);
	this->TriggerSlowConfigShift();
      }

      this->SetConfigReadoutMode();
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    
  void CaliceMaps1TestSystemInterface::ReadBackConfig(u8 * aBufOut, 
						      u16 aRowLen,
						      u16 aNumRows){
    try{
      u8 * lPtr=aBufOut;
      for(u16 lRow=0;lRow<aNumRows;lRow++,lPtr+=aRowLen){
	//std::cout<<"reading back row "<<lRow<<std::endl;
	this->TriggerSlowConfigShift();
	this->ReadBackBottomRow(lPtr, aRowLen);
      }
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
    

  void CaliceMaps1TestSystemInterface::LoadTopConfigRow(const u8 * aBufIn,
							u8 * aBufOut,
							u16 aBufLen){

    try{
      u16 lBitCount=8*aBufLen;
      i16 lRecapLen=aBufLen-21;
      
      // should test the buffer size. 
      //std::cout<<"Loading     "<<aBufLen<<" bytes"<<std::endl;
      //std::cout<<"Recapturing "<<lRecapLen<<" bytes"<<std::endl;
      //std::cout<<"Setting     "<<lBitCount<<" clock cycles"<<std::endl;
      
      this->ClearConfigReadoutMode();
      this->SetFastConfigSRBitCount(lBitCount);
      //std::cout<<"Readback = "<<this->ReadFastConfigSRBitCount()<<std::endl;
      mInterface->WriteBlock(aBufIn, aBufLen);
      this->TriggerTopConfigLoad();
      
      u32 lCount=0;
      // we're looking at a busy bit and waiting 
      // for it to clear:
      while(this->ReadStatus(0x1)){
	lCount++;
	
	if(lCount>10000){
	  PollLimitException lExc(lCount);
	  RAISE(lExc);
	}
      }
      
      this->SetConfigReadoutMode();
      mInterface->ReadBlock(aBufOut, 
			    lRecapLen);
      
      this->ClearConfigReadoutMode();
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::ReadBackBottomRow(u8 * aBuf, 
							 u16 aBufLen, 
							 bool aInhibitVShift)
  {
    try{ 
      u16 lBitCount=8*aBufLen;
      if(!aInhibitVShift){
	this->SetSlowConfigSRBitCount(1);
	this->SetReadbackConfigSRBitCount(lBitCount);
      }
      this->TriggerReadbackConfigShift();
      this->SetConfigReadoutMode();
      u32 lCount=0;
      // we're looking at a busy bit and waiting 
      // for it to clear:
      while(this->ReadStatus(0x1)){
	lCount++;
	//std::cout<<"count - "<<lCount<<std::endl;
	if(lCount>10000){
	  PollLimitException lExc(lCount);
	  RAISE(lExc);
	}
      }
      mInterface->ReadBlock(aBuf, aBufLen);
      this->ClearConfigReadoutMode();
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
   
  void CaliceMaps1TestSystemInterface::LoadBottomRow(u8 * aBuf)
  { 
    u16 lNumBytes=21;
    u16 lBitCount=lNumBytes*8;
        
    try{
      //
      this->SetBottomRowLoad(1);
      //this->ReadBottomRowLoad()
      //
      this->ClearConfigReadoutMode();
      this->SetReadbackConfigSRBitCount(lBitCount);
      this->SetFastConfigSRBitCount(lBitCount);
      // 
      // 
      mInterface->WriteBlock(aBuf, lNumBytes);
      //
      this->TriggerReadbackConfigShift();

      u32 lCount=0;
      while(this->ReadStatus(0x1)){
	lCount++;
	///std::cout<<"count - "<<lCount<<std::endl;
	if(lCount>10000){
	  PollLimitException lExc(lCount);
	  RAISE(lExc);
	}
      }
      
      this->SetBottomRowLoad(0);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetBottomRowLoad(u16 aBottomLoad)
  {
    try{
      u32 lTemp=aBottomLoad&1;
      lTemp<<=22;
      mInterface->Write(GEN_SETTINGS_2, lTemp, (1<<22));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadBottomRowLoad()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      mInterface->Read(GEN_SETTINGS_2, lTemp);
      lRet=(lTemp>>22)&0x1;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
  

   
  void CaliceMaps1TestSystemInterface::TriggerTopConfigLoad()
  {
    try{
      this->IssueTrigger(2);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  void CaliceMaps1TestSystemInterface::IssueTopConfigReset(){
    try{
      this->IssueTrigger(1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void CaliceMaps1TestSystemInterface::TriggerSlowConfigShift(u16 aNoClks)
  {
    try{
      this->SetReadbackConfigSRBitCount(1);
      this->SetSlowConfigSRBitCount(aNoClks);
      this->IssueTrigger(1<<3);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void CaliceMaps1TestSystemInterface::IssueSlowConfigReset()
  {
    try{
      this->IssueTrigger(1<<2);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::TriggerReadbackConfigShift()
  {
    try{
      this->IssueTrigger(1<<5);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  //   void CaliceMaps1TestSystemInterface::TriggerReadbackConfigPLoad()
  //   {
  //     try{
  //       // not implemented yet: timing needs clarification.
  //       //this->IssueTrigger(1<<2);
  //     }catch(USBDAQException & aExc){
  //       RETHROW(aExc);
  //     }
  //   }

  void CaliceMaps1TestSystemInterface::IssueReadbackConfigReset()
  {
    try{
      this->IssueTrigger(1<<4);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }



  u16 CaliceMaps1TestSystemInterface::ReadReadBackConfigSRBitCount()
  {
    u32 lTemp=0;  
    try{
      mInterface->Read(PHI_CTRL_1, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xffff);
  }
   
  void CaliceMaps1TestSystemInterface::SetReadbackConfigSRBitCount(u16 aBC)
  {
    try{
      u32 lTemp=(aBC);//<<16);
      mInterface->Write(PHI_CTRL_1, lTemp, 0xffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    
  }

  u16 CaliceMaps1TestSystemInterface::ReadFastConfigSRBitCount()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(PHI_CTRL_0, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xffff);
  }
  
  void CaliceMaps1TestSystemInterface::SetFastConfigSRBitCount(u16 aBC)
  {
    try{
      mInterface->Write(PHI_CTRL_0, aBC, 0xffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  
  u16 CaliceMaps1TestSystemInterface::ReadSlowConfigSRBitCount()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(PHI_CTRL_0, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp>>16);
  }
  
  void CaliceMaps1TestSystemInterface::SetSlowConfigSRBitCount(u16 aBC)
  {
    try{
      u32 lTemp=aBC<<16;
      mInterface->Write(PHI_CTRL_0, lTemp, 0xffff0000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  
  /////////////////////////////
  // these methods have been added to try to understand 
  // the configuration system errors. 
  // 
  
  void CaliceMaps1TestSystemInterface::SetSlowConfigPhi2Override(bool aEnable, bool aValue)
  {
    try{
      u32 lTemp=0;
      if(aEnable){
	lTemp|=0x20000000;
      }
      if(aValue){
	lTemp|=0x10000000;
      }
      mInterface->Write(PHI_CTRL_1, lTemp, 0x30000000); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadSlowConfigPhi2Override()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(PHI_CTRL_1, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>((lTemp>>28)&0x3);
  }
  
  void CaliceMaps1TestSystemInterface::SetSlowConfigPhi3Override(bool aEnable, bool aValue)
  {
    try{
      u32 lTemp=0;
      if(aEnable){
	lTemp|=0x80000000;
      }
      if(aValue){
	lTemp|=0x40000000;
      }
      mInterface->Write(PHI_CTRL_1, lTemp, 0xc0000000); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadSlowConfigPhi3Override()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(PHI_CTRL_1, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>((lTemp>>30)&0x3);
  }

  void CaliceMaps1TestSystemInterface::SetSlowConfigRstOverride(bool aEnable, bool aValue)
  {
    try{
      u32 lTemp=0;
      if(aEnable){
	lTemp|=0x08000000;
      }
      if(aValue){
	lTemp|=0x04000000;
      }
      mInterface->Write(PHI_CTRL_1, lTemp, 0x0c000000); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    
  }
  u16 CaliceMaps1TestSystemInterface::ReadSlowConfigRstOverride()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(PHI_CTRL_1, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>((lTemp>>26)&0x3);
  }

  //
  //////////////////////////////////


  u32 CaliceMaps1TestSystemInterface::ReadDAQWordCount()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(G_STATUS, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&0xffff);
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadSenseDelay()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_1, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp>>16)&0xff;
  }
  
  void CaliceMaps1TestSystemInterface::SetSenseDelay(u16 aDly)
  {
    try{
      u32 lTemp=(aDly&0xff)<<16;
      mInterface->Write(SPILL_CTRL_1, lTemp, (0xff<<16)); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  

  u16 CaliceMaps1TestSystemInterface::ReadSpillControllerStatus()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_0, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp>>28)&0xf;
  }

  u16 CaliceMaps1TestSystemInterface::ReadPreAmpResetDuration()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_1, lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xffff);
  }
  
  void CaliceMaps1TestSystemInterface::SetPreAmpResetDuration(u16 aRstLen)
  {
    try{
      mInterface->Write(SPILL_CTRL_1, aRstLen, 0xffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadShaperResetDuration()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_0,lTemp); 
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (static_cast<u16>(lTemp>>16)&0xffff);
  }

  void CaliceMaps1TestSystemInterface::SetShaperResetDuration(u16 aRstLen)
  {
    try{
      mInterface->Write(SPILL_CTRL_0, 
			static_cast<u32>(aRstLen)<<16, 
			0xffff0000);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadDiodeResetDuration()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_0,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xffff);
  }
  
  void CaliceMaps1TestSystemInterface::SetDiodeResetDuration(u16 aRstLen)
  {
    try{
      mInterface->Write(SPILL_CTRL_0, aRstLen, 0xffff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  

  u16 CaliceMaps1TestSystemInterface::ReadTimeStampPrescale()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(SPILL_CTRL_2, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp&0xff);
  }
  
  void CaliceMaps1TestSystemInterface::SetTimeStampPrescale(u16 aPS)
  {
    try{
      
      mInterface->Write(SPILL_CTRL_2, aPS,0xff);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  
  void CaliceMaps1TestSystemInterface::TriggerDebugReset()
  {
    try{
      this->IssueTrigger(1<<11);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::SetDebugReset200Duration(u16 aRstLen)
  {
    try{
      u32 lTemp=((aRstLen&0xff)<<12);
      mInterface->Write(DEBUG_SETTINGS, lTemp, (0xff<<12));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }   
  }


  void CaliceMaps1TestSystemInterface::SetDebugReset600Duration(u16 aRstLen)
  {
    try{
      u32 lTemp=((aRstLen&0xff)<<4);
      mInterface->Write(DEBUG_SETTINGS, lTemp, (0xff<<4));      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  void CaliceMaps1TestSystemInterface::SetDebugDiodeResetDuration(u16 aRstLen){
    try{
      u32 lTemp=((aRstLen&0xff)<<24);
      mInterface->Write(G_CONFIG, lTemp, (0xff<<24));      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }


  void CaliceMaps1TestSystemInterface::SetDebugReset200Hold(bool aHold){
    try{
      u32 lTemp=aHold?1<<26:0;
	mInterface->Write(DEBUG_SETTINGS, lTemp, 1<<26);      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
 
  bool CaliceMaps1TestSystemInterface::ReadDebugReset200Hold(){
    u32 lTemp=0;
    try{
      mInterface->Read(DEBUG_SETTINGS, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<bool>(lTemp&(1<<26));
  }
  
  void CaliceMaps1TestSystemInterface::SetDebugReset600Hold(bool aHold){
    try{
      u32 lTemp=aHold?1<<25:0;
	mInterface->Write(DEBUG_SETTINGS, lTemp, 1<<25);      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  
  bool CaliceMaps1TestSystemInterface::ReadDebugReset600Hold(){
    u32 lTemp=0;
    try{
      mInterface->Read(DEBUG_SETTINGS, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<bool>(lTemp&(1<<25));
  }

  void CaliceMaps1TestSystemInterface::SetDebugDiodeResetHold(bool aHold){
    try{
      u32 lTemp=aHold?(1<<24):0;
      mInterface->Write(DEBUG_SETTINGS, lTemp, 1<<24);      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
 
  bool CaliceMaps1TestSystemInterface::ReadDebugDiodeResetHold(){
    u32 lTemp=0;
    try{
      mInterface->Read(DEBUG_SETTINGS, lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<bool>(lTemp&(1<<24));
  }

  void CaliceMaps1TestSystemInterface::SetDiodeReset12Hold(bool aHold)
  {
    try{
      u32 lTemp=aHold?0x10:0;
      mInterface->Write(GEN_SETTINGS_2, lTemp, 0x10);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  bool CaliceMaps1TestSystemInterface::ReadDiodeReset12Hold(){
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&0x10);
  }
  
  void CaliceMaps1TestSystemInterface::SetSampleReset34Hold(bool aHold)
  {    
    try{
      u32 lTemp=aHold?0x40:0;
      mInterface->Write(GEN_SETTINGS_2, lTemp, 0x40);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  bool CaliceMaps1TestSystemInterface::ReadSampleReset34Hold()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&0x40);
  }
  
  void CaliceMaps1TestSystemInterface::SetShaperReset34Hold(bool aHold)
  {
    try{
      u32 lTemp=aHold?0x20:0;
      mInterface->Write(GEN_SETTINGS_2, lTemp, 0x20);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  bool CaliceMaps1TestSystemInterface::ReadShaperReset34Hold()
  {  
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2,lTemp);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lTemp&0x20);
  }
    



  void CaliceMaps1TestSystemInterface::SetDebugTrim(u16 aTrim)
  {
    /*
     * Note: sensor v1.1 has two more trim bits than sensor v1.0
     * including on the alanalogue test pixels. The pads previously
     * assigned to DEBUG_RST_600 and ENABLE34 were moved over to 
     * provide the DEBUG_TRIM[4] and DEBUG_TRIM[5] respectively. 
     * there was no way to allocate these sequentially in the firmware
     * without breaking the original interface. At the time of writing
     * are a set of undiagnosed problems with v1.1, thus it's probably 
     * a good idea to maintain backwards compatibility. These two bits
     * have thus been allocated to bits 20 and 21 in GEN_SETTINGS_2. Bit 
     * 19 is now used as a select to switch between the original functions
     * of the pads and the new. (see Read/SetSensorVersion(...)).
     *
     * M.N. 20.10.08.
     *
     */
    try{
      u32 lTemp=(aTrim&0x30);
      lTemp<<=16;
      lTemp|=(aTrim&0xf);
      //std::cout<<"Trim value passed = "<<std::hex<<lTemp<<std::dec<<std::endl;
      mInterface->Write(GEN_SETTINGS_2, lTemp, (0x30000f));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadDebugTrim()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2, lTemp);
      //std::cout<<"Trim value read back = "<<std::hex<<(lTemp&0x30000f)<<std::dec<<std::endl;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    } 
    return static_cast<u16>(((0x300000&lTemp)>>16)|(0xf&lTemp));
  }



  u16 CaliceMaps1TestSystemInterface::ReadSensorId()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      mInterface->Read(ID_REG, lTemp);
      lRet=static_cast<u16>(lTemp&0x3f);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;  
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadDAQAddress()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      mInterface->Read(ID_REG, lTemp);
      lRet=static_cast<u16>((lTemp>>10)&0xf);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }  
    return lRet;
  }  

  void CaliceMaps1TestSystemInterface::SetDiscriminatorThresholds(u16 aCh0, u16 aCh1)
  {
    try{
      // mask out anything higher than bit 7:
      aCh0&=0xff; 
      aCh1&=0xff; 
      
      u32 lTemp=(aCh1<<4);
      
      // write data to DAC B and buffer:
      mInterface->Write(TRIG_UNIT_CTRL_0, lTemp, 0xffff);
      this->IssueTrigger(1<<12);
      u32 lStat=0;
      // wait for busy bit to be cleared:
      // cout<<" wait for busy bit to be cleared: "<<endl;
      u32 lPollCount=0;
      do{
	mInterface->Read(TRIG_UNIT_CTRL_0, lStat);
	lPollCount++;
	if(lPollCount>4000){
	  PollLimitException lExc(4000);
	  RAISE(lExc);
	}
      }while(0x80000000 & lStat);

      lTemp=(0x8000)|(aCh0<<4);
      // write data to DAC A and update DAC B with content of buffer.
      mInterface->Write(TRIG_UNIT_CTRL_0, lTemp, 0xffff);
      this->IssueTrigger(1<<12);
      // wait for busy bit to be cleared:
      do{
	mInterface->Read(TRIG_UNIT_CTRL_0, lStat);
      }while(0x80000000 & lStat);
      // that's it.
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadDiscriminatorThreshold(u16 aCh)
  {
    u16 lLevel=0;
    try{
      u32 lTemp=0; 
      mInterface->Read(TRIG_UNIT_CTRL_0, lTemp);
      if(aCh==0){
	lLevel=(lTemp>>8)&0xff;
      }else{
	lLevel=0xff&lTemp;
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lLevel;
  }
 
  

  void CaliceMaps1TestSystemInterface::SetupDiscriminatorReferenceDAC()
  {
    try{  
      
      mInterface->Write(TRIG_UNIT_CTRL_0, 0x9002);
      this->IssueTrigger(1<<12);
      u32 lStat=0;
      do{
	mInterface->Read(TRIG_UNIT_CTRL_0, lStat);
      }while(0x80000000 & lStat);

    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  bool CaliceMaps1TestSystemInterface::IsMaster(){
    bool lIsMaster=false;
    try{
      u16 lTemp=this->ReadDAQAddress();
      if((lTemp&0x7)==MASTER){
	lIsMaster=true;
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lIsMaster;
  }

  
  void CaliceMaps1TestSystemInterface::SetTriggerMask(unsigned short aMask)
  {
    try{
      aMask<<=7;
      aMask&=0xf80;
      mInterface->Write(GEN_SETTINGS_2, 
			static_cast<u32>(aMask),
			0x0f80);

    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 CaliceMaps1TestSystemInterface::ReadTriggerMask()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2, lTemp);
      lTemp>>=7;
      lTemp&=0x1f;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp);
  }
  
  void CaliceMaps1TestSystemInterface::SetTriggerSource(unsigned short aSource)
  {
    try{
      aSource<<=12;
      aSource&=0x3000;
      mInterface->Write(GEN_SETTINGS_2, 
			static_cast<u32>(aSource),
			0x3000);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 CaliceMaps1TestSystemInterface::ReadTriggerSource()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2, lTemp);
      lTemp>>=12;
      lTemp&=3;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>(lTemp);    
  }
  
  void CaliceMaps1TestSystemInterface::SetTriggerEnable(bool aEn)
  {
    try{
      u32 lTemp=aEn?(0x4000):(0);
      
      mInterface->Write(GEN_SETTINGS_2, 
			lTemp,
			0x4000);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  bool CaliceMaps1TestSystemInterface::ReadTriggerEnable()
  {
    u32 lTemp=0;
    try{
      mInterface->Read(GEN_SETTINGS_2,lTemp);
      lTemp&=0x4000;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return !(lTemp==0);
  }
  
  i32 CaliceMaps1TestSystemInterface::HasTriggered()
  {
    i32 lTemp=0;
    try{
      lTemp=static_cast<i32>(this->ReadStatus(0x8));
      if(lTemp==0){
	lTemp=1;
      }else{
	lTemp=0;
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }

  double CaliceMaps1TestSystemInterface::Temperature(){
    double lTemperature=0.0;
    try{
      lTemperature=
	CaliceMaps1TestSystemInterface::Temperature(this->ReadSensorCardTemperature());
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemperature;
  }

  double CaliceMaps1TestSystemInterface::Temperature(u16 aTemp){
    double lTemperature=0.0;
    // see table 2 on page 5 of the datasheet.
    // the data have been right-shifted by 2 bits
    if((aTemp&0x2000)){
      aTemp=aTemp&0xdfff;
      lTemperature=-0.0625-(static_cast<double>(8192-aTemp)/32.0);
    }else{
      lTemperature=static_cast<double>(aTemp)*(150.0/4800) ;
    }
    return lTemperature;
  }

  u16 CaliceMaps1TestSystemInterface::ReadSensorCardTemperature(){
    u16 lTemp=0;
    u32 lStatus=0;
    try{
      mInterface->Read(SPI_STATUS, lStatus);
      if(lStatus&0x800){
	// temperature chip hasn't finished yet:
	//std::cout<<"- ";
	return mLastTemp;
      }
      //std::cout<<"***** ";
      lTemp=static_cast<u16>(this->SpiRead(READ16|R_EDGE,
					   TEMPERATURE));
      lTemp&=0xfffc;
      lTemp>>=2;
      
      mLastTemp=lTemp;
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }
  
  
  void CaliceMaps1TestSystemInterface::SetReferenceLevel(u16 aReference, u16 aLevel){
    try{
      // max 5591: 12 bit DAC.
      // the write interface is 16 bits. 
      // There are 8 references per chip and 4 chips on each board:
      
      // the references will be indexed as 
      //    0->7 = CS1, 0->7
      //    8-15 = CS2, 0->7
      //   16-23 = CS3, 0->7
      //   24-31 = CS4, 0->7
      
      // hack for the debug signals:
      static u16 lLastLevel=0;

      u16 lSlave=(aReference/8)&0x3;
      u16 lDAC=(aReference&0x7);
      aLevel&=0xfff;
      
      u32 lData=(lDAC<<12)|aLevel;
      //std::cout<<"Slave = "<<lSlave<<" Channel = "<<lDAC<<"  Level = "<<aLevel<<std::endl;
      
      // this loads the input register
      this->SpiWrite(WRITE, lSlave, lData);
      // then need to shift to the DAC register:
      //u32 lTest=0;
      if(lLastLevel>aLevel){
	//std::cout<<"setting indicator enable: "<<lLastLevel<<" : "<<aLevel<<std::endl;
	mInterface->Write(SPI_CTRL, 1<<10, 1<<10);
	//mInterface->Read(SPI_STATUS, lTest);
	//std::cout<<"indicator bit set to "<<(lTest&(1<<10))<<std::endl;
      }
      this->SpiWrite(WRITE, lSlave, 0x8000|(1<<lDAC));
      //      mInterface->Read(SPI_STATUS,lTest);
      //std::cout<<"indicator bit set to "<<(lTest&(1<<10))<<std::endl; 

      mInterface->Write(SPI_CTRL, 0,1<<10);
      // update the value just written:
      lLastLevel=aLevel;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


//   void CaliceMaps1TestSystemInterface::ReadSensorWordCount(u16 * aBuf, u16 aLen){
//     u32 * lPtr=reinterpret_cast<u32 *>(aBuf);
//     try{ 
//       mInterface->Read(G_STATUS, lPtr[0]);
//       mInterface->Read(WC_REG0, lPtr[1]);
//     }catch(USBDAQException & aExc){
//       RETHROW(aExc);
//     }
//   }
  
  u16 CaliceMaps1TestSystemInterface::ReadSensorWordCount(u16 aCol)
  {
    u32 lTemp=0;
    u16 lCount=0;
    try{ 
      switch (aCol){
      case 0:
	mInterface->Read(G_STATUS, lTemp);
	lCount=0xffff&lTemp;
	break;
      case 1:
	mInterface->Read(G_STATUS, lTemp);
	lCount=0xffff&(lTemp>>16);
	break;
      case 2:
	mInterface->Read(WC_REG0, lTemp);
	lCount=0xffff&lTemp;
	break;
      case 3:
	mInterface->Read(WC_REG0, lTemp);
	lCount=0xffff&(lTemp>>16);
	break;
      default:
	break;
      }
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lCount;
  }


  u32 CaliceMaps1TestSystemInterface::ReadBufferInputCount(){
    u32 lCount=0;
    try{ 
      mInterface->Read(DATA_COUNT_1,lCount);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lCount;
  }

  u32 CaliceMaps1TestSystemInterface::ReadBufferOutputCount(){
    u32 lCount=0;
    try{ 
      mInterface->Read(DATA_COUNT_0,lCount);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lCount;
  }


  
  u32 CaliceMaps1TestSystemInterface::ReadTriggerHistoryWordCount(){
    u32 lCount=0;
    try{ 
      mInterface->Read(DATA_COUNT_2,lCount);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return (lCount&0xffff);
  }


  u32 CaliceMaps1TestSystemInterface::ReadTriggerHistory(u8 * aBuf, u32 aBufLen)
  {
    u32 lWC=0;
   
    try{
      u32 lTemp=0;
      u32 lPollCount=0;
      do{
	mInterface->Read(DATA_COUNT_2,lTemp);
	//std::cout<<__PRETTY_FUNCTION__<<" : "<<std::hex<<lTemp
	//	 <<"  "<<std::dec<<lPollCount<<std::endl;
	lWC=lTemp&0x7fff;
	lWC<<=2;
	if(++lPollCount>1000){
	  PollLimitException lExc(lPollCount);
	  RAISE(lExc);
	}
      }while(lWC<mSpillCycleCount);//(lTemp&0x8000));

      this->ClearConfigReadoutMode();
      
      //mInterface->Read(DATA_COUNT_2, lTemp);
      //this->ReadTriggerHistoryWordCount();

      //lWC=lWC<<2;
      
      if(lWC<mSpillCycleCount){
	std::cerr<<"Consistency check: word count = "
		 <<lWC<<", spill cycle = "
		 <<mSpillCycleCount<<". Temp = "<<std::hex<<lTemp<<std::endl;
	HardwareAccessException lExc("Word count error.");
	RAISE(lExc);
      }
      
      mInterface->ReadBlock(aBuf, lWC);
      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return mSpillCycleCount;
  }


  u32 CaliceMaps1TestSystemInterface::DAQ(u32 * aBuf, u32 aBufLen)
  {
    u32 lTot=0;
    static u32 lDAQCount=0;
    //u32 lHistory=0;
    try{
      // ensure that the DAQ buffer is selected:
      this->ClearConfigReadoutMode();
      
      // if(mMode&ENABLE_PMT_RECORD){
	// initially we must readout the PMT activity record:
	//lHistory=this->
      //}

      
      u16 lCount=0;
      u16 lVol=0;
      u32 lLastCount=3;// = the header length in u32s
      
      u32 lPollCount=0;
      
      for(u16 i=0; i<4; i++){
	// zero the designators:
	reinterpret_cast<u16 *>(aBuf)[i+2]=0;
      }
      
      for(u16 i=mReadoutStartIndex; i<4; i++){
	//std::cout<<"attempting quadrant "<<std::dec<<i<<" "<<this->ReadDAQWordCount()<<std::endl;
	while(!(0x8000&(lCount=this->ReadSensorWordCount(i)))){
	  //std::cout<<"awaiting done flag for column "<<i<<" Polled "<<lPollCount<<" Word Count = "<<lCount<<std::endl;
	  if(++lPollCount>2000){
	    PollLimitException lExc(lPollCount);
	    lExc<<" Column = ";
	    lExc<<i;
	    RAISE(lExc);
	  }
	}
	if(lPollCount>19)
	  std::cout<<i<<" DAQ_POLL_COUNT = "<<lPollCount<<std::endl;
	
	lPollCount=0;

	lCount&=0x7fff;
	lVol=lCount&0xfff;// the ls 12 bits are the data volume:
	lTot+=lVol;

	reinterpret_cast<u16 *>(aBuf)[i+2]=lCount;
	// readout 4 times the column word count:
	//std::cout<<"attempting daq of byte count : "<<(lCount<<2)<<std::endl;
 	//for(u16 j=0; j<lCount; j++){
	//  std::cout<<j<<std::endl;
	//mInterface->ReadBlock(reinterpret_cast<u8*>(&aBuf[12+(j*lCount)]),lCount);// lCount*sizeof(u32));
	mInterface->ReadBlock(reinterpret_cast<u8*>(&aBuf[lLastCount]),lVol*sizeof(u32));
	lLastCount+=lVol;
      }
      aBuf[0]=lTot;
    }catch(USBDAQException & aExc){
//       for(u32 k=0; k<10; k++){
// 	std::cout<<"sample "<<std::dec<<k<<" = "<<std::hex<<aBuf[k]<<std::endl;
//       }
      

      RETHROW(aExc);
    }
    this->SetConfigReadoutMode();
    lDAQCount++;
    return lTot;
  }





  void CaliceMaps1TestSystemInterface::SetupReferences()
  {
    try{ 
      for(u16 i=0; i<4;i++){
	this->SpiWrite(WRITE, i, 0x0b660);
      }
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 CaliceMaps1TestSystemInterface::ReadReferenceLevel(u16 aReference){
    u32 lRet=0;
    try{      
      u16 lSlave=(aReference/8)&0x3;
      u32 lDAC=(aReference&0x7);
      u32 lCmd=0xd0000000|(lDAC<<25)|0xffffff;
      
      //    std::cout<<std::dec<<"Slave = "<<lSlave
      // 	       <<" DAC = "<<lDAC
      // 	       <<" CMD =  "<<std::hex<<lCmd;
      
      lRet=this->SpiRead((R_EDGE|READ32),lSlave, lCmd);
      // std::cout<<" Return = "<<lRet<<std::endl;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return static_cast<u16>((lRet>>12)&0x0fff);
  }


  void CaliceMaps1TestSystemInterface::SpiWrite(u16 aMode, u16 aSlave, u32 aData){
    try{
      
      u32 lCtrlWord=(aMode<<4)|(0x8|(aSlave&0x7));
      
      mInterface->Write(SPI_CTRL,lCtrlWord,0xff);
      mInterface->Write(SPI_DATA,aData,0xffffffff);
      
      this->IssueTrigger(1<<6);
      u32 lStatus=0;
      u32 lCount=0;
      do{
	mInterface->Read(SPI_STATUS, lStatus);

	if(lCount++>3000){
	  PollLimitException lExc(lCount);
	  RAISE(lExc);
	}
      }while(lStatus&(1<<8));
     
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  u32 CaliceMaps1TestSystemInterface::SpiRead(u16 aMode, u16 aSlave, u32 aCmd){
    u32 lRet=0;
    try{
      
      u32 lCtrlWord=(aMode<<4)|(0x8|(aSlave&0x7));
      mInterface->Write(SPI_CTRL,lCtrlWord,0xff);
      mInterface->Write(SPI_DATA, aCmd, 0xffffffff);
      
      this->IssueTrigger(1<<6);
      

      u32 lStatus=0; 
      u32 lCount=0;
      do{
	//std::cout<<"poll count : "<<lCount<<std::endl;
	mInterface->Read(SPI_STATUS, lStatus);
    	if(lCount++>3000){
	  PollLimitException lExc(lCount);
	  RAISE(lExc);
	}
      }while(lStatus&(1<<8));
      // when the busy line goes low, 
      // we can read the data from the data register: 
      mInterface->Read(SPI_DATA, lRet);

    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }

  void CaliceMaps1TestSystemInterface::IssueTrigger(u32 aTrigVector,
						    u32 aDuration){
    try{
      
      mInterface->Write(TRIGGER,aTrigVector,aDuration);
      

    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void CaliceMaps1TestSystemInterface::EnableLVDSBuffers(){
    try{
      mInterface->Write(G_CONFIG, 0x10, 0x10);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  void CaliceMaps1TestSystemInterface::DisableLVDSBuffers()
  {
    try{
      mInterface->Write(G_CONFIG, 0x0, 0x10);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

}//~namespace CALICEMAPS1
