#include "TPACMasterController.hh" 

#include <iostream>
#include <iomanip>
#include <cstring>

#include "USBDAQException.hh"
#include "LinuxLibUsbInterface.hh"

#include <time.h>


namespace USBDAQ
{

  TPACMasterController::TPACMasterController():
    mInterface(0), 
    mClockLock(0), 
    mTriggerRecordLength(0)
  {
    
    try{

#ifdef Linux
      mInterface=new USBDAQ::LinuxLibUsbInterface("USBDAQ",
						  0xfeed,
						  0xbeef,
						  0);
#endif

#ifdef Win32
      // Win32 implementation goes here:
      mInterface=new USBDAQ::LinuxLibUsbInterface("USBDAQ",
						  0xfeed,
						  0xbeef,
						  0);
      
      
#endif

            
      mInterface->SetAddressSpaceSize(ADDRESS_SIZE_);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  TPACMasterController::~TPACMasterController()
  {
    try{
      delete mInterface;
      mInterface=0;
    }catch(USBDAQ::USBDAQException & aExc){
      std::cerr<<aExc.What()<<std::endl;
    }

  }
  

  u32 TPACMasterController::ReadStatus(){

    // status(0) = spill active indicator
    // status(1) = buffer overflow
    // status(2) = buffer empty
    // status(3) = jtag tdo;
    // status(4) = jtag busy
    // status(5) = spill emulator busy
    


    u32 lTemp=0;
    try{
      mInterface->Read(STATUS, lTemp);
      //std::cout<<std::hex<<"Status = 0x"<<lTemp<<std::endl;
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }


  u16 TPACMasterController::ReadTriggerInputState()
  { 
    u32 lTemp=0;
    try{
      mInterface->Read(STATUS, lTemp);
      //std::cout<<std::hex<<"Status = 0x"<<lTemp<<std::endl;
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    u16 lState=static_cast<u16>(lTemp>>16);
    return lState;
  }
   
  void TPACMasterController::BufferFlush()
  {
    try{
      // issue the reset events:
      this->Trigger(0x18, 20);
      
      // allow the sx2 to finish: 
      this->Pause(100);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }  




  bool TPACMasterController::Busy()
  {
    u32 lRet=0;
    try{
      lRet=this->ReadStatus();
      
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet&0x20;
  }



  void TPACMasterController::Trigger(u32 aVector, u32 aDuration)
  {
    try{
      mInterface->Write(TRIGGER, aVector, aDuration);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void TPACMasterController::IssueTestTrigger(u16 aIdx, 
					      u32 aLen)
  {
    try{
      switch (aIdx) {
      case 0: 
	mInterface->Write(TRIGGER, 0x2, aLen);
	break;						
      case 1: 
	mInterface->Write(TRIGGER, 0x4, aLen);
	break;
      default: 
	std::cerr<<"\n\nWarning: only software triggers\n";
	std::cerr<<"         0 and 1 are implemented.\n\n";
	break;  
      }
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  void TPACMasterController::SetTestBits(u16 aTest)
  {
    try{
      // there are 3 test bits: 
      u32 lTemp=aTest&0x7;
      lTemp<<=29;
      
      mInterface->Write(REG_RST_BXSTART, 
			lTemp, 
			0xe0000000);
      

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 TPACMasterController::ReadTestBits()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      
      mInterface->Read(REG_RST_BXSTART, 
		       lTemp);
      
      lRet=static_cast<u16>((lTemp>>29)&0x7);
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
 

  void TPACMasterController::StartSpill()
  {
    try{      
      this->Trigger(0x1, 5);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  void TPACMasterController::SetResetLength(u16 aResetLength)
  {
    try{
      mInterface->Write(REG_RST_BXSTART, 
			static_cast<u32>(aResetLength), 
			0x0000ffff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  u16 TPACMasterController::ReadResetLength()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0; 
      mInterface->Read(REG_RST_BXSTART, 
		       lTemp);
      lRet=static_cast<u16>(lTemp&0xffff);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
  
  void TPACMasterController::SetSpillCycleStart(u16 aSCStart)
  {
    try{ 
      // 13 bottom bits only: 
      u32 lTemp=aSCStart&0x1fff;
      lTemp<<=16;
      u32 lMask=0x1fff<<16; 
      mInterface->Write(REG_RST_BXSTART, 
			lTemp,
			lMask);
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 TPACMasterController::ReadSpillCycleStart()
  {
    u16 lRet=0;
    try{ 
      u32 lTemp=0;
      mInterface->Read(REG_RST_BXSTART, 
		       lTemp); 
      lRet=static_cast<u16>((lTemp>>16)&0x1fff);
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet; 
  }

  void TPACMasterController::SetBunchCrossingLength(u16 aLength)
  {
    try{ 
      u32 lTemp=aLength&0xff;
      mInterface->Write(REG_BT_BX_LENGTH, 
			lTemp, 
			0xff);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
   
  u16 TPACMasterController::ReadBunchCrossingLength()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      mInterface->Read(REG_BT_BX_LENGTH, 
		       lTemp);
	
      lRet=lTemp&0xff;
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet; 
  }

  void TPACMasterController::SetBunchTrainLength(u16 aLength)
  {
    if(aLength>8191){
      // aLength is out of range. 
      // valid range: [0,8192)
      // 
    }
    
    try{
      mTriggerRecordLength=aLength&0x1fff;
      
      mTriggerRecordLength++;

      u32 lTemp=aLength&0x1fff;
      lTemp<<=8;
      mInterface->Write(REG_BT_BX_LENGTH,
			lTemp, 
			0x001fff00);
      
      
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  
  u16 TPACMasterController::ReadBunchTrainLength()
  {
    u16 lRet=0;
    try{
      u32 lTemp=0;
      mInterface->Read(REG_BT_BX_LENGTH,
		       lTemp);
      lRet=static_cast<u16>((lTemp>>8)&0x1fff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    
    return lRet; 
  }

  
  void TPACMasterController::SetTriggerEnableMask(u16 aMask)
  {
    try{ 
      u32 lTemp=aMask;
      mInterface->Write(REG_INVERT_ENABLE, 
			lTemp, 
			0x0000ffff);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u16 TPACMasterController::ReadTriggerEnableMask()
  {
    u16 lRet=0;
    try{ 
      u32 lTemp=0;
      mInterface->Read(REG_INVERT_ENABLE, 
		       lTemp);
      lRet=static_cast<u16>(lTemp&0xffff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
    
  void TPACMasterController::SetTriggerInversionMask(u16 aMask)
  {
    try{ 
      u32 lTemp=aMask;
      lTemp<<=16;
      mInterface->Write(REG_INVERT_ENABLE, 
			lTemp, 
			0xffff0000);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u16 TPACMasterController::ReadTriggerInversionMask()
  {
    u16 lRet=0;
    try{ 
      u32 lTemp=0;
      mInterface->Read(REG_INVERT_ENABLE, lTemp);
      lRet=static_cast<u16>((lTemp>>16)&0xffff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
  
  
  void TPACMasterController::SetTestTriggerStart(u32 aStart)
  {
    try{ 
      mInterface->Write(REG_START_TT, 
			aStart, 
			0xffffffff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  u32 TPACMasterController::ReadTestTriggerStart()
  {
    u32 lRet=0;
    try{ 
      mInterface->Read(REG_START_TT, 
		       lRet);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
  
  void TPACMasterController::SetTestTriggerEnd(u32 aEnd)
  {
    try{ 
      mInterface->Write(REG_END_TT,
			aEnd,
			0xffffffff);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  u32 TPACMasterController::ReadTestTriggerEnd()
  {
    u32 lRet=0;
    try{ 
      mInterface->Read(REG_END_TT, 
		       lRet);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }
  
  
  u32 TPACMasterController::ReadFirmwareBuildTime()
  {
    u32 lRet=0;
    try{ 
      mInterface->Read(BUILD_TIME,
		       lRet);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet; 
  }

  u32 TPACMasterController::ReadFirmwareVersion()
  {
    u32 lRet=0;
    try{ 
      mInterface->Read(BUILD_VERSION, 
		       lRet);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lRet;
  }

  
  u32 TPACMasterController::ReadTluBuffer()
  {
    u32 lRet=0;
#ifndef TPACMASTERCONTROLLER_FIRMWARE_V6
    try{ 
      mInterface->Read(TLU_BUFFER,
		       lRet);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
#endif
    return lRet;
  }

  void TPACMasterController::PopTluBuffer()
  {
#ifndef TPACMASTERCONTROLLER_FIRMWARE_V6
    try{      
      this->Trigger(0x20,1);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
#endif
  }

  u16 TPACMasterController::ReadTriggerRecordBuffer(u16 * aBuf, 
						    u32 aBufLen)
  {
    
    try{
      if(aBufLen<mTriggerRecordLength){
	// not enough space in buffer: 
	// raise an exception...
	
      }

      mInterface->ReadBlock(reinterpret_cast<u8 *>(aBuf),
			    2*mTriggerRecordLength);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return mTriggerRecordLength;
  }  
  

  void TPACMasterController::Pause(u32 ams)
  {
    struct timespec lReq;
    struct timespec lRem;
    
    lReq.tv_sec=ams/1000;
    lReq.tv_nsec=(ams%1000)*1000000;
    nanosleep(&lReq, &lRem);
  }



}//~namespace T2KDAQ


