#include "TestInterface.hh" 

#include <iostream>
#include <iomanip>
#include <cstring>

#include "USBDAQException.hh"
#include "LinuxLibUsbInterface.hh"

#include <time.h>


namespace USBDAQ
{

  TestInterface::TestInterface():
    mInterface(0)
  {
    
    try{

#ifdef Linux
      mInterface=new USBDAQ::LinuxLibUsbInterface("USBDAQ",
						  0xfeed,
						  0xbeef,
						  0);
#endif

#ifdef Win32
      // Win32 implementation goes here:
      mInterface=new USBDAQ::LinuxLibUsbInterface("USBDAQ",
						  0xfeed,
						  0xbeef,
						  0);
      
      
#endif

            
      mInterface->SetAddressSpaceSize(ADDRESS_SIZE_);

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  TestInterface::~TestInterface()
  {
    try{
    
      delete mInterface;
      mInterface=0;
    }catch(USBDAQ::USBDAQException & aExc){
      std::cerr<<aExc.What()<<std::endl;
    }

  }
  
  
  void TestInterface::TestRegister(u32 aReg, u32 aVal){
    try{
      mInterface->Write(REG0+aReg, aVal);
     }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  

  u32 TestInterface::TestRegister(u32 aReg){
    u32 lTemp=0;
    try{
      mInterface->Read(REG0+aReg, lTemp);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }
    
  u32 TestInterface::ReadStatus(){
    u32 lTemp=0;
    try{
      mInterface->Read(STATUS, lTemp);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
    return lTemp;
  }

  void TestInterface::Trigger(u32 aVector, u32 aDuration)
  {
    try{
      mInterface->Write(TRIGGER, aVector, aDuration);
    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  
  void TestInterface::ManipulateBits()
  {
    try{
      mInterface->Write(REG0, 0);
      u32 lTemp=0;

      std::cout.fill('0');
      for(u16 i(0); i<32; i++){
	mInterface->SetBit(REG0, i);
	mInterface->Read(REG0, lTemp);
	std::cout<<std::setw(2)<<std::dec<<i<<" : "<<std::setw(8)
		 <<std::hex<<lTemp<<std::endl;
	mInterface->ClearBit(REG0, i);
      }

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  
  void TestInterface::FlashLeds()
  {
    try{
      std::cout<<"turning on LED 0..."<<std::endl;
      mInterface->SetBit(REG0, 0);
      Pause(1000);
      std::cout<<"turning off LED 0"<<std::endl;
      mInterface->ClearBit(REG0, 0);
      
      
      std::cout<<"Strobing LED 0"<<std::endl;
      this->Trigger(1, 120000000);
      
      

    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void TestInterface::BulkTest(){
    try{
      const u32 lLen=1000;
      u32 lBufIn[lLen];
      u32 lBufOut[lLen];
      //std::memset(lBufIn, 0, sizeof(u32)*lLen);
      //std::memset(lBufOut, 1 sizeof(u32)*lLen);
      for(u32 j(0); j<10; j++){
	for(u32 i(0);i<lLen;i++){
	  lBufIn[i]=std::rand();
	}
	
	mInterface->WriteBlock(reinterpret_cast<u8*>(lBufIn),
			       lLen*sizeof(u32));
	mInterface->ReadBlock(reinterpret_cast<u8*>(lBufOut), 
			      lLen*sizeof(u32));
	
	if(std::memcmp(lBufIn, lBufOut, lLen*sizeof(u32))){
	  std::cout<<j<<" Buffers Do Not Match"<<std::endl;
	}else{
	  std::cout<<j<<" Buffers Match"<<std::endl;
	}
      }


    }catch(USBDAQ::USBDAQException & aExc){
      RETHROW(aExc);
    }
  
  }


  void TestInterface::Pause(u32 ams)
  {
    struct timespec lReq;
    struct timespec lRem;
    
    lReq.tv_sec=ams/1000;
    lReq.tv_nsec=(ams%1000)*1000000;
    nanosleep(&lReq, &lRem);
  }



}//~namespace T2KDAQ


