#include "TypeDefs.hh"
#include "UsbDevice.hh"

#include "SoftFifo.hh"

#include <string>
#include <vector>
#include <map>

#include <iostream>

extern "C" {
#include "usb.h"
}

namespace USBDAQ
{
  /*
   * Forward declarations.
   *
   */

  class UsbEndPoint;

  
  /*
   * Implementation of the UsbDevice ABC.
   *
   * M. Noy. (matthew.noy@imperial.ac.uk)
   * 15.03.05
   */
  
  class LibUsbDevice:public UsbDevice
  {
  public:
    LibUsbDevice(const std::string & aDeviceId);
    
    LibUsbDevice(const u16 aIdVendor, const u16 aIdProduct, const u16 aBcdDevice);

    virtual ~LibUsbDevice();
    
    
    
    /*
     * Reads 32 bits from the specified endpoint.
     *
     */
    virtual void Read(const UsbEndPoint & aEp, u32 & aData);


    /*
     * Writes 32 bits to the specified endpoint.
     *
     */   
    virtual void Write(const UsbEndPoint & aEp, u32 aData);


    /*
     * Reads a block of data of length aLength from the 
     * specified endpoint and places it in the buffer
     * aBuf. Caller has responsibility for aBuf.
     *
     */    
    virtual u32 ReadBlock(const UsbEndPoint & aEp, u8 * aBuf, 
			  u32 aLength);
  
    
    u32 ReadBlock2(u8 * aBuf, 
		   u32 aLength);
    
    
    /*
     * Writes aLength bytes from aBuf to the specified 
     * endpoint. Caller has responsibility for aBuf. 
     *
     */   
    virtual void WriteBlock(const UsbEndPoint & aEp, const u8 * aBuf, 
			    u32 aLength);
    


    
  virtual void BusReset();
    

    /*
     * Class private interface begins here,
     * where the specifics will be performed.
     *
     */
  private:
    /*
     * Initialises the usb interface, finding the 
     * devices with the matching id (calling Locate())
     * and requesting one of them (Request()). 
     *
     */   
    void Init(int aRoute=0);
    
    /*
     * Finds all devices on the system with the device id
     * passed to the constructor.
     *
     */   
    void LocateID();
    void LocateNumbers();

    /*
     * Iterates over all the devices found by Locate()
     * and requests each one in turn. A given device can 
     * only be claimed once, and any subsequent attempts will fail.
     * so the result is that the first unclaimed
     * interface is claimed.  
     *
     */   
    void Request();
    /*
     * Releases the interface held by the 
     * instance. 
     *
     */   

    
    void Reset();


    void Release();

  private:
    //
    // private collection for the 
    // important parameters when dealing 
    // with multiple devices.  
    //
    class DeviceParams{
    public:
      DeviceParams(struct usb_bus * aUsbBus,
		   struct usb_dev_handle * aUsbDeviceHandle,
		   i32 aInterfaceNumber);
      
      void Print(std::ostream & aOs)const;
				     

      struct usb_bus * mUsbBus;
      struct usb_dev_handle * mUsbDeviceHandle;
      i32 mInterfaceNumber;
    };
    
    
    static bool & IsFirstGo();




  private:
    //
    // pointer the structure containing 
    // information about the bus upon 
    // which the device is found. 
    //
    struct usb_bus * mUsbBus;

    // 
    // pointer to the device identifier
    // of the device. 
    //
    struct usb_dev_handle * mUsbDeviceHandle;
    
    //
    // holds the interface number required for
    // requesting and releasing the interface.
    // this is cached because it's 
    // particularly awkward to obtain.
    //
    i32 mInterfaceNumber;
    
    //
    // flag to indicate whether the device was
    // found in the search,
    //
    bool mDeviceFound;
    
    //
    // flag indicating that the current process 
    // has successfully requested control of the 
    // device.
    //
    bool mObtained;
    
    //
    // transaction timout.
    //
    i32 mTimeout;
    
    std::map <int, SoftFifo> mFifos;
    
    u8 mTempBuf[512];
    
    // container for all 
    std::vector<DeviceParams *> mDevList;

  };//~class LibUsbDevice

  
 
}//~namespace USBDAQ
