#ifndef UsbDevice_HH
#define UsbDevice_HH

#include <string>

#include "TypeDefs.hh"



namespace USBDAQ
{
  /*
   * Forward declaration of the 
   * end point specification class.
   *
   */
  class UsbEndPoint;

  
  /*
   * Abstract Base Class defining the interface to the 
   * USB bus hardware access libraries. 
   * 
   * Should be implemented for each library used to allow
   * migration between different USB libraries.
   * 
   * M. Noy. (matthew.noy@imperial.ac.uk)
   * 13.03.05
   *
   */
  
  class UsbDevice
  {
  public:
    
    /*
     * Constructor that initialises the ID
     * member.
     *
     */
    UsbDevice(const std::string & aDeviceId):
      mDeviceId(aDeviceId){}

   
    UsbDevice(const u16 aIdVendor,const u16 aIdProduct,const u16 aBcdDevice):
      mIdVendor(aIdVendor),mIdProduct(aIdProduct),mBcdDevice(aBcdDevice) 
    {}


    /*
     * virtual destructor.
     *
     */
    virtual ~UsbDevice(){};

    

    /*
     * Reads 32 bits from the specified device and end-point. 
     * Data should be present in aData when function returns.
     * 
     */
    virtual void Read(const UsbEndPoint & aEp, u32 & aData)=0;

    /*
     * Writes 32 bits to the specified device and end-point.
     *
     */
    virtual void Write(const UsbEndPoint & aEp, u32 aData)=0;

    /*
     * Reads aLength number of bytes from the specified device 
     * and end-point, and place in the memory pointed to by aBuf. 
     * There must be enough space allowed for the transaction.
     *
     */
    virtual u32 ReadBlock(const UsbEndPoint & aEp, u8 * aBuf, 
			   u32 aLength)=0;
    
    /*
     * Writes aLength number of bytes to the specified device 
     * and end-point of the data pointed to  by aBuf. 
     * There must be a mimimum of aLength bytes pointed to by aBuf.
     *
     */
    virtual void WriteBlock(const UsbEndPoint & aEp, const u8 * aBuf, 
			    u32 aLength)=0;
    
    
    /*
     * Internal: reset the bus
     *
     */
    virtual void BusReset()=0;


  private:
    /*
     * Default constructor: private such that no implementations
     * may be instantiated without the Device ID required to 
     * identify the device on the usb bus. 
     *
     */
    UsbDevice();
    
    /*
     * copy constructor: private such that copies may not be 
     * made. a process should have a unique access point to the 
     * hardware. 
     *
     */
    UsbDevice(const UsbDevice &);

    /*
     * assignment: no UsbDevice should be assignable or assigned.
     * essentially, it is not a l-value. 
     *
     */
    UsbDevice & operator=(const UsbDevice &);
    
  protected:

    /*
     * Identifies the name of the device 
     * to be found on the system.
     *
     */
    std::string mDeviceId;
    u16 mIdVendor;
    u16 mIdProduct;
    u16 mBcdDevice;
    
  };//~class UsbDevice
  

}//~namespace USBDAQ

#endif //~UsbDevice_HH
