#include "SoftFifo.hh" 

#include<iostream>

#include <cstring>

namespace USBDAQ
{

  SoftFifo::SoftFifo(int aBufDepth):
    mDepth(aBufDepth),
    mIndexMask(1),
    mR(0), 
    mW(0),
    mBuf(0)
  {
    if((mDepth&(mDepth-1))){
      std::cerr<<__PRETTY_FUNCTION__<<" : Fifo buffer depth must be a power of 2"<<std::endl;
    }else{
      int lMask=1;
      do{
	mIndexMask|=lMask;
	lMask<<=1;
      }while(!(lMask&mDepth));

      //std::cout<<" mDepth  = "<<mDepth<<" index mask = "<<mIndexMask<<std::endl;
      
      if(mDepth>0){
	mBuf=new u8 [mDepth];
	if(mBuf==0){
	  std::cerr<<__PRETTY_FUNCTION__<<" : Error allocating memory"<<std::endl;
	}
      }
    }
  }

  SoftFifo::SoftFifo(const SoftFifo & aFifo):
    mDepth(aFifo.mDepth),
    mIndexMask(aFifo.mIndexMask),
    mR(aFifo.mR), 
    mW(aFifo.mW),
    mBuf(0)
  {
    if(mDepth>0){
      mBuf=new u8 [mDepth];
      if(mBuf==0){
	std::cerr<<__PRETTY_FUNCTION__<<" : Error allocating memory"<<std::endl;
      }
      std::memcpy(mBuf, aFifo.mBuf, mDepth);
    }
  }
  

  SoftFifo & SoftFifo::operator=(const SoftFifo & aRhs){
    if(this!=(&aRhs)){
      if(mDepth!=aRhs.mDepth){
	delete [] mBuf;
	mBuf=new u8 [aRhs.mDepth];
      }
      mIndexMask=aRhs.mIndexMask;
      mR=aRhs.mR;
      mW=aRhs.mW;
      std::memcpy(mBuf, aRhs.mBuf, mDepth);
    }
    return (*this);
  }
  
  SoftFifo::~SoftFifo()
  {
    delete [] mBuf;
    mDepth=0;
    mR=0;
    mW=0;
  }


  void SoftFifo::Reset(){
    mR=0;
    mW=0;
  }
  
  bool SoftFifo::Empty(){
    return (mR==mW);
  }
  
  bool SoftFifo::Full(){
    //  std::cout<<"R = "<<mR<<" "<<(mR&mIndexMask)
// 	     <<" W = "<<mW<<" "<<(mW&mIndexMask)<<" c "<<(mR&mDepth)<<" d "
// 	     <<(mW&mDepth)
// 	     <<" a "<<((mR&mIndexMask)==(mW&mIndexMask))
// 	     <<" b "<<((mR&mDepth)!=(mW&mDepth))<<std::endl;
    return ((mR&mIndexMask)==(mW&mIndexMask)) && ((mR&mDepth)!=(mW&mDepth));
  }
  
  int SoftFifo::Content(){
    int lCont=mW-mR;
    return lCont;
  }
 

  int SoftFifo::Extract(u8 * aDest, int aLength){
    int lXfer=Content();
    if(lXfer>aLength){
      lXfer=aLength;
    }
    for(int i=0; i<lXfer; i++){
      aDest[i]=mBuf[mR&mIndexMask];
      mR++;
    }
    return lXfer;
  }

  int SoftFifo::Insert(const u8 * aSrc, int aLength){
   //  if(aLength==0){
//       return 0;
//     }
    int lSpace=mDepth-Content();
    int lXfer=0;
    if(lSpace<aLength){
      lXfer=lSpace;
    }else{
      lXfer=aLength;
    }
    //std::cout<<"inserting "<<lXfer<<" bytes"<<std::endl;
    for(int i=0; i<lXfer; i++){
      mBuf[mW&mIndexMask]=aSrc[i];
      mW++;
    }    
    return lXfer;
  }
  
  
  


}//~namespace USBDAQ


