#include "LibUsbDevice.hh"
#include "UsbEndPoint.hh" 

#include "USBDAQException.hh"
#include "HardwareAccessException.hh"
#include "PollLimitException.hh"

#include <iostream>


extern "C" {
#include "error.h"
}

namespace USBDAQ
{    
  
  LibUsbDevice::LibUsbDevice(const std::string & aDeviceId):
    UsbDevice(aDeviceId), 
    mUsbBus(0),
    mUsbDeviceHandle(0),
    mInterfaceNumber(-1),
    mDeviceFound(false),
    mObtained(false),
    mTimeout(10000)
  {
    //
    // call initialisation:
    //
    try{
      this->Init();

      
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
    
  }


  LibUsbDevice::LibUsbDevice(const u16 aIdVendor, 
			     const u16 aIdProduct, 
			     const u16 aBcdDevice):
    UsbDevice(aIdVendor,aIdProduct,aBcdDevice),
    mUsbBus(0),
    mUsbDeviceHandle(0),
    mInterfaceNumber(-1),
    mDeviceFound(false),
    mObtained(false),
    mTimeout(10000)
  {
    //
    // call initialisation:
    //
    try{
      //std::cout<<__PRETTY_FUNCTION__<<std::endl;
      
      // ensure that the correct location 
      // code is found.
      this->Init(1);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  
  LibUsbDevice::~LibUsbDevice()
  {// release the interface.
    
    try{
      if(mDeviceFound)
	this->Release();

      
    }catch(USBDAQException & aExc){
      std::cerr<<aExc.What()<<std::endl;
      std::cerr<<aExc.History()<<std::endl;
    }catch(std::exception & aExc){
      std::cerr<<aExc.what()<<std::endl;
    }catch(...){
      std::cerr<<"Unknown exception caught in function "
	       <<__PRETTY_FUNCTION__<<std::endl;
    }
        
  }


  bool & LibUsbDevice::IsFirstGo(){
    static bool lIsFirstGo=true;
    return lIsFirstGo;
  }


  /*
   * This function initialises the usblib library then looks 
   * for the device id of the IDAQ. 
   *  
   * if found, it requests control of the IDAQ. 
   * 
   */
  void LibUsbDevice::Init(int aRoute)
  {
    if(IsFirstGo())
      usb_init();

    try{
      //
      // find the device.
      //
      if(aRoute==0)
	this->LocateID();
      else
 	this->LocateNumbers();
      

      //
      // request control of a device interface.
      //
      if(mDeviceFound){
	this->Request();
	

	// Don't know why but if you don't reset the device
	// here it can lock up on first read second time you run
	// an application - something to do with a cache not being cleared
	usb_reset(mUsbDeviceHandle);
      
      }

      IsFirstGo()=false;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  
  
  void LibUsbDevice::LocateID()
  {
    i32 lRet=usb_find_busses();
    if(lRet<0){
      HardwareAccessException lExc("usb_find_busses returned ");
      lExc<<lRet;
      lExc<<". Do you have any USB busses on your system?";
      RAISE(lExc);
    }
    lRet=usb_find_devices();
    if(lRet<0){
      HardwareAccessException lExc("usb_find_devices returned ");
      lExc<<lRet;
      lExc<<". Do you have any USB busses/devices on your system?";
      RAISE(lExc);
    }
    u16 lNoBusses=0;
    bool lDevFound=false;
    
    for(usb_bus * lBus=usb_get_busses();
	(lBus && !lDevFound); 
	lBus=lBus->next,++lNoBusses) 
      {
	unsigned short lNoDevices=0;
	// now loop over all the devices and place the
	// found identifiers in the device list container:
	for(struct usb_device *lDev=lBus->devices;lDev;lDev=lDev->next,++lNoDevices) 
	  {
	    //std::cout<<__PRETTY_FUNCTION__<<": Device number : "<<lNoDevices<<std::endl;
	    struct usb_dev_handle * lUsbDev=0;
	    lUsbDev = usb_open(lDev);
	    char lStrBuf[256]={0};
	    if(lUsbDev!=0)
	      {
		if(lDev->descriptor.iManufacturer) 
		  {
		    int lRet=usb_get_string_simple(lUsbDev,
						   lDev->descriptor.iProduct,
						   lStrBuf,sizeof(lStrBuf));

		    //std::cout<<"Testing string "<<lStrBuf<<std::endl;

		    if(lRet>0)
		      {
			std::string lTempProduct(lStrBuf);
			
			if(lTempProduct==mDeviceId)
			  {
			    DeviceParams * lDevParm=new DeviceParams(lBus, 
								     lUsbDev, 
								     lDev->config->interface->altsetting->bInterfaceNumber);
			    
			    //std::cout<<"lUsbDev = "<<lUsbDev<<std::endl;
			    // at least one device found.
			    lDevFound=true;
			    mDevList.push_back(lDevParm);
			    
			  }			
		      }//~if(lRet>0)
		  }
		//		usb_close(udev);
	      }//~if(lUsbDev)
	  }//~for()
      }//~for()
    mDeviceFound=lDevFound;
    if(!mDeviceFound)
      {
	HardwareAccessException lExc("Unable to find any devices with ID: \"");
	lExc<<mDeviceId;
	lExc<<"\"";	
	lExc<<"\nAre any plugged in, powered and correctly enumerated?";
	RAISE(lExc);
      }
  }
  

  
  
 void LibUsbDevice::LocateNumbers()
  {
    i32 lRet=usb_find_busses();
    if(lRet<0){
      HardwareAccessException lExc("usb_find_busses returned ");
      lExc<<lRet;
      lExc<<". Do you have any USB busses on your system?";
      RAISE(lExc);
    }
    lRet=usb_find_devices();
    if(lRet<0){
      HardwareAccessException lExc("usb_find_devices returned ");
      lExc<<lRet;
      lExc<<". Do you have any USB busses/devices on your system?";
      RAISE(lExc);
    }
    u16 lNoBusses=0;
    bool lDevFound=false;
    
    for(usb_bus * lBus=usb_get_busses();
	(lBus);// && !lDevFound); 
	lBus=lBus->next,++lNoBusses)
      {
	unsigned short lNoDevices=0;
	// now loop over all the devices and place the
	// found identifiers in the device list container:
	for(struct usb_device *lDev=lBus->devices;lDev;lDev=lDev->next,++lNoDevices) 
	  {
	    //std::cout<<__PRETTY_FUNCTION__<<": Device number : "<<lNoDevices<<std::endl;
	    struct usb_dev_handle * lUsbDev=0;
	    lUsbDev = usb_open(lDev);
	    char lStrBuf[256]={0};
	    if(lUsbDev!=0)
	      {
		bool lRightType=false;
		//std::cout << lUsbDev << std::endl;
		if(lDev->descriptor.iManufacturer) 
		  {
		    //std::cout<<lDev->descriptor.iManufacturer << std::endl;
		    
		    //int lRet=
		    usb_get_string_simple(lUsbDev,
					  lDev->descriptor.iProduct,
					  lStrBuf,sizeof(lStrBuf));
		    
// 		    std::cout<<"bus number = "<<lNoBusses<<" : "
// 			     <<"device number = "<<lNoDevices<<" : "
// 			     <<lDev->descriptor.idVendor <<" : "
// 			     <<lDev->descriptor.idProduct <<" : "
// 			     <<lDev->descriptor.bcdDevice <<" \n";
		    
		    //		    if(lRet>0)
		    // {
		    //	std::string lTempProduct(lStrBuf);
		    //std::cout<<"Testing string "<<lStrBuf<<std::endl;
			
		    if(lDev->descriptor.idVendor==mIdVendor
		       && lDev->descriptor.idProduct==mIdProduct
		       && lDev->descriptor.bcdDevice==mBcdDevice)
			  {

			    //std::cout << "device found matching ids" << std::endl;

			    DeviceParams * lDevParm=new DeviceParams(lBus, 
								     lUsbDev, 
								     lDev->config->interface->altsetting->bInterfaceNumber);
			    
			    //std::cout<<"lUsbDev = "<<lUsbDev<<std::endl;
			    // at least one device found.
			    lDevFound=true;
			    mDevList.push_back(lDevParm);
			    
			    lRightType=true;
		      }//~if(lRet>0)
		  }
		if(!lRightType)
		  usb_close(lUsbDev);

	      }//~if(lUsbDev)
	  }//~for()
      }//~for()
    mDeviceFound=lDevFound;
    if(!mDeviceFound)
      {
	std::ostringstream lOss;
	lOss<<std::hex<<"0x"<<mIdVendor<<"/0x"<<mIdProduct<<"/0x"<<mBcdDevice;


	HardwareAccessException lExc("Unable to find any devices with IDs: \"");
	lExc<<lOss.str();
	lExc<<"\"";
	lExc<<"\nAre any plugged in, powered and correctly enumerated?";
	RAISE(lExc);
      }
  }
  

  void LibUsbDevice::Request()
  {
    mObtained=false;
    i32 lRet=0;
    
    // debug
//     if(mDevList.size()==1)
//       std::cout<<"there was "<<mDevList.size()<<" device found"<<std::endl;      
//     else
//       std::cout<<"there were "<<mDevList.size()<<" devices found"<<std::endl;
    
    // loop over the filled device list container and
    // try to claim the interface. 
    // accept the first one that returns success, and 
    // exit the loop.
    // this means that the first free correctly enumerated 
    // interface will always be selected. 
    u16 i=0;
    for(i=0;i<mDevList.size();i++){
      
      
      lRet=usb_claim_interface(mDevList[i]->mUsbDeviceHandle, 
			       mDevList[i]->mInterfaceNumber);
      
      //std::cout<<"usb_claim_interface() returns "<<lRet<<std::endl;
      //mDevList[i]->Print(std::cout);
      // int retval;
      //       char dname[128] = {0};
      //       retval = usb_get_driver_np(mDevList[i]->mUsbDeviceHandle, 0, dname, 127);
      
      //       std::cout<<"get driver returns "<<retval<<" and driver name = "
      //                <<dname<<"\nerror str = "<<usb_strerror()<<std::endl;
      
      
      //lRet=-1;
      //if (!retval)
      //	usb_detach_kernel_driver_np(d, 0);
      
      if(lRet==0){
	// interface successfully claimed,
	// copy the relevant identifiers to the 
	// original data members.
	
	mObtained=true;
	mUsbBus=mDevList[i]->mUsbBus;
	mUsbDeviceHandle=mDevList[i]->mUsbDeviceHandle;
	mInterfaceNumber=mDevList[i]->mInterfaceNumber;
	break;
      }
      
      
    }
    
    // then free up the device list contents: 
    for(u16 j=0; j<mDevList.size();j++){
      
      if(i!=j){
	// closes all those we don't want here:
	usb_close(mDevList[j]->mUsbDeviceHandle);
      }

      delete mDevList[j];
      mDevList[j]=0;
    }
    // and empty the vector:
    mDevList.clear();

    // no unclaimed interfaces
    if(mObtained==false)
      {
	
	
	HardwareAccessException lExc("Unable to claim interface of device with ID ");
	lExc<<mDeviceId;
	lExc<<"\nVendor ID "<<mIdVendor;
	lExc<<"\nProduct ID "<<mIdProduct;
	lExc<<"\nAll interfaces devices may already be claimed.";
	lExc<<"\nError code : ";
	lExc<<lRet;
	lExc<<", error string: ";
	lExc<<usb_strerror();
	RAISE(lExc);
      }
    
    //std::cout<<"mObtained = "<<mObtained<<std::endl;
  }
  

  

  void LibUsbDevice::Release()
  {
    //std::cout<<__PRETTY_FUNCTION__<<std::endl;
    
    mObtained=false;
    
    i32 lRet=usb_release_interface(mUsbDeviceHandle, mInterfaceNumber);
    if(lRet!=0)
      {	
	HardwareAccessException lExc("Unable to release interface, error code: ");
	lExc<<lRet;
	RAISE(lExc);
      }
    
    lRet=usb_close(mUsbDeviceHandle);
    if(lRet!=0)
      {
	HardwareAccessException lExc("Unable to close device handle, error code: ");
	lExc<<lRet;
	RAISE(lExc);
      }
    mUsbDeviceHandle=0;
  }

  
  
  /*
   *
   *
   */
  void LibUsbDevice::Read(const UsbEndPoint & aEp, u32 & aData)
  {
    if(!mObtained){
      HardwareAccessException lExc("Unable to read from unobtained interface.");
      RAISE(lExc);
    }
    if(!aEp.IsReadable()){
      HardwareAccessException lExc("Unable to read from end point ");
      lExc<<aEp.GetId();
      lExc<<" because it is not readable.";
      RAISE(lExc);
    }
    try{
      this->ReadBlock(aEp, reinterpret_cast<u8 *>(&aData), sizeof(u32));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  /*
   *
   *
   */   
  void LibUsbDevice::Write(const UsbEndPoint & aEp, u32 aData)
  {
    if(!mObtained){
      HardwareAccessException lExc("Unable to write to unobtained interface.");
      RAISE(lExc);
    }
    if(!aEp.IsWritable()){
      HardwareAccessException lExc("Unable to write to end point ");
      lExc<<aEp.GetId();
      lExc<<" because it is not writable.";
      RAISE(lExc);
    }

    try{
      this->WriteBlock(aEp, reinterpret_cast<const u8 *>(&aData), sizeof(aData));
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }


  u32 LibUsbDevice::ReadBlock2(u8 * aBuf, 
			       u32 aLength)
  {
    int lTemp=usb_bulk_read(mUsbDeviceHandle,
			    8,
			    reinterpret_cast<char *>(aBuf),
			    aLength,
			    mTimeout);
    if(lTemp>0){
      return static_cast<u32>(lTemp);
    }
    return 0;    
  }
  

  /*
   *
   *
   */

  u32 LibUsbDevice::ReadBlock(const UsbEndPoint & aEp, u8 * aBuf, 
			      u32 aLength)
  {
    
    if(!mObtained){
      HardwareAccessException lExc("Unable to read from unobtained interface.");
      RAISE(lExc);
    }
    if(!aEp.IsReadable()){
      HardwareAccessException lExc("Unable to read from end point ");
      lExc<<aEp.GetId();
      lExc<<" because it is not readable.";
      RAISE(lExc);
    }
    
    // number of bytes read back from device/ep:
    u32 lRead=0;
    u32 lPollCount=0;

    
    //std::cout<<"rb: "<<"Reading "<<aLength<<" Bytes"<<std::endl;
    if(mFifos.find(aEp.GetId())==mFifos.end()){
      //std::cout<<"rb: "<<"initialising fifo: EP "<<aEp.GetId()<<std::endl;      
      // initialise if needed:
      SoftFifo lTemp(512);
      mFifos[aEp.GetId()]=lTemp;
    }
    SoftFifo & lFifo=mFifos[aEp.GetId()];
    
    i32 lContent=0;
    if(0<(lContent=lFifo.Content())){
      
      i32 lXfer=(lContent>static_cast<i32>(aLength))?aLength:lContent;
      lRead=lXfer;
      //std::cout<<"rb: "<<"extracting "<<lXfer<<" from the fifo which contains "<<lContent<<std::endl;
      lFifo.Extract(aBuf, lXfer);
    }
    //std::cout<<"rb: "<<"fifo contents = "<<lContent<<std::endl;

    if(lRead==aLength){
      //      std::cout<<"returning directly"<<std::endl;
      return lRead;
    }
    
    i32 lRet=0;
    while(lRead<aLength){
      // acquire the n*512 byte block
      //std::cout<<"rb: "<<"bulk xfer = "<<lRead<<std::endl;
      
      lRet=usb_bulk_read(mUsbDeviceHandle,
			 aEp.GetId(),
			 reinterpret_cast<char *>(mTempBuf),
			 512,
			 mTimeout);
      
      //std::cout<<"usb_bulk_read() returns "<<lRet<<'\n';
      if(lRet<0){
 	HardwareAccessException lExc("Bulk Read failed with error code: ");
 	lExc<<lRet;
 	lExc<<"\n error string: ";
 	lExc<<usb_strerror();
 	RAISE(lExc);
      }
      if(lRet==0){
	if(++lPollCount>1000){
	  PollLimitException lExc(lPollCount);
	  RAISE(lExc);
	}
	continue;
      }
      
      
      if((lRet+lRead)<aLength){
	std::memcpy(reinterpret_cast<void *>(aBuf+lRead),
		    mTempBuf, 
		    lRet);
	
	lRead+=lRet;
      }else{
	std::memcpy(reinterpret_cast<void *>(aBuf+lRead),
		    mTempBuf, 
		    aLength-lRead);
	//std::cout<<"rb: lRet= "<<lRet<<" lread= "<<lRead<<" residual: "<<aLength-lRead<<" lRet-(aLength-lRead)= "<<lRet-(aLength-lRead)<<std::endl;
	lFifo.Insert(&mTempBuf[(aLength-lRead)], (lRead+lRet-aLength));
	break;
      }
    }
    return aLength;
  }



  /*    
  u32 LibUsbDevice::ReadBlock(const UsbEndPoint & aEp, u8 * aBuf, 
			      u32 aLength)
  {
    
    if(!mObtained){
      HardwareAccessException lExc("Unable to read from unobtained interface.");
      RAISE(lExc);
    }
    if(!aEp.IsReadable()){
      HardwareAccessException lExc("Unable to read from end point ");
      lExc<<aEp.GetId();
      lExc<<" because it is not readable.";
      RAISE(lExc);
    }
    
    // number of bytes read back from device/ep:
    u32 lRead=0;
    u32 lPollCount=0;

    
    //std::cout<<"rb: "<<"Reading "<<aLength<<" Bytes"<<std::endl;
    if(mFifos.find(aEp.GetId())==mFifos.end()){
      //std::cout<<"rb: "<<"initialising fifo: EP "<<aEp.GetId()<<std::endl;      
      // initialise if needed:
      SoftFifo lTemp(512);
      mFifos[aEp.GetId()]=lTemp;
    }
    SoftFifo & lFifo=mFifos[aEp.GetId()];
    
    i32 lContent=0;
    if(0<(lContent=lFifo.Content())){

      i32 lXfer=(lContent>static_cast<i32>(aLength))?aLength:lContent;
      lRead=lXfer;
      //std::cout<<"rb: "<<"extracting "<<lXfer<<" from the fifo"<<std::endl;
      lFifo.Extract(aBuf, lXfer);
    }
    //std::cout<<"rb: "<<"fifo contents = "<<lContent<<std::endl;

    if(lRead==aLength){
      return lRead;
    }
    
    // more to do: calculate the bulk and residuals:
    u32 lBulk=((aLength-lRead)/512)*512;
    u32 lResidual=(aLength-lRead)%512;
    
    //std::cout<<"rb: "<<"bulk xfer = "<<lBulk<<" residuals = "<<lResidual<<std::endl;
    
    i32 lRet=0;
    u32 lBulkCount=0;
    while(lBulkCount<lBulk){
      // acquire the n*512 byte block
      //std::cout<<"rb: "<<"bulk xfer = "<<lBulk<<" Bulk count = "<<lBulkCount<<std::endl;
      lRet=usb_bulk_read(mUsbDeviceHandle,
			 aEp.GetId(),
			 reinterpret_cast<char *>(aBuf+lRead), 
			 lBulk-lBulkCount, 
			 mTimeout);
      
      if(lRet<0){
 	HardwareAccessException lExc("Bulk Read failed with error code: ");
 	lExc<<lRet;
 	lExc<<"\n error string: ";
 	lExc<<usb_strerror();
 	RAISE(lExc);
      }
      if(lRet==0){
	if(++lPollCount>1000){
	  PollLimitException lExc(lPollCount);
	  RAISE(lExc);
	}
      }
      
      lBulkCount+=lRet;
      lRead+=lRet;
    }

    lPollCount=0;

    // sort out the residuals if there are any:
    u32 lRCount=0;
    while(lRCount<lResidual){
      lRet=usb_bulk_read(mUsbDeviceHandle,
			 aEp.GetId(),
			 reinterpret_cast<char *>(mTempBuf+lRCount), 
			 512-lRCount, 
			 mTimeout);

      if(lRet<0){
 	HardwareAccessException lExc("Bulk Read failed with error code: ");
 	lExc<<lRet;
 	lExc<<"\n error string: ";
 	lExc<<usb_strerror();
 	RAISE(lExc);
      }
          //lRead+=lResidual;
      lRCount+=lRet;
    }
    
    if(lResidual){
      std::memcpy(aBuf+lRead, mTempBuf, lResidual);
      lRead+=lResidual;
      //std::cout<<lRet<<" bytes read, "<<"placing "<<(lRet-lResidual)<<" in the fifo"<<std::endl;
      lFifo.Insert(mTempBuf+lResidual, (lRet-lResidual));
    }
    
    return lRead;
  }*/
    
  /*
   *
   *
   */   
  void LibUsbDevice::WriteBlock(const UsbEndPoint & aEp, const u8 * aBuf, 
				u32 aLength)
  {
    if(!mObtained){
      HardwareAccessException lExc("Unable to write to unobtained interface.");
      RAISE(lExc);
    }

    if(!aEp.IsWritable()){
      HardwareAccessException lExc("Unable to write to end point ");
      lExc<<aEp.GetId();
      lExc<<" because it is not writable.";
      RAISE(lExc);
    }

    // lWritten sums the number of bytes written 
    // to the device during this call, which won't
    // return until the requested operation is completed.
    u32 lWritten=0;
    while(lWritten<aLength){
      i32 lRet=0;
	  
      lRet=usb_bulk_write(mUsbDeviceHandle, aEp.GetId(),
	   reinterpret_cast<char *>(const_cast<u8 *>(aBuf+lWritten)), 
			  static_cast<i32>(aLength-lWritten),
			  mTimeout);
      if(lRet<0){
	HardwareAccessException lExc("Bulk Write failed with error code: ");
	lExc<<lRet;
	lExc<<"\n error string: ";
	lExc<<usb_strerror();
	RAISE(lExc);
      }
      lWritten+=static_cast<u32>(lRet);
    }
  }


  void LibUsbDevice::BusReset(){
    i32 lret=usb_reset(mUsbDeviceHandle);
    if(lret){
      std::cerr<<__PRETTY_FUNCTION__<<" usb_reset failed."<<std::endl;
    }
  }


  //
  LibUsbDevice::DeviceParams::DeviceParams(struct usb_bus * aUsbBus,
					   struct usb_dev_handle * aUsbDeviceHandle,
					   i32 aInterfaceNumber):
    mUsbBus(aUsbBus),
    mUsbDeviceHandle(aUsbDeviceHandle),
    mInterfaceNumber(aInterfaceNumber)
    
  {
  }
    
  void LibUsbDevice::DeviceParams::Print(std::ostream & aOs)const
  {
    aOs<<mUsbBus->location<<"  "<<mUsbDeviceHandle<<"  "<<mInterfaceNumber<<"\n";
  }
}//~namespace USBDAQ
