#include "LinuxLibUsbInterface.hh"

#include "USBDAQException.hh"
#include "MemoryAllocationException.hh"
#include "HardwareAccessException.hh"

namespace USBDAQ
{

  LinuxLibUsbInterface::LinuxLibUsbInterface(const std::string & aConfig,
					     u16 aVendorId, 
					     u16 aDeviceId, 
					     u16 aBcdDeviceId):
    
    USBDAQHardwareInterface(aConfig),
    mDeviceName(""),
    mVendorId(aVendorId),
    mDeviceId(aDeviceId),
    mBcdDeviceId(aBcdDeviceId),
    mUsbDevice(0),
    mConfigIn("CmdIn"),
    mStatOut("StatOut"),
    mRawIn("RawIn"),
    mRawOut("RawOut")
  {
    try{

      mConfigIn.SetId(2);
      mConfigIn.IsWritable(true);
      mStatOut.SetId(6);
      mStatOut.IsReadable(true);

      mRawIn.SetId(4);
      mRawIn.IsWritable(true);
      mRawOut.SetId(8);
      mRawOut.IsReadable(true);

      //mUsbDevice=new LibUsbDevice(mVendorId,
      //				    mDeviceId,
      //				    mBcdDeviceId);
      
      this->Reset();
      
      //Sleep(10);
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
       
  LinuxLibUsbInterface::~LinuxLibUsbInterface()
  {
    try{
      //std::cout<<"calling bus reset..."<<std::endl;
      mUsbDevice->BusReset();
      
      //mUsbDevice->Write(mRawIn,1);
      

      delete mUsbDevice;
      mUsbDevice=0;
    }catch(USBDAQException & aExc){
      std::cerr<<aExc.What()<<"\n"<<aExc.History()<<std::endl;
    }catch(std::exception & aExc){
      std::cerr<<aExc.what()<<std::endl;
    }catch(...){
      std::cerr<<"unknown exception caught in "
	       <<__PRETTY_FUNCTION__<<std::endl;
    }
  }
  
  
  void LinuxLibUsbInterface::Read(u32 aAddr,
				  u32 & aData,
				  u32 aMask)
  {
    try {
      //u16 lCmdBuf[2]={(0x7fff&aStatusAddress),1};
      //std::cout<<"Writing buffer: "<<lCmdBuf[0]<<" "<<lCmdBuf[1]<<std::endl;
      //std::cout<<std::dec<<"reading from address : "<<aAddr<<" out of "<<mAddrMap.GetAddressSpaceSize()<<std::endl;
      aAddr=mAddrMap.GetReadAddress(aAddr);
      //std::cout<<std::dec<<"Which is : "<<aAddr<<std::endl;
      mUsbDevice->WriteBlock(mConfigIn, reinterpret_cast<u8 *>(&aAddr), 4);
      //std::cout<<"reading result 0,1"<<std::endl;
      mUsbDevice->ReadBlock(mStatOut, reinterpret_cast<u8 *>(&aData), 4);
      //std::cout<<"reading result 1"<<std::endl;
      //mUsbDevice->ReadBlock(mStatOut, reinterpret_cast<u8 *>(&aData), 1);
      //std::cout<<"done."<<std::endl;
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }
  

  
  
  
  void LinuxLibUsbInterface::Write(u32 aAddr, 
				   u32 aData, 
				   u32 aMask)
  {
    try {
      aAddr=mAddrMap.GetWriteAddress(aAddr);
      
      u32 lCmdBuf[3]={aAddr,aMask,aData};
      // for(int i=0; i<3; i++)
// 	std::cout<<std::hex<<lCmdBuf[i]<<std::endl;
      
      mUsbDevice->WriteBlock(mConfigIn, reinterpret_cast<u8 *>(lCmdBuf), 12);
      
      
//       if(EN_DEBUG_READBACK){
// 	u16 lReadBackTemp=0;
// 	this->Read(aCmdAddress, lReadBackTemp);
// 	if(aData!=lReadBackTemp){
	  
// 	  std::cerr<<"readback anomalie: register address ="
// 		   <<std::hex<<aCmdAddress
// 		   <<" data in = "<<std::hex<<aData
// 		   <<" data out = "<<lReadBackTemp<<std::endl;

// 	}
		
//       }
      

    } catch (USBDAQException & aExc) {
      RETHROW(aExc);
    }
  }
  
  
   void LinuxLibUsbInterface::SetBit(u32 aAddr, u16 aBit)
  {
    try{
      u32 lTemp=(1<<aBit);
      this->Write(aAddr, lTemp, lTemp);
    }catch(USBDAQException & aExc) {
      RETHROW(aExc);
    }
  }

  void LinuxLibUsbInterface::ClearBit(u32 aAddr, u16 aBit)
  {
    try{
      u32 lTemp=(1<<aBit);
      this->Write(aAddr, 0, lTemp);
    }catch(USBDAQException & aExc) {
      RETHROW(aExc);
    }

  }
  bool LinuxLibUsbInterface::ReadBit(u32 aAddr, u16 aBit)
  {
    u32 lTemp=0;
    try{
      this->Read(aAddr, lTemp);
    }catch(USBDAQException & aExc) {
      RETHROW(aExc);
    }
    return !((lTemp&(1<<aBit))==0);
  }

  
  void LinuxLibUsbInterface::ReadBlock(u8 * aBuf, u32 aLength)
  {
    try{
      mUsbDevice->ReadBlock(mRawOut, aBuf, aLength);     
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void LinuxLibUsbInterface::ReadBlock2(u8 * aBuf, u32 aLength)
  {
    mUsbDevice->ReadBlock2(aBuf,aLength);
  }
    
  void LinuxLibUsbInterface::WriteBlock(const u8 * aBuf, u32 aLength)
  {
    try{
      mUsbDevice->WriteBlock(mRawIn, aBuf, aLength);     
    }catch(USBDAQException & aExc){
      RETHROW(aExc);
    }
  }

  void LinuxLibUsbInterface::SetAddressSpaceSize(u32 aAddrSize)
  {
    mAddrMap.SetAddressSpaceSize(aAddrSize);
  }


  void LinuxLibUsbInterface::Reset(){
   
    
    try{
      
      // ensure there's no memory leak here
      // of course, this will fail if somebody 
      // has destructed the LibUsbDevice object and
      // not set the pointer to 0.
      delete mUsbDevice;
      mUsbDevice=0;
      
      if(mVendorId==0 && mDeviceId==0 && mBcdDeviceId==0){
	mUsbDevice=new LibUsbDevice(mDeviceName);
	if(mUsbDevice==0){
	  // this is an error:  
	  MemoryAllocationException lExc("LibUsbDevice");
	  RAISE(lExc);
	}
      }else{
	mUsbDevice=new LibUsbDevice(mVendorId,
				    mDeviceId,
				    mBcdDeviceId);
	if(mUsbDevice==0){
	  // this is an error;
	  // 
	  MemoryAllocationException lExc("LibUsbDevice");
	  RAISE(lExc);
	}
      }
      

      // this is the reset command:
      //mUsbDevice->Write(mRawIn,1);
      
      // new reset command due to new RMM EMU firmware:
      //u16 lResetVector[]={TRIG_REG, TRIG_RESET};
      //mUsbDevice->WriteBlock(mConfigIn,
      //			     reinterpret_cast<const u8 *>(lResetVector), 
      //			     4);
      
      // this device needs to be destroyed:
      delete mUsbDevice;
      mUsbDevice=0;
    }catch(MemoryAllocationException & aExc){
      RETHROW(aExc);
    }catch(USBDAQException & aExc){
      // need to work out exactly what we
      // do here.
      
    }
    
    try{
      bool lDone=false;
      u16 lTryCount=0;
      u16 lTotTries=5;
      //      u32 lDelay=200;


      do{
	// don't loop too quickly:
	//Sleep(lDelay);
	
	//std::cout<<"Trying : "<<lTryCount<<" of "<<lTotTries<<std::endl;
	try{
	  
	  if(mVendorId==0 && mDeviceId==0 && mBcdDeviceId==0){
	    mUsbDevice=new LibUsbDevice(mDeviceName);
	    
	  }else{
	    mUsbDevice=new LibUsbDevice(mVendorId,
					mDeviceId,
					mBcdDeviceId);
	  }
	  
	  lDone=true;
	}catch(USBDAQException & aExc){
	  //std::cerr<<aExc.What()<<std::endl;
	  //lDelay+=10;
	  // set the done flag so that the loop repeats:
	  //lDone=false;
	} 
	
	// let's give it a few tries
      }while(!lDone && (++lTryCount<lTotTries));

      
      
      // we could still have a problem here.
      // so raise the exception:
      if(!lDone){
	HardwareAccessException lExc("Unable to locate the device.");
	RAISE(lExc);
      }
      
      
      
    }catch(USBDAQException & aExc){
      // this is where we have a problem:
      RETHROW(aExc);
    }
  }

}//~namespace USBDAQ
