#!/bin/bash

#######################################
# Set up variables                    #
#######################################

TEX_FILE_EXT="tex"
MASTER_TEX_FILE="ProgRep.$TEX_FILE_EXT"

TEX_LIST_FILE="tex_list.txt"
TEX_LIST_ENTRY=""

FIG_LIST_FILE="fig_list.txt"
FIG_LIST_ENTRY=""
FIG_LIST_DELIMITER=":"

FIG_PATH_FULL=""
FIG_PATH_NAME=""

FIG_NAME_CORE=""
FIG_NAME_OLD=""
FIG_NAME_NEW=""

# Legacy fig name = Figure-mmm-<core file name>
# New fig name = Figure-mmm-nn-<core file name>
FIG_PREFIX_PATTERN_LEGACY='Figure-[0-9][0-9][0-9]-*'
FIG_PREFIX_LENGTH_LEGACY=11

FIG_PREFIX_PATTERN='Figure-[0-9][0-9][0-9]-[0-9][0-9]-*'
FIG_PREFIX_LENGTH=14

FIG_INDEX_LENGTH=3
FIG_SUB_INDEX_LENGTH=2

FIG_INDEX=1
FIG_INDEX_STR=""

FIG_SUB_INDEX=1
FIG_SUB_INDEX_STR=""

#######################################
# Functions                           #
#######################################

function ParseTEXFileNames()
{
	# Usage: ParseTEXFileNames <input TEX file>
	
	# Return values:
	# 0: OK
	# 1: Missing arguments
	
	# Check for existence of argument
	if test -z "$1"
	then
		return 1
	fi
	
	# Set up 'variables'
	local INPUT_TEX_FILE=$1
	local LINE=""
	local PARSED_TEX_FILE=""
	local PARSED_TEX_FILE_EXT=""
	
	# Ensure that TEX file is correctly formatted...
	dos2unix -q "$INPUT_TEX_FILE"
	
	# Read TEX file
	cat "$INPUT_TEX_FILE" | while read -r LINE; do
		
		# Check whether we have an input line
		if [[ ${LINE} == \\input* ]]
		then
			
			PARSED_TEX_FILE="$(echo "$LINE" | cut -d " " -f 2)"
			PARSED_TEX_FILE_EXT="$(echo "$PARSED_TEX_FILE" | awk -F . '{if (NF > 1) {print $NF}}')"
			
			# Add extension, if required
			if [[ "$PARSED_TEX_FILE_EXT" == "" ]]
			then
				
				PARSED_TEX_FILE="${PARSED_TEX_FILE}.$TEX_FILE_EXT"
				
			fi
			
			# Add file to list
			echo "$PARSED_TEX_FILE" >> "$TEX_LIST_FILE"
			
			# Parse the file that was identified
			ParseTEXFileNames "$PARSED_TEX_FILE"
			
		fi
		
	done
	
	return 0
}

#######################################

function ParseImageFileNames()
{
	# Usage: ParseTEXFileNames <input TEX file>
	
	# Return values:
	# 0: OK
	# 1: Missing arguments
	
	# Check for existence of argument
	if test -z "$1"
	then
		return 1
	fi
	
	# Set up 'variables'
	local INPUT_TEX_FILE=$1
	local LINE=""
	local NEXT_LINE_OVERRIDE=0
	local PARSE_LINE=0
	local PARSED_IMAGE_FILE=""
	local PARSED_IMAGE_FILE_NEW=""
	local PARSED_IMAGE_FILE_EXT=""
	local LAST_FIG_LIST_ENTRY=""
	
	# Ensure that TEX file is correctly formatted...
	dos2unix -q "$INPUT_TEX_FILE"
	
	# Read TEX file
	cat "$INPUT_TEX_FILE" | while read -r LINE; do
		
		# Check whether the current line is a continuation of an image line...
		if [[ $NEXT_LINE_OVERRIDE -eq 1 ]]
		then
			
			PARSE_LINE=1
			NEXT_LINE_OVERRIDE=0
			
		else
			
			# Check whether we have an image line
			if [[ ${LINE} == \\includegraphics* ]]
			then
				
				if [[ ${LINE} == *% ]]
				then
					
					NEXT_LINE_OVERRIDE=1
					
				else
					
					PARSE_LINE=1
					
				fi
				
			# Check whether we need to update the indicies
			elif [[ ${LINE} == \\end\{figure\}* ]]
			then
				
				((FIG_INDEX++))
				FIG_SUB_INDEX=1
				
			fi
			
		fi
		
		# Check whether we need to parse the current line
		if [[ $PARSE_LINE -eq 1 ]]
		then
			
			PARSED_IMAGE_FILE=$(echo $LINE | awk -F "{" '{print $2}' | awk -F "}" '{print $1}')
			PARSED_IMAGE_FILE_EXT="$(echo "$PARSED_IMAGE_FILE" | awk -F . '{if (NF > 1) {print $NF}}')"
			
			# Get full file name, if extension is missing
			if [[ "$PARSED_IMAGE_FILE_EXT" == "" ]]
			then
				
				PARSED_IMAGE_FILE_NEW="$(ls -1t "${PARSED_IMAGE_FILE}."* | head -n 1)"
				
				# If the extension is missing we need to add it to the TEX file,
				# otherwise updating the file names with a prefix will fail
				# later on...
				FindAndReplace "." "$TEX_FILE_EXT" "$PARSED_IMAGE_FILE" "$PARSED_IMAGE_FILE_NEW"
				
				PARSED_IMAGE_FILE="$PARSED_IMAGE_FILE_NEW"
				
			fi
			
			# Generate figure list entry
			FIG_INDEX_STR=$(printf "%0${FIG_INDEX_LENGTH}d" $FIG_INDEX)
			FIG_SUB_INDEX_STR=$(printf "%0${FIG_SUB_INDEX_LENGTH}d" $FIG_SUB_INDEX)
			FIG_LIST_ENTRY="${FIG_INDEX_STR}${FIG_LIST_DELIMITER}${FIG_SUB_INDEX_STR}${FIG_LIST_DELIMITER}${PARSED_IMAGE_FILE}"
			
			# Add entry to list
			echo "$FIG_LIST_ENTRY" >> "$FIG_LIST_FILE"
			
			((FIG_SUB_INDEX++))
			PARSE_LINE=0
			
		fi
		
	done
	
	# Sigh... due to the fact that the above ((FIG_INDEX++)) operation
	# happens in a new shell, the current value is lost once we reach this
	# point... so we have to read it back in from the file and increment
	# again... nasty hack...
	
	if [[ -e "$FIG_LIST_FILE" ]]
	then
		
		LAST_FIG_LIST_ENTRY=$(tail -q -n 1 "$FIG_LIST_FILE")
		FIG_INDEX=$(echo "$LAST_FIG_LIST_ENTRY" | cut -d "$FIG_LIST_DELIMITER" -f 1 | sed 's/0*//')
		((FIG_INDEX++))
		
	fi
	
	return 0
}

#######################################

function FindAndReplace()
{
	# Usage: FindAndReplace <top level directory> <file extension> <string to find> <string to replace>
	
	# Return values:
	# 0: OK
	# 1: Missing arguments
	
	# Check for existence of arguments
	if test -z "$1" || test -z "$2" || test -z "$3" || test -z "$4"
	then
		return 1
	fi
	
	# Set up 'variables'
	local DIRECTORY=$1
	local FILE_EXT=$2
	local FIND_STRING=$3
	local REPLACE_STRING=$4
	
	# Loop over all files of type $FILE_EXT
	for FILENAME in $(find $DIRECTORY -name "*.$FILE_EXT")
	do
		
		echo "sed -i \"s#\<$FIND_STRING\>#$REPLACE_STRING#g\" $FILENAME"
		sed -i "s#\<$FIND_STRING\>#$REPLACE_STRING#g" $FILENAME
		
	done
	
	return 0
}

#######################################
# Compile list of TEX files           #
#######################################

rm -f "$TEX_LIST_FILE"
ParseTEXFileNames "$MASTER_TEX_FILE"

#######################################
# Compile list of image files         #
#######################################

rm -f "$FIG_LIST_FILE"
cat "$TEX_LIST_FILE" | while read -r TEX_LIST_ENTRY; do
	ParseImageFileNames "$TEX_LIST_ENTRY"
done

#######################################
# Loop through all image files        #
#######################################

cat "$FIG_LIST_FILE" | while read -r FIG_LIST_ENTRY; do
	
	# Extract entry segments
	FIG_INDEX_STR="$(echo "$FIG_LIST_ENTRY" | cut -d "$FIG_LIST_DELIMITER" -f 1)"
	FIG_SUB_INDEX_STR="$(echo "$FIG_LIST_ENTRY" | cut -d "$FIG_LIST_DELIMITER" -f 2)"
	FIG_PATH_FULL="$(echo "$FIG_LIST_ENTRY" | cut -d "$FIG_LIST_DELIMITER" -f 3)"
	
	FIG_PATH_NAME=$(dirname "$FIG_PATH_FULL")
	FIG_NAME_OLD=$(basename "$FIG_PATH_FULL")
	
	# Determine whether we need to strip off an existing prefix
	if [[ "$FIG_NAME_OLD" =~ $FIG_PREFIX_PATTERN ]]
	then
		
		FIG_NAME_CORE="${FIG_NAME_OLD:FIG_PREFIX_LENGTH}"
		
	elif [[ "$FIG_NAME_OLD" =~ $FIG_PREFIX_PATTERN_LEGACY ]]
	then
		
		FIG_NAME_CORE="${FIG_NAME_OLD:FIG_PREFIX_LENGTH_LEGACY}"
		
	else
		
		FIG_NAME_CORE="$FIG_NAME_OLD"
		
	fi
	
	# Generate new name
	FIG_NAME_NEW="Figure-${FIG_INDEX_STR}-${FIG_SUB_INDEX_STR}-${FIG_NAME_CORE}"
	
	if [[ "$FIG_NAME_NEW" != "$FIG_NAME_OLD" ]]
	then
		
		# 'Rename' image file
		echo "mv \"$FIG_PATH_NAME/$FIG_NAME_OLD\" \"$FIG_PATH_NAME/$FIG_NAME_NEW\""
 		mv "$FIG_PATH_NAME/$FIG_NAME_OLD" "$FIG_PATH_NAME/$FIG_NAME_NEW"
		
		# Update references in LATEX files
		FindAndReplace "." "$TEX_FILE_EXT" "$FIG_NAME_OLD" "$FIG_NAME_NEW"
		
	fi
	
done
